/**************************************************************************//**
 * @file
 * @brief ADC Examples for EFM32 Gecko Series 0
 * @version 1.1.1
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "em_adc.h"
#include "em_chip.h"
#include "em_cmu.h"
#include "em_device.h"
#include "em_dma.h"
#include "em_emu.h"
#include "em_core.h"
#include "em_lcd.h"
#include "em_prs.h"
#include "em_rtc.h"
#include "em_timer.h"
#include "dmactrl.h"
#include "segmentlcd.h"

#define ADC_CLOCK               11000000        /* ADC conversion clock */
#define ILLEGAL_VALUE           0xffff          /* Invalid 12 bit ADC value */
#define ADC_GAIN_CAL_VALUE      0xffd0          /* ADC gain calibration value */
#define ADC_CAL_INPUT           adcSingleInputCh4

/* Constants for voltage calculation */
#define ADC_SE_VFS_X100         330             /* Reference VDD x 100 */
#define ADC_SE_VFS_X1000        3300            /* Reference VDD x 1000 */
#define ADC_DIFF_VFS_X100       660             /* Reference 2xVDD x 100 */
#define ADC_12BIT_MAX           4096            /* 2^12 */
#define ADC_16BIT_MAX           65536           /* 2^16 */

/* Push buttons used on different EFM32 STKs */
#if defined(_EFM32_GIANT_FAMILY) || defined(_EFM32_WONDER_FAMILY) || defined(_EFM32_LEOPARD_FAMILY) || defined(_EFM32_GECKO_FAMILY)
#define GPIO_KEY0_PORT          gpioPortB
#define GPIO_KEY0_PIN           9
#define GPIO_KEY1_PORT          gpioPortB
#define GPIO_KEY1_PIN           10
#elif defined(_EFM32_TINY_FAMILY)
#define GPIO_KEY0_PORT          gpioPortD
#define GPIO_KEY0_PIN           8
#define GPIO_KEY1_PORT          gpioPortB
#define GPIO_KEY1_PIN           11
#else
#error "STK does not support segment LCD"
#endif

/* DMA channel used for scan sequence sampling adc channels. */
#define ADC_DMA_CHANNEL         0
#define ADC_SCAN_CH_NUMBER      3

/* Change this to increase or decrease sampling frequency. */
#define RTC_WAKEUP_MS           10
#define RTC_WAKEUP_COUNT        (((32768 * RTC_WAKEUP_MS) / 1000) - 1)

/* Change this to increase or decrease number of samples. */
#define N_SAMPLES               512

/* Menu items */
#define SING                    0
#define SCAN                    1
#define DIFF                    2
#define OVS                     3
#define TPRS                    4
#define INT                     5
#define WFE                     6
#define CAB                     7
#define MENU_MAX                8

volatile bool menuKey;
volatile bool runKey;
volatile bool adcFinished = true;
volatile uint8_t menuLevel;
volatile uint16_t sampleCount;
volatile uint16_t sampleValue;
int16_t sampleBuffer[N_SAMPLES];

/**********************************************************
 * @brief Interrupt handler for push button PB0. 
 **********************************************************/
#if defined(_EFM32_TINY_FAMILY)
void GPIO_EVEN_IRQHandler(void)
#else
void GPIO_ODD_IRQHandler(void)
#endif
{
  GPIO->IFC = GPIO->IF;

  /* Next menu item if adc is not running */
  if (adcFinished)
  {
    menuKey = true;
    if (++menuLevel == MENU_MAX)
    {
      menuLevel = 0;
    }
  }
  else
  {
    adcFinished = true;
  }    
}

/**********************************************************
 * @brief Interrupt handler for push button PB1. 
 **********************************************************/
#if defined(_EFM32_TINY_FAMILY)
void GPIO_ODD_IRQHandler(void)
#else
void GPIO_EVEN_IRQHandler(void)
#endif
{
  GPIO->IFC = GPIO->IF;

  /* No action if ADC is running */
  if (adcFinished)
  {
    runKey = true;
  }
}

/**************************************************************************//**
 * @brief ADC0_IRQHandler
 *****************************************************************************/
void ADC0_IRQHandler(void)
{
  /* Clear ADC0 interrupt flag */
  ADC_IntClear(ADC0, ADC_IFC_SINGLE);

  /* Read conversion result to clear Single Data Valid flag */
  sampleValue = ADC_DataSingleGet(ADC0);
}

/***************************************************************************//**
 * @brief RTC Interrupt Handler.
 ******************************************************************************/
void RTC_IRQHandler(void)
{
  /* Start ADC conversion as soon as RTC wake up. */
  ADC_Start(ADC0, adcStartSingle);
  
  /* Clear the interrupt flag */
  RTC_IntClear(RTC_IFC_COMP0);
  
  /* Wait while conversion is active */
  while (ADC0->STATUS & ADC_STATUS_SINGLEACT)
    ;
  
  /* Get ADC result */
  sampleBuffer[sampleCount++] = ADC_DataSingleGet(ADC0);
  
  if(sampleCount == N_SAMPLES)
  {
    adcFinished = true;
  }
}

/**************************************************************************//**
 * @brief Setup GPIO interrupt for push buttons
 *****************************************************************************/
void gpioSetup(void)
{
  /* Enable GPIO clock in CMU */
  CMU_ClockEnable(cmuClock_GPIO, true);

  /* Configure push buttons PB0 and PB1 */
  GPIO_PinModeSet(GPIO_KEY0_PORT, GPIO_KEY0_PIN, gpioModeInputPullFilter, 1);
  GPIO_IntConfig(GPIO_KEY0_PORT, GPIO_KEY0_PIN, false, true, true);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);

  GPIO_PinModeSet(GPIO_KEY1_PORT, GPIO_KEY1_PIN, gpioModeInputPullFilter, 1);
  GPIO_IntConfig(GPIO_KEY1_PORT, GPIO_KEY1_PIN, false, true, true);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);
  
  /* Enable interrupt in core for even and odd GPIO interrupts */
  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
}

/***************************************************************************//**
 * @brief Setup RTC for ADC trigger
 ******************************************************************************/
void rtcSetup(void)
{
  RTC_Init_TypeDef init = RTC_INIT_DEFAULT;

  /* LE and LFRCO clocks are enabled at LCD initialization */
  /* Enable clock to RTC module */
  CMU_ClockEnable(cmuClock_RTC, true);

  init.enable   = false;
  RTC_Init(&init);
  
  RTC_CompareSet(0, RTC_WAKEUP_COUNT);
}

/***************************************************************************//**
* @brief Configure Timer for ADC PRS trigger.
*******************************************************************************/
void adcTimerPrsSetup(void)
{
  /* Use default timer settings */
  TIMER_Init_TypeDef timerInit = TIMER_INIT_DEFAULT;

  CMU_ClockEnable(cmuClock_TIMER0, true);

  /* Change prescaler to 64, gives roughly 3 overflows per
   * second at 11MHz with 0xffff as top value */
  timerInit.enable = false;
  timerInit.prescale = timerPrescale64;
  TIMER_Init(TIMER0, &timerInit);
}

/***************************************************************************//**
* @brief Configure DMA for ADC scan mode.
*******************************************************************************/
void adcDmaSetup(void)
{
  DMA_Init_TypeDef       dmaInit;
  DMA_CfgDescr_TypeDef   descrCfg;
  DMA_CfgChannel_TypeDef chnlCfg;

  CMU_ClockEnable(cmuClock_DMA, true);
  
  /* Configure general DMA issues */
  dmaInit.hprot        = 0;
  dmaInit.controlBlock = dmaControlBlock;
  DMA_Init(&dmaInit);

  /* Configure DMA channel used */
  chnlCfg.highPri   = false;
  chnlCfg.enableInt = false;
  chnlCfg.select    = DMAREQ_ADC0_SCAN;
  chnlCfg.cb        = NULL;
  DMA_CfgChannel(ADC_DMA_CHANNEL, &chnlCfg);

  descrCfg.dstInc  = dmaDataInc2;
  descrCfg.srcInc  = dmaDataIncNone;
  descrCfg.size    = dmaDataSize2;
  descrCfg.arbRate = dmaArbitrate1;
  descrCfg.hprot   = 0;
  DMA_CfgDescr(ADC_DMA_CHANNEL, true, &descrCfg);
}

/***************************************************************************//**
 * @brief
 *   Calibrate offset and gain for the specified reference.
 *   Supports currently only single ended gain calibration.
 *   Could easily be expanded to support differential gain calibration.
 *
 * @details
 *   The offset calibration routine measures 0 V with the ADC, and adjust
 *   the calibration register until the converted value equals 0.
 *   The gain calibration routine needs an external reference voltage equal
 *   to the top value for the selected reference. For example if the 2.5 V
 *   reference is to be calibrated, the external supply must also equal 2.5V.
 *
 * @param[in] adc
 *   Pointer to ADC peripheral register block.
 *
 * @param[in] ref
 *   Reference used during calibration. Can be both external and internal
 *   references.
 *   The SCANGAIN and SINGLEGAIN calibration fields are not used when the 
 *   unbuffered differential 2xVDD reference is selected.
 * 
 * @return
 *   The final value of the calibration register, note that the calibration
 *   register gets updated with this value during the calibration.
 *   No need to load the calibration values after the function returns.
 ******************************************************************************/
uint32_t ADC_Calibration(ADC_TypeDef *adc, ADC_Ref_TypeDef ref)
{
  int32_t  sample;
  uint32_t cal;

  /* Binary search variables */
  uint8_t high;
  uint8_t mid;
  uint8_t low;

  /* Reset ADC to be sure we have default settings and wait for ongoing */
  /* conversions to be complete. */
  ADC_Reset(adc);

  ADC_Init_TypeDef       init       = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  /* Init common settings for both single conversion and scan mode */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(ADC_CLOCK, 0);

  /* Set an oversampling rate for more accuracy */
  init.ovsRateSel = adcOvsRateSel4096;
  ADC_Init(adc, &init);

  /* Init for single conversion use, measure DIFF0 with selected reference. */
  singleInit.reference = ref;
  singleInit.input     = adcSingleInputDiff0;
  singleInit.acqTime   = adcAcqTime16;
  singleInit.diff      = true;
  /* Enable oversampling rate */
  singleInit.resolution = adcResOVS;

  ADC_InitSingle(adc, &singleInit);

  /* ADC is now set up for offset calibration */
  /* Offset calibration register is a 7 bit signed 2's complement value. */
  /* Use unsigned indexes for binary search, and convert when calibration */
  /* register is written to. */
  high = 128;
  low  = 0;

  /* Do binary search for offset calibration*/
  while (low < high)
  {
    /* Calculate midpoint */
    mid = low + (high - low) / 2;

    /* Midpoint is converted to 2's complement and written to both scan and */
    /* single calibration registers */
    cal      = adc->CAL & ~(_ADC_CAL_SINGLEOFFSET_MASK | _ADC_CAL_SCANOFFSET_MASK);
    cal     |= (uint8_t)(mid - 63) << _ADC_CAL_SINGLEOFFSET_SHIFT;
    cal     |= (uint8_t)(mid - 63) << _ADC_CAL_SCANOFFSET_SHIFT;
    adc->CAL = cal;
    
    /* Do a conversion */
    ADC_Start(adc, adcStartSingle);
    while (adc->STATUS & ADC_STATUS_SINGLEACT)
      ;

    /* Get ADC result */
    sample = ADC_DataSingleGet(adc);

    /* Check result and decide in which part of to repeat search */
    /* Calibration register has negative effect on result */
    if (sample < 0)
    {
      /* Repeat search in bottom half. */
      high = mid;
    }
    else if (sample > 0)
    {
      /* Repeat search in top half. */
      low = mid + 1;
    }
    else
    {
      /* Found it, exit while loop */
      break;
    }
  }

  /* Now do gain calibration, only INPUT and DIFF settings needs to be changed */
  adc->SINGLECTRL &= ~(_ADC_SINGLECTRL_INPUTSEL_MASK | _ADC_SINGLECTRL_DIFF_MASK);
  adc->SINGLECTRL |= (ADC_CAL_INPUT << _ADC_SINGLECTRL_INPUTSEL_SHIFT);
  adc->SINGLECTRL |= (false << _ADC_SINGLECTRL_DIFF_SHIFT);

  /* Gain calibration register is a 7 bit unsigned value. */
  high = 128;
  low  = 0;

  /* Do binary search for gain calibration */
  while (low < high)
  {
    /* Calculate midpoint and write to calibration register */
    mid = low + (high - low) / 2;
    cal      = adc->CAL & ~(_ADC_CAL_SINGLEGAIN_MASK | _ADC_CAL_SCANGAIN_MASK);
    cal     |= mid << _ADC_CAL_SINGLEGAIN_SHIFT;
    cal     |= mid << _ADC_CAL_SCANGAIN_SHIFT;
    adc->CAL = cal;

    /* Do a conversion */
    ADC_Start(adc, adcStartSingle);
    while (adc->STATUS & ADC_STATUS_SINGLEACT)
      ;

    /* Get ADC result */
    sample = ADC_DataSingleGet(adc);

    /* Check result and decide in which part to repeat search */
    /* Compare with a value atleast one LSB's less than top to avoid overshooting */
    /* Since oversampling is used, the result is 16 bits, but a couple of lsb's */
    /* applies to the 12 bit result value, if 0xffd is the top value in 12 bit, this */
    /* is in turn 0xffd0 in the 16 bit result. */
    /* Calibration register has positive effect on result */
    if (sample > ADC_GAIN_CAL_VALUE)
    {
      /* Repeat search in bottom half. */
      high = mid;
    }
    else if (sample < ADC_GAIN_CAL_VALUE)
    {
      /* Repeat search in top half. */
      low = mid + 1;
    }
    else
    {
      /* Found it, exit while loop */
      break;
    }
  }
  return adc->CAL;
}

/**************************************************************************//**
 * @brief Reset ADC registers and disable clocks.
 *****************************************************************************/
void adcReset(void)
{
  /* Rest ADC registers */
  ADC_Reset(ADC0);
  NVIC_DisableIRQ(ADC0_IRQn);

  /* Disable clocks */
  CMU_ClockEnable(cmuClock_DMA, false);
  CMU_ClockEnable(cmuClock_PRS, false);
  CMU_ClockEnable(cmuClock_TIMER0, false);

  /* Enable key interrupt, for INT and WFE example */
  runKey = false;
  menuKey = false;
  CMU_ClockEnable(cmuClock_GPIO, true);
  GPIO->IFC = _GPIO_IFC_MASK;
  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);
  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
}

/***************************************************************************//**
* @brief Configure ADC for single channel.
*******************************************************************************/
void adcSingle(void)
{
  ADC_Init_TypeDef       init       = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  /* Init common settings for both single conversion and scan mode */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(ADC_CLOCK, 0);
  ADC_Init(ADC0, &init);

  /* Init for single conversion, measure VDD/3 with 1.25V reference. */
  singleInit.input      = adcSingleInpVDDDiv3;
  /* The datasheet specifies a minimum aquisition time when sampling VDD/3 */
  /* 32 cycles should be safe for all ADC clock frequencies */
  singleInit.acqTime = adcAcqTime32;
  ADC_InitSingle(ADC0, &singleInit);

  ADC_Start(ADC0, adcStartSingle);
  
  /* Wait while conversion is active */
  while (ADC0->STATUS & ADC_STATUS_SINGLEACT)
    ;
  
  /* Get ADC result */
  sampleValue = ADC_DataSingleGet(ADC0);
  
  /* Calculate supply voltage relative to 1.25V reference */
  sampleValue = (sampleValue * 1250 * 3) / ADC_12BIT_MAX;
  
  /* Write to LCD */
  SegmentLCD_Number(sampleValue);
  adcReset();
}

/***************************************************************************//**
* @brief Configure ADC for scan mode, using DMA to fetch ADC data.
*******************************************************************************/
void adcScanDma(void)
{
  ADC_Init_TypeDef     init     = ADC_INIT_DEFAULT;
  ADC_InitScan_TypeDef scanInit = ADC_INITSCAN_DEFAULT;

  adcDmaSetup();

  /* Init common issues for both single conversion and scan mode */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(ADC_CLOCK/2, 0);
  ADC_Init(ADC0, &init);

  /* Init for scan mode. */
  scanInit.reference = adcRefVDD;
  scanInit.input     = ADC_SCANCTRL_INPUTMASK_CH3 |
                       ADC_SCANCTRL_INPUTMASK_CH4 |
                       ADC_SCANCTRL_INPUTMASK_CH5;
  ADC_InitScan(ADC0, &scanInit);

  /* Start DMA */
  DMA_ActivateBasic(ADC_DMA_CHANNEL,
                    true,
                    false,
                    sampleBuffer,
                    (void *)((uint32_t)&(ADC0->SCANDATA)),
                    (ADC_SCAN_CH_NUMBER - 1));
  
  /* Start Scan */
  ADC_Start(ADC0, adcStartScan);
  
  /* Poll for scan comversion complete */
  while (ADC0->STATUS & ADC_STATUS_SCANACT)
    ;
  
  /* Write to LCD */
  sampleBuffer[0] = (sampleBuffer[0] * ADC_SE_VFS_X100) / ADC_12BIT_MAX;
  sampleBuffer[1] = (sampleBuffer[1] * ADC_SE_VFS_X1000) / ADC_12BIT_MAX;
  sampleBuffer[2] = (sampleBuffer[2] * ADC_SE_VFS_X1000) / ADC_12BIT_MAX;
  SegmentLCD_Number(sampleBuffer[2]);
  SegmentLCD_LowerNumber((sampleBuffer[0] * 10000) + sampleBuffer[1]);
  adcReset();
}

/***************************************************************************//**
* @brief Configure ADC for differential single conversion.
*******************************************************************************/
void adcSingleDiff(void)
{
  ADC_Init_TypeDef       init       = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  /* Init common settings for both single conversion and scan mode */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(ADC_CLOCK, 0);
  ADC_Init(ADC0, &init);

  /* Init for single conversion, differential channels on CH4 & CH5. */
  singleInit.diff = true;
  singleInit.input = adcSingleInputCh4Ch5;
  singleInit.reference = adcRef2xVDD;
  ADC_InitSingle(ADC0, &singleInit);

  ADC_Start(ADC0, adcStartSingle);
  
  /* Wait while conversion is active */
  while (ADC0->STATUS & ADC_STATUS_SINGLEACT)
    ;

  /* Get ADC result */
  sampleBuffer[0] = ADC_DataSingleGet(ADC0);

  /* Write to LCD, postive or negaive value */
  sampleBuffer[0] = (sampleBuffer[0] * ADC_DIFF_VFS_X100) / ADC_12BIT_MAX;
  SegmentLCD_Number(sampleBuffer[0]);
  adcReset();
}

/***************************************************************************//**
* @brief Configure ADC for measuring VDD/3 with oversampling.
*******************************************************************************/
void adcSingleOvs(void)
{
  ADC_Init_TypeDef       init       = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  /* Init common settings for both single conversion and scan mode */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(ADC_CLOCK, 0);
  /* Set oversampling rate */
  init.ovsRateSel = adcOvsRateSel256;
  ADC_Init(ADC0, &init);

  /* Init for single conversion, measure VDD/3 with 1.25V reference. */
  singleInit.input = adcSingleInpVDDDiv3;
  /* The datasheet specifies a minimum aquisition time when sampling VDD/3 */
  /* 32 cycles should be safe for all ADC clock frequencies */
  singleInit.acqTime = adcAcqTime32;
  /* Enable oversampling rate */
  singleInit.resolution = adcResOVS;
  ADC_InitSingle(ADC0, &singleInit);

  ADC_Start(ADC0, adcStartSingle);
  
  /* Wait while conversion is active */
  while (ADC0->STATUS & ADC_STATUS_SINGLEACT)
    ;
  
  /* Get ADC result */
  sampleValue = ADC_DataSingleGet(ADC0);
  
  /* Calculate supply voltage relative to 1.25V reference */
  sampleValue = (sampleValue * 1250 * 3) / ADC_16BIT_MAX;
  
  /* Write to LCD */
  SegmentLCD_Number(sampleValue);
  adcReset();
}

/***************************************************************************//**
* @brief Configure ADC for TIMER PRS trigger.
*******************************************************************************/
void adcTImerPrs(void)
{
  ADC_Init_TypeDef       init       = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  adcTimerPrsSetup();  
  
  /* Init common settings for both single conversion and scan mode */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(ADC_CLOCK, 0);
  ADC_Init(ADC0, &init);

  /* Init for single conversion. */
  singleInit.input = adcSingleInputCh3;
  singleInit.reference  = adcRefVDD;
  /* Enable PRS for ADC */
  singleInit.prsEnable  = true; 
  ADC_InitSingle(ADC0, &singleInit);

  /* Select TIMER0 as source and TIMER0OF (Timer0 overflow) as signal (rising edge) */
  CMU_ClockEnable(cmuClock_PRS, true);
  PRS_SourceSignalSet(0, PRS_CH_CTRL_SOURCESEL_TIMER0, PRS_CH_CTRL_SIGSEL_TIMER0OF, prsEdgePos);
  
  /* Enable ADC Interrupt when Single Conversion Complete */
  ADC_IntEnable(ADC0, ADC_IEN_SINGLE);

  /* Enable ADC interrupt vector in NVIC*/
  NVIC_ClearPendingIRQ(ADC0_IRQn);
  NVIC_EnableIRQ(ADC0_IRQn);

  adcFinished = false;
  sampleValue = ILLEGAL_VALUE;

  /* Start TIMER0 */
  SegmentLCD_Write("PRS RUN");
  TIMER_Enable(TIMER0, true);
  
  /* Exit if PB0 is pressed */
  while (!adcFinished)
  {
    /* Enter EM1 and wait for timer triggered ADC conversion */
    EMU_EnterEM1();
    if (sampleValue != ILLEGAL_VALUE)
    {
      /* Write result to LCD */
      sampleValue = (sampleValue * ADC_SE_VFS_X1000) / ADC_12BIT_MAX;
      SegmentLCD_Number(sampleValue);
      sampleValue = ILLEGAL_VALUE;
    }
  }

  /* Stop TIMER0 */
  TIMER_Enable(TIMER0, false);
  LCD_NUMBER_OFF();
  SegmentLCD_Write("PRS");
  adcReset();
}

/***************************************************************************//**
* @brief Configure ADC for single channel interrupt.
*******************************************************************************/
void adcSingleInt(void)
{
  ADC_Init_TypeDef       init       = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  /* Init common settings for both single conversion and scan mode */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(ADC_CLOCK, 0);
  ADC_Init(ADC0, &init);

  /* Init for single conversion. */
  singleInit.input = adcSingleInputCh3;
  singleInit.reference  = adcRefVDD;
  ADC_InitSingle(ADC0, &singleInit);

  /* Enable interrupt on COMP0 */
  RTC_IntEnable(RTC_IF_COMP0);
  NVIC_ClearPendingIRQ(RTC_IRQn);
  NVIC_EnableIRQ(RTC_IRQn);  

  /* Turn off LCD and GPIO to reduce current consumption, start RTC */
  LCD_Enable(false);
  LCD_SyncBusyDelay(LCD_SYNCBUSY_CTRL);
  CMU_ClockEnable(cmuClock_LCD, false);

  CMU_ClockEnable(cmuClock_GPIO, false);
  NVIC_DisableIRQ(GPIO_EVEN_IRQn);
  NVIC_DisableIRQ(GPIO_ODD_IRQn);
  
  sampleCount = 0;
  adcFinished = false;
  RTC_Enable(true);

  CORE_DECLARE_IRQ_STATE;
  CORE_ENTER_ATOMIC();
  /* Exit if end of sampling */
  while(!adcFinished)
  {
   EMU_EnterEM2(false); 
   CORE_EXIT_ATOMIC();
   CORE_ENTER_ATOMIC();
  }
  CORE_EXIT_ATOMIC();    
  
  /* Stop RTC and disable interrupt on COMP0 */
  RTC_Enable(false);
  RTC_IntDisable(_RTC_IF_MASK);
  NVIC_DisableIRQ(RTC_IRQn);  

  /* Enable LCD */
  CMU_ClockEnable(cmuClock_LCD, true);
  LCD_Enable(true);
  LCD_SyncBusyDelay(LCD_SYNCBUSY_CTRL);
  SegmentLCD_Write("INT END");
  adcReset();
}

/***************************************************************************//**
* @brief Configure ADC for single channel WFE.
*******************************************************************************/
void adcSingleWfe(void)
{
  ADC_Init_TypeDef       init       = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  /* Init common settings for both single conversion and scan mode */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(ADC_CLOCK, 0);
  ADC_Init(ADC0, &init);

  /* Init for single conversion. */
  singleInit.input = adcSingleInputCh3;
  singleInit.reference  = adcRefVDD;
  ADC_InitSingle(ADC0, &singleInit);

  /* Enable interrupt generation from RTC, needed for WFE (wait for event). */
  /* Notice that enabling the interrupt in the NVIC is not needed. */
  RTC_IntEnable(RTC_IF_COMP0);
  NVIC_ClearPendingIRQ(RTC_IRQn);

  /* Enable ADC Interrupt when Single Conversion Complete. */
  /* This is necessary for WFE (wait for event) to work. */
  /* Notice that enabling the interrupt in the NVIC is not needed. */  
  ADC_IntEnable(ADC0, ADC_IEN_SINGLE);
  NVIC_ClearPendingIRQ(ADC0_IRQn);

  /* Turn off LCD and GPIO to reduce current consumption, start RTC */
  LCD_Enable(false);
  LCD_SyncBusyDelay(LCD_SYNCBUSY_CTRL);
  CMU_ClockEnable(cmuClock_LCD, false);

  CMU_ClockEnable(cmuClock_GPIO, false);
  NVIC_DisableIRQ(GPIO_EVEN_IRQn);
  NVIC_DisableIRQ(GPIO_ODD_IRQn);

  sampleCount = N_SAMPLES;
  RTC_Enable(true);

  /* Exit if end of sampling */
  while(sampleCount > 0)
  {    
    /* Enable deep sleep to enter EM2 between samples. */
    SCB->SCR = 0x14;
    
    /* Go to sleep while waiting for RTC event (set by RTC_IRQ pending bit) */
    /* Since IRQ is not enabled in the NVIC, no ISR will be entered */
    __WFE();
    
    /* Start ADC conversion as soon as we wake up. */
    ADC_Start(ADC0, adcStartSingle);
    
    /* Clear the interrupt flag */
    RTC_IntClear(RTC_IF_COMP0);
    
    /* Clear pending RTC IRQ */
    NVIC_ClearPendingIRQ(RTC_IRQn);

    /* Wait while conversion is active in EM1, should be almost finished since it */
    /* takes 13 cycles + warmup (1us), and it was started a while ago. */
    /* Disable deep sleep so we wait in EM1 for conversion to finish. */
    SCB->SCR = 0x10; 
    __WFE();
    
    /* Clear the interrupt flag */
    ADC_IntClear(ADC0, ADC_IF_SINGLE);
    
    /* Clear pending IRQ */
    NVIC_ClearPendingIRQ(ADC0_IRQn);
    
    /* Get ADC result */
    sampleBuffer[sampleCount--] = ADC_DataSingleGet(ADC0);
  }
  
  /* Stop RTC and disable interrupt on COMP0 */
  RTC_Enable(false);
  RTC_IntDisable(_RTC_IF_MASK);

  /* Enable LCD */
  CMU_ClockEnable(cmuClock_LCD, true);
  LCD_Enable(true);
  LCD_SyncBusyDelay(LCD_SYNCBUSY_CTRL);
  SegmentLCD_Write("WFE END");
  adcReset();
}

/***************************************************************************//**
* @brief Calibrate ADC offset and gain for the specified reference.
*******************************************************************************/
void adcCalibration(void)
{
  /* Get existing offset and gain for 1.25V reference */
  uint16_t oldGainOffsetValue = (uint16_t)DEVINFO->ADC0CAL0;
  uint32_t newGainOffsetValue;

  SegmentLCD_Write("CAL RUN");

  /* Calibrated offset and gain for 1.25V reference */
  newGainOffsetValue = ADC_Calibration(ADC0, adcRef1V25);

  /* Write to LCD */
  SegmentLCD_UnsignedHex(oldGainOffsetValue);
  SegmentLCD_LowerHex(0x0CAB0000 + (uint16_t)newGainOffsetValue);
  adcReset();
}

/******************************************************************************
 * @brief Main function
 *****************************************************************************/
int main(void)
{
  /* Initialize chip */
  CHIP_Init();

  /* Set the clock frequency to 11 MHz */
  CMU_HFRCOBandSet(cmuHFRCOBand_11MHz);

  /* Enable ADC clock */
  CMU_ClockEnable(cmuClock_ADC0, true);

  /* Initialize LCD, use LFRCO */
  SegmentLCD_Init(false);

  /* Initialize RTC with LFRCO */
  rtcSetup();

  /* Setup GPIO with interrupts to serve the pushbuttons */
  gpioSetup();
  
  /* Wait key press to process */
  SegmentLCD_Write("SING");
  while (1)
  {
    EMU_EnterEM2(false);

    /* Push button PB0 to browse menu */
    if (menuKey)
    {
      menuKey = false;
      LCD_NUMBER_OFF();
      
      switch (menuLevel)
      {
      case SING:
        SegmentLCD_Write("SING");
        break;
        
      case SCAN:
        SegmentLCD_Write("SCAN");
        break;
        
      case DIFF:
        SegmentLCD_Write("DIFF");
        break;

      case OVS:
        SegmentLCD_Write("OVS");
        break;        

      case TPRS:
        SegmentLCD_Write("PRS");
        break;

      case INT:
        SegmentLCD_Write("INT");
        break;
        
      case WFE:
        SegmentLCD_Write("WFE");
        break;

      case CAB:
        SegmentLCD_Write("CAL");
        break;

      default:
        break;
      }
    }

    /* Push button PB1 to run the selected menu */
    if (runKey)
    {
      runKey = false;
      switch (menuLevel)
      {
      case SING:
        adcSingle();
        break;
        
      case SCAN:
        adcScanDma();
        break;
        
      case DIFF:
        adcSingleDiff();
        break;

      case OVS:
        adcSingleOvs();
        break;
        
      case TPRS:
        adcTImerPrs();
        break;

      case INT:
        adcSingleInt();
        break;
        
      case WFE:
        adcSingleWfe();
        break;

      case CAB:
        adcCalibration();
        break;

      default:
        break;
      }
    }
  }
}
