/**************************************************************************//**
 * @file main_prs_example.c
 * @brief PRS Examples for EFM32 Gecko Series 0 and 1
 * @version 1.0.8
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "main_prs_example.h"

/* Global variables */
volatile bool runKey;
volatile bool menuKey;
volatile bool progRun;
volatile uint8_t menuLevel;
volatile uint32_t bufferTemp;

/**************************************************************************//**
 * @brief Main function
 *****************************************************************************/
int main(void)
{
  /* Chip errata */
  CHIP_Init();

  /* Init DCDC regulator if available */
#if defined( _EMU_DCDCCTRL_MASK )  
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif
  
  /* Setup MCU clock */
#if defined( _CMU_HFRCOCTRL_BAND_MASK )
  CMU_HFRCOBandSet(cmuHFRCOBand_14MHz);
#else
  CMU_HFRCOFreqSet(cmuHFRCOFreq_16M0Hz);
#endif
  
#if defined (MSC_CTRL_IFCREADCLEAR)  
  /* Enable atomic read-clear operation on reading IFC register */
  MSC->CTRL |= MSC_CTRL_IFCREADCLEAR;
#endif
  
  /* Setup GPIO with interrupts to serve the pushbuttons */
  gpioSetup();

#if defined(USE_SEGMENT_LCD)
  /* Initialize LCD, use LFRCO for LFA */
  SegmentLCD_Init(false);
#else
  /* Select ULFRCO as clock source for LFA so the PCNT can work in EM3 */
  CMU_ClockSelectSet(cmuClock_LFA, cmuSelect_ULFRCO);  

  /* Initialize the display module. */
  DISPLAY_Init();

  /* Retarget stdio to a text display. */
  if (RETARGET_TextDisplayInit() != TEXTDISPLAY_EMSTATUS_OK)
  {
    /* Text display initialization failed. */
    while (1) ;
  }
#endif
  
  /* Display menu */
  printMenu();
  
  /* Enter EM2 or EM3 to wait key press */
  while (1)
  {
#if defined(USE_SEGMENT_LCD)
    EMU_EnterEM2(false);
#else
    EMU_EnterEM3(false);
#endif
    
    /* Push button PB0 or BTN1 to browse menu */
    if (menuKey)
    {
      printMenu();
    }

    /* Push button PB1 or BTN0 to run the selected menu */
    if (runKey)
    {
      runKey = false;
      progRun = true;
#if !defined(USE_SEGMENT_LCD)
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
#endif
      
      switch (menuLevel)
      {
      case PRS_ADC:
#if !defined(USE_SEGMENT_LCD)
        printf("\n\n\n\n");
        printf("Trigger interval ~1s\n");
        printf("AVDD: %1.4fV\n", 0.0);
        printf("                   ");
        printf("\nPress BTN1 to exit"); 
#endif
        prsTimerAdc();
        break;
        
      case PRS_PUL:
#if !defined(USE_SEGMENT_LCD)
        printf("\n\n\n\n\n");
        printf("Positive pulse width:\n");
        printf("%1.4fs\n", 0.0);
        printf("                   ");
        printf("\nPress BTN1 to exit"); 
#endif
        prsAcmpCapture();
        break;
        
      case PRS_USART:
#if !defined(USE_SEGMENT_LCD)
        printf("\n\n\n\n");
        printf("Press BTN0 to trigger\n");
        printf("USART TX on PC7\n");
        printf("                   ");
        printf("\nCharacter:");
        printf("\n\nPress BTN1 to exit"); 
#endif
        prsGpioUsart();
        break;

#if defined(USE_SEGMENT_LCD)
      case PRS_DAC:
        prsSwDac();
        break;
        
      case PRS_MON:
        prsMonitor();
        break;
#endif
        
#if !defined(_EFM32_GECKO_FAMILY)
      case PRS_PCNT:
#if !defined(USE_SEGMENT_LCD)
        printf("\n\n\n\n");
        printf("Press BTN0 to trigger\n");
        printf("PCNT\n");
        printf("                   ");
        printf("\nPCNT CNT: %05d", 0);
        printf("\n\nPress BTN1 to exit"); 
#endif
        prsGpioPcnt();
        break;
#endif
        
#if !defined(USE_SEGMENT_LCD)
      case PRS_WFE:
        printf("\n\n\n\n");
        printf("Periodic (~1s) PRS \n");
        printf("Event from TIMER\n");
        printf("                   ");
        printf("\nPRS Wakeup: %05d", 0);
        printf("\n\nPress BTN1 to exit"); 
        prsTimerWfe();
        break;
        
      case PRS_LOGIC:
        printf("\n\n\n\n\n\n");
        printf("Sleep in EM3      \n");
        printf("    ");
        printf("\nWakeup if voltage in\n");
        printf("PA0 is outside the\n");
        printf("compare window\n");
        printf("(0.5V - 3V)");
        printf("\n\nPress BTN1 to exit"); 
        prsConfigLogic();          
        break;        
#endif
        
      default:
        break;
      }
    }
  }
}
