/*****************************************************************************
 * @file prs_common.c
 * @brief PRS demo application, common functions for this application note
 * @version 1.0.8
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "main_prs_example.h"

/* Define for PRS */
#define PRS_DEMO_CH             5

/**********************************************************
 * @brief Interrupt handler for push button PB0 or BTN1. 
 **********************************************************/
#if defined(_EFM32_TINY_FAMILY)
void GPIO_EVEN_IRQHandler(void)
#else
void GPIO_ODD_IRQHandler(void)
#endif
{
  GPIO->IFC = GPIO->IF;

  /* Next menu item if no example is running */
  if (!progRun && !menuKey)
  {
    menuKey = true;
    if (++menuLevel == MENU_MAX)
    {
      menuLevel = PRS_ADC;
    }
  }
  else
  {
    progRun = false;
  }    
}

/**********************************************************
 * @brief Interrupt handler for push button PB1 or BTN0. 
 **********************************************************/
#if defined(_EFM32_TINY_FAMILY)
void GPIO_ODD_IRQHandler(void)
#else
void GPIO_EVEN_IRQHandler(void)
#endif
{
  GPIO->IFC = GPIO->IF;

  /* No action if example is running */
  if (!progRun)
  {
    runKey = true;
  }
}

/**************************************************************************//**
 * @brief Setup GPIO interrupt for push buttons.
 *****************************************************************************/
void gpioSetup(void)
{
  /* Enable GPIO clock */
  CMU_ClockEnable(cmuClock_GPIO, true);

  /* Configure push button 0 as input and enable interrupt  */
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, BSP_GPIO_PB0_PIN, false, true, true);

  /* Configure push button 1 as input and enable interrupt */
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInputPullFilter, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, BSP_GPIO_PB1_PIN, false, true, true);
  
  /* Enable interrupt in core for even and odd GPIO interrupts */
  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
}

#if !defined(USE_SEGMENT_LCD)
/**************************************************************************//**
* @brief Dummy funtion for Memory LCD initialization.
*        RTCC is used for PRS trigger so no toggle on
*        Memory LCD external COM inversion signal pin.
*****************************************************************************/
EMSTATUS PAL_GpioPinAutoToggle (unsigned int gpioPort,
                                unsigned int gpioPin,
                                unsigned int frequency)
{
  EMSTATUS status = EMSTATUS_OK;
  return status;
}
#endif

/***************************************************************************//**
* @brief Print the menu.
*******************************************************************************/
void printMenu(void)
{
#if defined(USE_SEGMENT_LCD)
  switch (menuLevel)
  {
  case PRS_ADC:
    SegmentLCD_Write("ADC");
    break;
    
  case PRS_PUL:
    SegmentLCD_Write("PUL");
    break;
    
  case PRS_USART:
    SegmentLCD_Write("UART");
    break;
    
  case PRS_DAC:
    SegmentLCD_Write("DAC");
    break;
    
  case PRS_MON:
    SegmentLCD_Write("MON");
    break;
    
  case PRS_PCNT:        
    SegmentLCD_Write("PCNT");
    break;
    
  default:
    break;
  }
  LCD_NUMBER_OFF();
  menuKey = false;
#else
  printf("\f");
  printf("Example %d\n", menuLevel+1);
  
  switch (menuLevel)
  {
  case PRS_ADC:
    printf("Timer Triggered\n");
    printf("ADC Conversion\n\n"); 
    break;
    
  case PRS_PUL:
    printf("Pulse Width\n");
    printf("Measurement on PA4\n"); 
    printf("with ACMP and TIMER\n\n"); 
    break;
    
  case PRS_USART:
    printf("GPIO Triggered USART\n");
    printf("Transmission\n\n"); 
    break;
    
  case PRS_PCNT:
    printf("Asynchronous GPIO PRS\n");
    printf("Triggered PCNT in EM3\n\n"); 
    break;

  case PRS_WFE:
    printf("Event on PRS - WFE in\n");
    printf("EM1\n\n"); 
    break;
    
  case PRS_LOGIC:
    printf("RTCC Triggered\n");
    printf("ADC Conversion\n"); 
    printf("with Configurable\n"); 
    printf("PRS Logic in EM3\n\n"); 
    break;        
    
  default:
    break;
  }
  printf("Press BTN1 to next\n");
  printf("menu\n");
  printf("Press BTN0 to start"); 
  menuKey = false;
#endif
}

/**************************************************************************//**
 * @brief Disable clocks and reset PRS to exit
 *****************************************************************************/
void prsDemoExit(void)
{
  CMU_ClockEnable(cmuClock_PRS, true);
  PRS_SourceSignalSet(PRS_DEMO_CH, PRS_CH_CTRL_SOURCESEL_NONE, 0, prsEdgeOff);

  CMU_ClockEnable(cmuClock_ACMP0, false);
  CMU_ClockEnable(cmuClock_ADC0, false);
  CMU_ClockEnable(cmuClock_PCNT0, false);
  CMU_ClockEnable(cmuClock_PRS, false);
  CMU_ClockEnable(cmuClock_TIMER0, false);
  CMU_ClockEnable(USART_CLOCK, false);
#if defined(USE_SEGMENT_LCD)
  CMU_ClockEnable(cmuClock_DAC0, false);
#else
  CMU_ClockEnable(cmuClock_CORELE, false);
#endif
  
#if defined(USE_SEGMENT_LCD)
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInputPullFilter, 1);
#else
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeInputPullFilter, 1);
#endif  
  printMenu();
}
