/*****************************************************************************
 * @file prs_gpio_pcnt.c
 * @brief PRS demo application, GPIO triggering PCNT
 * @version 1.0.8
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "main_prs_example.h"

/* Defines for PCNT */
#define PRS_PCNT_CH             5
#define PCNT_PRS_CH             pcntPRSCh5
#define PCNT_PRS_CH_LEVEL       PRS_SWLEVEL_CH5LEVEL
#define PCNT_TOP_VALUE          9999

/***************************************************************************//**
 * @brief
 *   Use GPIO as asynchronous PRS producer for PCNT.
 *
 * @details
 *   This example uses an external signal coming through the GPIO to clock the
 *   pulse counter (PCNT). Asynchronous mode PRS is used since both GPIO and 
 *   PCNT can support this feature.
 *
 * @note
 *   The GPIO interrupt is enabled to wake up the MCU from EM2 or EM2.
 ******************************************************************************/
void prsGpioPcnt(void)
{
  /* Enable necessary clocks */
  CMU_ClockEnable(cmuClock_CORELE, true);
  CMU_ClockEnable(cmuClock_PCNT0, true);
  CMU_ClockEnable(cmuClock_PRS, true);

  PCNT_Init_TypeDef initPCNT = PCNT_INIT_DEFAULT;
  
  /* Initialize PCNT. */
  initPCNT.mode = pcntModeExtSingle;    /* External, single mode. */
  initPCNT.top = PCNT_TOP_VALUE;        /* Counter top value. */
  initPCNT.s1CntDir = false;            /* Counter direction is given by S1. */
  initPCNT.s0PRS = PCNT_PRS_CH;         /* Select PRS channel for SOIN */
  PCNT_Init(PCNT0, &initPCNT);

  /* Enable PRS input S0 in PCNT. */
  PCNT_PRSInputEnable(PCNT0, pcntPRSInputS0, true);

  /* Sync up to end PCNT initialization */ 
  while(PCNT0->SYNCBUSY)
  {
    PRS_LevelSet(PCNT_PRS_CH_LEVEL, PCNT_PRS_CH_LEVEL);
    PRS_LevelSet(~PCNT_PRS_CH_LEVEL, PCNT_PRS_CH_LEVEL);
  }
  
  /* Select GPIO as source and PB1 or BTN0 as signal, GPIO is initialized by gpioSetup() */
#if defined(_EFM32_GIANT_FAMILY) || defined(_EFM32_WONDER_FAMILY) || defined(_EFM32_LEOPARD_FAMILY)
  PRS_SourceAsyncSignalSet(PRS_PCNT_CH, PRS_CH_CTRL_SOURCESEL_GPIOH, PRS_CH_CTRL_SIGSEL_GPIOPIN10);
#elif defined(_EFM32_TINY_FAMILY)
  PRS_SourceAsyncSignalSet(PRS_PCNT_CH, PRS_CH_CTRL_SOURCESEL_GPIOH, PRS_CH_CTRL_SIGSEL_GPIOPIN11);
#else
  PRS_SourceAsyncSignalSet(PRS_PCNT_CH, PRS_CH_CTRL_SOURCESEL_GPIOL, PRS_CH_CTRL_SIGSEL_GPIOPIN6);
#endif
  /* PRS clock is not required in ASYNC mode */
  CMU_ClockEnable(cmuClock_PRS, false);

#if defined(USE_SEGMENT_LCD)
  SegmentLCD_Write("PCNTRUN");
  SegmentLCD_Number(0);
#endif
  
  /* Enter EM2 or EM3 to wait GPIO trigger */
  while(1)
  {
#if defined(USE_SEGMENT_LCD)
    EMU_EnterEM2(false);
#else
    EMU_EnterEM3(false);
#endif    
    if (progRun)
    {
      /* Write PCNT counter on LCD if not exit */
#if defined(USE_SEGMENT_LCD)
      SegmentLCD_Number(PCNT0->CNT);
#else
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
      printf("\n\n\n\n\n\n\n");
      printf("PCNT CNT: %05lu", PCNT0->CNT);
#endif
    }
    else
    {
      break;
    }
  }
  prsDemoExit();
}
