/*****************************************************************************
 * @file prs_gpio_usart.c
 * @brief PRS Demo Application, GPIO input signal enables USART TX
 * @version 1.0.8
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "main_prs_example.h"

/* Defines for USART */
#define PRS_USART_CH            5
#define USART_PRS_CH            USART_TRIGCTRL_TSEL_PRSCH5

/***************************************************************************//**
 * @brief
 *   Use GPIO as PRS producer for USART.
 *
 * @details
 *   This example uses an external signal coming through the GPIO to enable the
 *   USART transmitter.
 *
 * @note
 *   The GPIO interrupt is enabled to wake up the MCU from EM1.
 *   The GPIO generates a level signal and the USART consumes a pulse so it has
 *   to use the edge detector to generate the pulse.
 ******************************************************************************/
void prsGpioUsart(void)
{
  uint8_t txChar = '0';
  
  /* Enable necessary clocks */
  CMU_ClockEnable(cmuClock_PRS, true);
  CMU_ClockEnable(USART_CLOCK, true);
  
  USART_InitAsync_TypeDef usartInit = USART_INITASYNC_DEFAULT;
  
  /* Initialize USART with TX and RX disable */
  usartInit.enable = usartDisable;
  USART_InitAsync(USART_USE, &usartInit);
  
  /* Select PRS channel to set TXEN to enable the transmiter */
  USART_USE->TRIGCTRL |= USART_PRS_CH | USART_TRIGCTRL_TXTEN;

#if defined(USE_SEGMENT_LCD)
  /* Enable TX on location 1 (PD0) */
  GPIO_PinModeSet(gpioPortD, 0, gpioModePushPull, 1); 
#else
  /* Enable TX on location 12 (PC7) */
  GPIO_PinModeSet(gpioPortC, 7, gpioModePushPull, 1); 
  USART_USE->ROUTELOC0 = USART_ROUTELOC0_TXLOC_LOC12;
#endif  
  
  /* Select GPIO as source and PB1 or BTN0 as signal */ 
#if defined(_EFM32_GIANT_FAMILY) || defined(_EFM32_WONDER_FAMILY) || defined(_EFM32_LEOPARD_FAMILY) || defined(_EFM32_GECKO_FAMILY)
  PRS_SourceSignalSet(PRS_USART_CH, PRS_CH_CTRL_SOURCESEL_GPIOH, PRS_CH_CTRL_SIGSEL_GPIOPIN10, prsEdgeNeg);
#elif defined(_EFM32_TINY_FAMILY)
  PRS_SourceSignalSet(PRS_USART_CH, PRS_CH_CTRL_SOURCESEL_GPIOH, PRS_CH_CTRL_SIGSEL_GPIOPIN11, prsEdgeNeg);
#else
  PRS_SourceSignalSet(PRS_USART_CH, PRS_CH_CTRL_SOURCESEL_GPIOL, PRS_CH_CTRL_SIGSEL_GPIOPIN6, prsEdgeNeg);
#endif

  if (USART_USE->STATUS & USART_STATUS_TXENS)
  {
    /* TXPEN is 0, wait dummy TX to debounce the push button */
    USART_Tx(USART_USE, txChar);
    while (!(USART_USE->STATUS & USART_STATUS_TXC))
      ;
    USART_USE->CMD = USART_CMD_TXDIS;
  }
  
  /* Put a char in the buffer - it won't be transmited because USART TX is disabled */
#if defined(USE_SEGMENT_LCD)
  SegmentLCD_Write("UARTRUN");
#endif
  USART_Tx(USART_USE, txChar);
#if defined(USE_SEGMENT_LCD)
  USART_USE->ROUTE |= USART_ROUTE_TXPEN | USART_ROUTE_LOCATION_LOC1;
#else
  USART_USE->ROUTEPEN = USART_ROUTEPEN_TXPEN;
#endif

  /* Enter EM1 to wait GPIO trigger */
  while(1)
  {
    EMU_EnterEM1();
    if (progRun)
    {
      /* Write TX char on LCD if not exit */
#if defined(USE_SEGMENT_LCD)
      SegmentLCD_Number(txChar - '0');
#else
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
      printf("\n\n\n\n\n\n\n");
      printf("Character: %c", txChar);
#endif
      if (++txChar == ':')
      {
        txChar = '0';
      }
      /* Disable USART TX after transmission */
      while (!(USART_USE->STATUS & USART_STATUS_TXC))
        ;
      USART_USE->CMD = USART_CMD_TXDIS;
      /* Fill TX buffer to wait next trigger */
      USART_Tx(USART_USE, txChar);  
    }
    else
    {
      break;
    }
  }

  USART_Reset(USART_USE);
  prsDemoExit();
}
