/*****************************************************************************
 * @file prs_pulse_width_channel_scan.c
 * @brief PRS demo application
 *        Pulse width capture width ACMP and TIMER
 *        Scan a PRS channel using a TIMER             
 * @version 1.0.8
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "main_prs_example.h"

/* Defines for ACMP and TIMER */
#if defined(_ACMP_INPUTSEL_VASEL_MASK)                  /* For EFM32 Gecko Series 1 */
#define ACMP_INPUT              acmpInputAPORT3XCH12    /* PA4 */
#else
#define ACMP_INPUT              acmpChannel4            /* PC4 */
#endif
#define ACMP_THRESHOLD_LEVEL    32                      /* VDD/2 or AVDD/2*/
#define PRS_TIMER_CH            5
#define TIMER_PRS_CH            timerPRSSELCh5

/**************************************************************************//**
 * @brief TIMER0 IRQ handler.
 *****************************************************************************/
void TIMER0_IRQHandler(void)
{ 
  /* Clear flag for TIMER0 CC0 interrupt, read capture value */
  TIMER_IntClear(TIMER0, TIMER_IF_CC0);
  bufferTemp = TIMER_CaptureGet(TIMER0, 0);
}

/***************************************************************************//**
 * @brief
 *   Use ACMP as PRS producer for TIMER capture.
 *
 * @details
 *   This example measures the pulse width or period of an arbitrary waveform.
 *   The analog comparator is used to send a level signal through the PRS. 
 *
 * @note
 *   TIMER0 consumes both pulse and level signals, so the PRS leaves the
 *   incoming signal unchanged. On TIMER0, the PRS signal is used as input for
 *   Compare/Capture channel 0 (CC0). TIMER0 starts counting on a rising edge
 *   and captures the counter value on a falling edge. 
 ******************************************************************************/
void prsAcmpCapture(void)
{
  /* Enable necessary clocks */
  CMU_ClockEnable(cmuClock_ACMP0, true);
  CMU_ClockEnable(cmuClock_PRS, true);
  CMU_ClockEnable(cmuClock_TIMER0, true);

  ACMP_Init_TypeDef acmpInit = ACMP_INIT_DEFAULT;
#if defined(_ACMP_INPUTSEL_VASEL_MASK)                  /* For EFM32 Gecko Series 1 */
  ACMP_VAConfig_TypeDef ACMP0_VAinit = ACMP_VACONFIG_DEFAULT;
#endif
  
  /* Init ACMP */
  acmpInit.enable = false;                              /* Do not enable after init */
#if !defined(_ACMP_INPUTSEL_VASEL_MASK)                 /* For EFM32 Gecko Series 0 */
  acmpInit.warmTime = acmpWarmTime256,                  /* >140 cycles for >10us warm-up @ 14MHz */
  acmpInit.hysteresisLevel = acmpHysteresisLevel0;      /* No hysteresis */
  acmpInit.vddLevel = ACMP_THRESHOLD_LEVEL;             /* Scale VDD reference */
#endif
  ACMP_Init(ACMP0, &acmpInit);
  
#if !defined(_ACMP_INPUTSEL_VASEL_MASK)                 /* For EFM32 Gecko Series 0 */
  /* Configure PC4 as input with pull down for ACMP channel 4 input */
  GPIO_PinModeSet(gpioPortC, 4, gpioModeInputPullFilter, 0);
  /* Set PC4 as positive input and scaled VDD as negative input */
  ACMP_ChannelSet(ACMP0, acmpChannelVDD, ACMP_INPUT);
#else                                                   /* For EFM32 Gecko Series 1 */
  /* Set hysteresis level */
  ACMP0->HYSTERESIS0 = acmpInit.hysteresisLevel_0;
  ACMP0->HYSTERESIS1 = acmpInit.hysteresisLevel_1;

  /* VA configuration */
  ACMP0_VAinit.div0 = ACMP_THRESHOLD_LEVEL;
  ACMP0_VAinit.div1 = ACMP_THRESHOLD_LEVEL;
  ACMP_VASetup(ACMP0, &ACMP0_VAinit);

  /* Set PA4 as positive input and scaled AVDD as negative input */
  ACMP_ChannelSet(ACMP0, acmpInputVADIV, ACMP_INPUT);
#endif
  
  TIMER_Init_TypeDef timerInit = TIMER_INIT_DEFAULT;
  TIMER_InitCC_TypeDef timerCCInit = TIMER_INITCC_DEFAULT;

  /* Initialize TIMER0 */
  timerInit.enable = false;                             /* Do not start counting when init complete */   
  timerInit.prescale = timerPrescale1024;               /* Prescaler of 1024 */
  timerInit.fallAction = timerInputActionStop;          /* Stop counter on falling edge */               
  timerInit.riseAction = timerInputActionReloadStart;   /* Reload and start on rising edge */
  TIMER_Init(TIMER0, &timerInit);  
  
  /* Initialize TIMER0 CC0 channel */
  timerCCInit.eventCtrl = timerEventFalling;    /* Input capture event control */
  timerCCInit.edge = timerEdgeFalling;          /* Input capture on falling edge */
  timerCCInit.prsSel = TIMER_PRS_CH;            /* Select PRS channel */
  timerCCInit.mode = timerCCModeCapture;        /* CC channel mode capture */
  timerCCInit.prsInput = true;                  /* CC channel PRS input */
  TIMER_InitCC(TIMER0, 0, &timerCCInit);

  /* Enable CC0 interrupt */
  TIMER_IntEnable(TIMER0, TIMER_IF_CC0);
  NVIC_ClearPendingIRQ(TIMER0_IRQn);
  NVIC_EnableIRQ(TIMER0_IRQn);
  
  /* Select ACMP as source and ACMP0OUT (ACMP0 OUTPUT) as signal */
  PRS_SourceSignalSet(PRS_TIMER_CH, PRS_CH_CTRL_SOURCESEL_ACMP0, PRS_CH_CTRL_SIGSEL_ACMP0OUT, prsEdgeOff);

  /* No wakeup from PB1 or BTN0, enable the ACMP */
#if defined(USE_SEGMENT_LCD)
  SegmentLCD_Write("PUL RUN");
  SegmentLCD_Number(0);
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeDisabled, 0);
#else
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeDisabled, 0);
#endif
  ACMP_Enable(ACMP0);

  /* Enter EM1 to wait TIMER capture trigger */
  while(1)
  {
    EMU_EnterEM1();
    if (progRun)
    {
      /* Write pulse width on LCD if not exit */
      bufferTemp <<= 10;                /* Prescaler of 1024 */
#if defined(USE_SEGMENT_LCD)
      SegmentLCD_Number(bufferTemp/(CMU_ClockFreqGet(cmuClock_HFPER)/1000));
#else
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
      printf("\n\n\n\n\n\n");
      printf("%1.4fs", (float)(bufferTemp)/CMU_ClockFreqGet(cmuClock_HFPER));
#endif      
    }
    else
    {
      break;
    }
  }
 
  ACMP_Disable(ACMP0);
  TIMER_Reset(TIMER0);
  prsDemoExit();
}

#if defined(USE_SEGMENT_LCD)
/***************************************************************************//**
 * @brief
 *   Use TIMER to monitor PRS signal.
 *
 * @details
 *   The PRS channels can be monitored using peripherals that consume PRS
 *   signals. One example is using a TIMER to make a capture when there is 
 *   activity on the PRS channel it is connected to.
 *
 * @note
 *   The Compare/Capture channel 0 (CC0) on TIMER0 is used to capture a falling
 *   edge. When a capture is triggered, the user knows that there was activity
 *   on the selected PRS channel.
 ******************************************************************************/
void prsMonitor(void)
{
  uint16_t prsCount = 0;

  /* Enable necessary clocks */
  CMU_ClockEnable(cmuClock_ACMP0, true);
  CMU_ClockEnable(cmuClock_PRS, true);
  CMU_ClockEnable(cmuClock_TIMER0, true);

  ACMP_Init_TypeDef acmpInit = ACMP_INIT_DEFAULT;

  /* Init ACMP */
  acmpInit.enable = false;                              /* Do not enable after init */
  acmpInit.warmTime = acmpWarmTime256,                  /* >140 cycles for >10us warm-up @ 14MHz */
  acmpInit.hysteresisLevel = acmpHysteresisLevel0;      /* No hysteresis */
  acmpInit.vddLevel = ACMP_THRESHOLD_LEVEL;             /* Scale VDD reference */
  ACMP_Init(ACMP0, &acmpInit);
  
  /* Configure PC4 as input with pull down for ACMP channel 4 input */
  GPIO_PinModeSet(gpioPortC, 4, gpioModeInputPullFilter, 0);
  /* Set PC4 as positive input and scaled VDD as negative input */
  ACMP_ChannelSet(ACMP0, acmpChannelVDD, ACMP_INPUT);

  TIMER_Init_TypeDef timerInit = TIMER_INIT_DEFAULT;
  TIMER_InitCC_TypeDef timerCCInit = TIMER_INITCC_DEFAULT;

  /* Initialize TIMER0 */
  timerInit.prescale = timerPrescale1024;       /* Prescaler of 1024 */
  TIMER_Init(TIMER0, &timerInit);  
  
  /* Initialize TIMER0 CC0 channel */
  timerCCInit.eventCtrl = timerEventFalling;    /* Input capture event control */
  timerCCInit.edge = timerEdgeFalling;          /* Input capture on falling edge */
  timerCCInit.prsSel = TIMER_PRS_CH;            /* Select PRS channel */
  timerCCInit.mode = timerCCModeCapture;        /* CC channel mode capture */
  timerCCInit.prsInput = true;                  /* CC channel PRS input */
  TIMER_InitCC(TIMER0, 0, &timerCCInit);

  /* Enable CC0 interrupt */
  TIMER_IntEnable(TIMER0, TIMER_IF_CC0);
  NVIC_ClearPendingIRQ(TIMER0_IRQn);
  NVIC_EnableIRQ(TIMER0_IRQn);

  /* Select ACMP as source and ACMP0OUT (ACMP0 OUTPUT) as signal */
  PRS_SourceSignalSet(PRS_TIMER_CH, PRS_CH_CTRL_SOURCESEL_ACMP0, PRS_CH_CTRL_SIGSEL_ACMP0OUT, prsEdgeOff);

  /* No wakeup from PB1, enable the ACMP */
  SegmentLCD_Write("MON RUN");
  SegmentLCD_Number(0);
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeDisabled, 0);
  ACMP_Enable(ACMP0);

  /* Enter EM1 to wait TIMER capture trigger */
  while(1)
  {
    EMU_EnterEM1();
    if (progRun)
    {
      /* Write PRS count on LCD if not exit */
      SegmentLCD_Number(++prsCount);
    }
    else
    {
      break;
    }
  }
 
  ACMP_Disable(ACMP0);
  TIMER_Reset(TIMER0);
  prsDemoExit();
}
#endif
