/*****************************************************************************
 * @file prs_rtcc_logic.c
 * @brief PRS demo application, use RTCC and PRS logic to trigger ADC
 * @version 1.0.8
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "main_prs_example.h"

/* Definess for ADC */
#define ADC_INPUT               adcPosSelAPORT3XCH8     /* PA0 */
#define ADC_ASYNC_CLOCK         cmuAUXHFRCOFreq_13M0Hz  /* Clock for ASYNC mode */
#define ADC_ASYNC_HFRCO         cmuHFRCOFreq_13M0Hz
#define ADC_CMP_GT_VALUE        3724                    /* ~3.0V for 3.3V AVDD */                   
#define ADC_CMP_LT_VALUE        620                     /* ~0.5V for 3.3V AVDD */
#define ADC_PRS_CH_SELECT       adcPRSSELCh4
#define ADC_AVDD_VFS            (float)3.3              /* AVDD */
#define ADC_12BIT_MAX           4096                    /* 2^12 */

/* Defines for RTCC */
#define RTCC_CC_CHANNEL         1
#define RTCC_PRS_CHANNEL        4                       /* =ADC_PRS_CH_SELECT */ 
#define RTCC_PRS_CH_SELECT      rtccPRSCh4              /* =ADC_PRS_CH_SELECT */
#define LFE_CLOCK_SOURCE        cmuSelect_ULFRCO        /* 1 KHz */
#define RTCC_WAKEUP_MS          10                      /* Trigger interval in ms */
#define RTCC_WAKEUP_COUNT       (((1000 * RTCC_WAKEUP_MS) / 1000) - 1)

/* Defines for PRS output */
#define PRS_DEBUG_OUT           0                       
#define PRS_TRIG_PORT           gpioPortD
#define PRS_TRIG_PIN            10
#define PRS_TRIG_ROUTE_REG      ROUTELOC1
#define PRS_TRIG_ROUTE_LOC      PRS_ROUTELOC1_CH4LOC_LOC1
#define PRS_TRIG_ROUTE_PEN      PRS_ROUTEPEN_CH4PEN

/**************************************************************************//**
 * @brief Setup RTCC as PRS producer to trigger ADC in EM2 or EM3.
 *****************************************************************************/
void rtccSetup(void)
{
  /* Enable necessary clocks, RTCC can work in EM3 if clock source is ULFRCO */
  CMU_ClockEnable(cmuClock_CORELE, true);
  CMU_ClockSelectSet(cmuClock_LFE, LFE_CLOCK_SOURCE);  
  CMU_ClockEnable(cmuClock_RTCC, true);

  RTCC_Init_TypeDef rtccInit = RTCC_INIT_DEFAULT;
  RTCC_CCChConf_TypeDef rtccInitCompareChannel = RTCC_CH_INIT_COMPARE_DEFAULT;
  
  /* Setting the compare value of the RTCC */
  rtccInitCompareChannel.prsSel = RTCC_PRS_CH_SELECT;
  RTCC_ChannelInit(RTCC_CC_CHANNEL, &rtccInitCompareChannel);
  RTCC_ChannelCCVSet(RTCC_CC_CHANNEL, RTCC_WAKEUP_COUNT);
  
  /* Clear counter on compare match, initialize the RTCC */
  rtccInit.cntWrapOnCCV1 = true;        
  rtccInit.presc = rtccCntPresc_1;
  rtccInit.enable = false;
  RTCC_Init(&rtccInit);
  RTCC_CounterSet(0);
}

/***************************************************************************//**
 * @brief
 *   Configure PRS logic to trigger ADC.
 *
 * @details
 *   Refer to application note PDF for details.
 ******************************************************************************/
void adcPrsLogic(void)
{
  /* Enable necessary clocks */
  CMU_ClockEnable(cmuClock_GPIO, true);
  CMU_ClockEnable(cmuClock_PRS, true);

  /* Use ADC SINGLE as an ASYNC PRS producer on CH0 */
  /* Maybe not properly synchronized, see errata ADC_E212 */
  PRS_SourceAsyncSignalSet(0, PRS_CH_CTRL_SOURCESEL_ADC0, PRS_CH_CTRL_SIGSEL_ADC0SINGLE);
  
  /* CH3 OR CH0 then INV on CH1 */
  PRS_SourceAsyncSignalSet(1, PRS_CH_CTRL_SOURCESEL_PRSL, PRS_CH_CTRL_SIGSEL_PRSCH3);
  PRS->CH[1].CTRL |= PRS_CH_CTRL_ORPREV | PRS_CH_CTRL_INV;
  
  /* Invert RTCC trigger on CH2 */
  PRS_SourceAsyncSignalSet(2, PRS_CH_CTRL_SOURCESEL_RTCC, PRS_CH_CTRL_SIGSEL_RTCCCCV1);
  PRS->CH[2].CTRL |= PRS_CH_CTRL_INV;
  
  /* CH1 OR CH2 then INV on CH3 */
  PRS_SourceAsyncSignalSet(3, PRS_CH_CTRL_SOURCESEL_PRSL, PRS_CH_CTRL_SIGSEL_PRSCH1);
  PRS->CH[3].CTRL |= PRS_CH_CTRL_ORPREV | PRS_CH_CTRL_INV;

  /* CH1 AND CH5 on CH4 */
  PRS_SourceAsyncSignalSet(4, PRS_CH_CTRL_SOURCESEL_PRSL, PRS_CH_CTRL_SIGSEL_PRSCH1);
  PRS->CH[4].CTRL |= PRS_CH_CTRL_ANDNEXT;

  /* Use RTCC as an ASYNC PRS producer on CH5 */
  PRS_SourceAsyncSignalSet(5, PRS_CH_CTRL_SOURCESEL_RTCC, PRS_CH_CTRL_SIGSEL_RTCCCCV1);

#if (PRS_DEBUG_OUT == 0)
  /* Enable PRS output for sensor enable */
  PRS->PRS_TRIG_ROUTE_REG |= PRS_TRIG_ROUTE_LOC;
  PRS->ROUTEPEN |= PRS_TRIG_ROUTE_PEN;
  GPIO_PinModeSet(PRS_TRIG_PORT, PRS_TRIG_PIN, gpioModePushPull, 0);
#else  
  /* Enable all PRS outputs for debugging */
  PRS->ROUTELOC0 = PRS_ROUTELOC0_CH0LOC_LOC2 | PRS_ROUTELOC0_CH1LOC_LOC2
                    | PRS_ROUTELOC0_CH2LOC_LOC2 | PRS_ROUTELOC0_CH3LOC_LOC2;
  PRS->ROUTELOC1 = PRS_ROUTELOC1_CH4LOC_LOC1 | PRS_ROUTELOC1_CH5LOC_LOC1;
  PRS->ROUTEPEN = PRS_ROUTEPEN_CH0PEN + PRS_ROUTEPEN_CH1PEN
                  + PRS_ROUTEPEN_CH2PEN + PRS_ROUTEPEN_CH3PEN
                  + PRS_ROUTEPEN_CH4PEN + PRS_ROUTEPEN_CH5PEN;
  
  GPIO_PinModeSet(gpioPortF, 2, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortF, 3, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortF, 4, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortF, 5, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortD, 10, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortD, 11, gpioModePushPull, 0);
#endif
  
  /* PRS clock is not required in ASYNC mode */
  CMU_ClockEnable(cmuClock_PRS, false);
}

/**************************************************************************//**
 * @brief Setup ADC for single conversion in EM2 or EM3.
 *****************************************************************************/
void adcSetup(void)
{
  /* Enable ADC clock */
  CMU_ClockEnable(cmuClock_ADC0, true);

  ADC_Init_TypeDef init = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  /* Select AUXHFRCO for ADC ASYNC mode so that ADC can run on EM2 or EM3 */
  CMU->ADCCTRL = CMU_ADCCTRL_ADC0CLKSEL_AUXHFRCO;

  /* Initialize compare threshold for both single and scan conversion */
  ADC0->CMPTHR = _ADC_CMPTHR_RESETVALUE;
  ADC0->CMPTHR = (ADC_CMP_GT_VALUE << _ADC_CMPTHR_ADGT_SHIFT) +
                 (ADC_CMP_LT_VALUE << _ADC_CMPTHR_ADLT_SHIFT);
  
  /* Initialize for single conversion */
  singleInit.prsSel = ADC_PRS_CH_SELECT;
  singleInit.reference = adcRefVDD;
  singleInit.posSel = ADC_INPUT;
  singleInit.negSel = adcNegSelVSS;
  singleInit.prsEnable = true;
  singleInit.fifoOverwrite = true;
  ADC_InitSingle(ADC0, &singleInit);
  
  /* Enable single window compare */
  ADC0->SINGLECTRL |= ADC_SINGLECTRL_CMPEN;
  
  /* Enable ADC single window compare interrupt, ADC ISR in prs_timer_adc.c */
  ADC_IntEnable(ADC0, ADC_IEN_SINGLECMP);

  /* Use LOWACC if not using bandgap reference to reduce current consumption */ 
  ADC0->BIASPROG = ADC_BIASPROG_GPBIASACC;
  
  /* Switch the ADCCLKMODE to ASYNC at the end of initialization */
  init.em2ClockConfig = adcEm2ClockOnDemand;
  
  /* Use AUXHFRCO frequency to setup ADC if run on EM2 or EM3 */
  CMU_AUXHFRCOFreqSet(ADC_ASYNC_CLOCK);
  CMU_HFRCOFreqSet(ADC_ASYNC_HFRCO);
  init.timebase = ADC_TimebaseCalc(CMU_AUXHFRCOBandGet());
  init.prescale = ADC_PrescaleCalc(CMU_AUXHFRCOBandGet(), CMU_AUXHFRCOBandGet());
  
  /* Init common issues for both single conversion and scan mode */
  ADC_Init(ADC0, &init);

  /* Restore main clock */
  CMU_HFRCOFreqSet(cmuHFRCOFreq_16M0Hz);
  
  /* Clear the FIFO and pending interrupt */
  ADC0->SINGLEFIFOCLEAR = ADC_SINGLEFIFOCLEAR_SINGLEFIFOCLEAR;
  NVIC_ClearPendingIRQ(ADC0_IRQn);
  NVIC_EnableIRQ(ADC0_IRQn);
}

/***************************************************************************//**
 * @brief
 *   Use configurable PRS logic to trigger ADC conversion.
 *
 * @details
 *   This example uses the configurable logic in PRS to reduce the current 
 *   consumption in a sensor application.
 *   This example is to evaluate the ADC result autonomously and only give the
 *   MCU an interrupt if the sample is outside or inside the given thresholds.
 ******************************************************************************/
void prsConfigLogic(void)
{
  /* Initialize RTCC */
  rtccSetup();
  
  /* Initialize PRS logic for ADC */
  adcPrsLogic();  

  /* Initialize ADC */
  adcSetup();
  
  /* No wakeup from BTN0 */
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeDisabled, 0);

  /* Wakeup from EM3 if sensor voltage is outside the compare window */
  while(1)
  {
    RTCC_Enable(true);          /* Disable in ADC interrupt */  
    EMU_EnterEM3(false);
    if (progRun)
    {
      /* Print ADC FIFO after SINGLECMP interrrupt if not exit */
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
      printf("\n\n\n\n\n\n");
      printf("ADC Compare Interrupt\n\n");
      bufferTemp = 0;
      while (ADC0->SINGLEFIFOCOUNT)
      {
        printf("FIFO %lu: %1.4fV     \n", bufferTemp, ((float)(ADC_DataSingleGet(ADC0)) * ADC_AVDD_VFS)/ADC_12BIT_MAX);
        bufferTemp++;
      }
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
      printf("\n\n\n\n\n\n");
      printf("Sleep in EM3         ");
      
      /* Errata ADC_E208, make sure the compare interrupt flag is clear */
      ADC0->CTRL &= ~ADC_CTRL_ADCCLKMODE;
      ADC0->SINGLECTRL &= ~ADC_SINGLECTRL_CMPEN;
      ADC_IntClear(ADC0, _ADC_IF_MASK);
      ADC0->SINGLECTRL |= ADC_SINGLECTRL_CMPEN;
      ADC0->CTRL |= ADC_CTRL_ADCCLKMODE;
      
      /* Clear the FIFO and pending interrupt */
      ADC0->SINGLEFIFOCLEAR |= ADC_SINGLEFIFOCLEAR_SINGLEFIFOCLEAR;
      NVIC_ClearPendingIRQ(ADC0_IRQn);
      NVIC_EnableIRQ(ADC0_IRQn);
    }
    else
    {
      break;
    }
  }

  /* Switch the ADCCLKMODE to SYNC to reset ADC */
  RTCC_Enable(false);  
  NVIC_DisableIRQ(ADC0_IRQn);
  ADC0->CTRL &= ~ADC_CTRL_ADCCLKMODE_ASYNC;
  ADC_Reset(ADC0);

  /* Reset all PRS channels */
  CMU_ClockEnable(cmuClock_PRS, true);
  PRS_SourceSignalSet(0, PRS_CH_CTRL_SOURCESEL_NONE, 0, prsEdgeOff);
  PRS_SourceSignalSet(1, PRS_CH_CTRL_SOURCESEL_NONE, 0, prsEdgeOff);
  PRS_SourceSignalSet(2, PRS_CH_CTRL_SOURCESEL_NONE, 0, prsEdgeOff);
  PRS_SourceSignalSet(3, PRS_CH_CTRL_SOURCESEL_NONE, 0, prsEdgeOff);
  PRS_SourceSignalSet(4, PRS_CH_CTRL_SOURCESEL_NONE, 0, prsEdgeOff);

  CMU_ClockEnable(cmuClock_RTCC, false);
  prsDemoExit();
}
