/*****************************************************************************
 * @file prs_soft_dac.c
 * @brief PRS Demo Application
 *        DAC conversion triggered by software generated PRS signal
 * @version 1.0.8
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "main_prs_example.h"

/* Defines for DAC */
#define DAC_PRS_CH              dacPRSSELCh5
#define DAC_REF_VDD             3.3             /* VDD as reference */
#define DAC_OUT_VOLT            0.5             /* 0.5V */
#define DAC_12BIT               4096            /* 2^12 */
#define PRS_CH_PULSE_MASK       0x20

/***************************************************************************//**
 * @brief
 *   Use software to generate a PRS pulse to trigger DAC.
 *
 * @details
 *   The PRS pulse will trigger a DAC conversion which outputs a 0.5 V signal
 *   on DAC output.
 *
 * @note
 *   It is possible to generate both pulse and level signals by software.
 *   In this case, a pulse signal is generated because it is the type of 
 *   signal consumed by the DAC.
 ******************************************************************************/
void prsSwDac(void)
{
  /* Enable necessary clocks */
  CMU_ClockEnable(cmuClock_DAC0, true);
  CMU_ClockEnable(cmuClock_PRS, true);

  DAC_Init_TypeDef init = DAC_INIT_DEFAULT;
  DAC_InitChannel_TypeDef initChannel = DAC_INITCHANNEL_DEFAULT;
  
  /* Initialize DAC and DAC channel 1 */
  init.prescale = DAC_PrescaleCalc(500000, 0);  /* DAC clock close to 500 KHz */         
  init.reference = dacRefVDD;                   /* Set reference to VDD */
  DAC_Init(DAC0, &init);

  initChannel.enable = true;                    /* Enable after init */
  initChannel.prsEnable = true;                 /* PRS enable */
  initChannel.prsSel = DAC_PRS_CH;              /* Select PRS channel */
  DAC_InitChannel(DAC0, &initChannel, 1);
  
  /* The voltage will only output when the PRS pulse is generated by software */  
  DAC_Channel1OutputSet(DAC0, (uint32_t)((DAC_OUT_VOLT * DAC_12BIT)/DAC_REF_VDD));
  
  /* Generate PRS pulse by writing 1 on corresponding channel in PRS_SWPULSE register */ 
  /* This will trigger the DAC conversion and voltage can be probed on PB12 */
  SegmentLCD_Write("DAC RUN");
  SegmentLCD_Number(500);
  PRS_PulseTrigger(PRS_CH_PULSE_MASK);
  
  /* Wait PB0 press to exit */
  while(1)
  {
    EMU_EnterEM1();
    if (!progRun)
    {
      break;
    }
  }
 
  DAC_Reset(DAC0);
  prsDemoExit();
}
