/*****************************************************************************
 * @file prs_timer_adc.c
 * @brief PRS demo application, TIMER triggering ADC conversion
 * @version 1.0.8
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "main_prs_example.h"

/* Defines for ADC */
#define PRS_ADC_CH      5               /* PRS channel */
#define ADC_PRS_CH      adcPRSSELCh5
#define ADC_START_CNT   65500           /* First PRS TIMER trigger count */
#define ADC_VDD_X1000   3300            /* VDD x 1000 */
#define ADC_AVDD_VFS    (float)3.3      /* AVDD */
#define ADC_12BIT_MAX   4096            /* 2^12 */

/**************************************************************************//**
 * @brief ADC0 IRQ handler.
 *****************************************************************************/
void ADC0_IRQHandler(void)
{
#if !defined(ADC_IF_SINGLECMP)
  /* Read data and clear ADC0 interrupt flag */
  ADC_IntClear(ADC0, ADC_IFC_SINGLE);
  bufferTemp = ADC_DataSingleGet(ADC0);
#else
  if ((ADC0->IEN & ADC_IEN_SINGLE) && (ADC0->IF & ADC_IF_SINGLE))
  {
    /* Read SINGLEDATA will clear SINGLE IF flag */
    bufferTemp = ADC_DataSingleGet(ADC0);  
  }

  /* The SCANCMP and SINGLECMP interrupt flags in ADCn_IF are not clearable */
  /* in certain scenarios (Errata ADC_E208). */
  /* Workaround is to clear CMPEN before clearing the SINGLECMP IF flag */
  /* but SINGLECTRL register can only be accessed on ADC SYNC mode. */
  /* Alternative is to disable ADC interrupt to avoid multiple SCANCMP or */
  /* SINGLECMP interrupts from the same source. */
  if (ADC0->IF & ADC_IF_SINGLECMP)
  {
    /* Stop PRS trigger from RTCC, clear interrupt flag */
    RTCC_Enable(false);  
    NVIC_DisableIRQ(ADC0_IRQn);
    ADC_IntClear(ADC0, _ADC_IF_MASK);
  }
#endif  
}

/***************************************************************************//**
 * @brief
 *   Use TIMER as PRS producer for ADC trigger.
 *
 * @details
 *   This example triggers an ADC conversion every time that TIMER0 overflows.
 *   TIMER0 sends a one HFPERCLK cycle high pulse through the PRS on each 
 *   overflow and the ADC does a single conversion.
 *
 * @note
 *   The ADC consumes pulse signals which is the same signal produced by the
 *   TIMER. In this case, there is no edge detection needed, and the PRS leaves
 *   the incoming signal unchanged.
 ******************************************************************************/
void prsTimerAdc(void)
{
  /* Enable necessary clocks */
  CMU_ClockEnable(cmuClock_ADC0, true);
  CMU_ClockEnable(cmuClock_PRS, true);
  CMU_ClockEnable(cmuClock_TIMER0, true);

  TIMER_Init_TypeDef timerInit = TIMER_INIT_DEFAULT;

  /* Initialize TIMER0 */
  timerInit.enable = false;                     /* Do not start after init */
  timerInit.prescale = timerPrescale256;        /* Overflow after aprox 1s */
  TIMER_Init(TIMER0, &timerInit);  
  
  ADC_Init_TypeDef init = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef initSingle = ADC_INITSINGLE_DEFAULT;

  /* Init common settings for both single conversion and scan mode */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(CMU_ClockFreqGet(cmuClock_HFPER)/2, 0);
  ADC_Init(ADC0, &init);
  
  /* Initialize ADC single sample conversion */
  initSingle.prsSel = ADC_PRS_CH;       /* Select PRS channel */
  initSingle.acqTime = adcAcqTime8;     /* Acquisition time of 8 ADC clock cycles */
  initSingle.reference = adcRefVDD;     /* VDD or AVDD as ADC reference */
#if defined(_ADC_SINGLECTRL_POSSEL_MASK)
  initSingle.posSel = adcPosSelAVDD;    /* AVDD as ADC input */
  initSingle.negSel = adcNegSelVSS;
#else
  initSingle.input = adcSingleInpVDD;   /* VDD as ADC input */
#endif
  initSingle.prsEnable = true;          /* PRS enable */
  ADC_InitSingle(ADC0, &initSingle);
  
  /* Enable ADC Interrupt when Single Conversion Complete */
  ADC_IntEnable(ADC0, ADC_IEN_SINGLE);
  NVIC_ClearPendingIRQ(ADC0_IRQn);
  NVIC_EnableIRQ(ADC0_IRQn);

  /* Select TIMER0 as source and timer overflow as signal */
  PRS_SourceSignalSet(PRS_ADC_CH, PRS_CH_CTRL_SOURCESEL_TIMER0, PRS_CH_CTRL_SIGSEL_TIMER0OF, prsEdgeOff);

  /* No wakeup from PB1 or BTN0, start timer to trigger ADC */
#if defined(USE_SEGMENT_LCD)
  SegmentLCD_Write("ADC RUN");
  SegmentLCD_Number(0);
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeDisabled, 0);
#else
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeDisabled, 0);
#endif
  TIMER_CounterSet(TIMER0, ADC_START_CNT);
  TIMER_Enable(TIMER0, true);

  /* Enter EM1 to wait ADC trigger */
  while(1)
  {
    EMU_EnterEM1();
    if (progRun)
    {
      /* Write VDD or AVDD on LCD if not exit */
#if defined(USE_SEGMENT_LCD)
      SegmentLCD_Number((bufferTemp * ADC_VDD_X1000)/ADC_12BIT_MAX);
#else
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
      printf("\n\n\n\n\n");
      printf("AVDD: %1.4fV\n",((float)bufferTemp * ADC_AVDD_VFS)/ADC_12BIT_MAX); 
#endif
    }
    else
    {
      break;
    }
  }
  TIMER_Reset(TIMER0);
  ADC_Reset(ADC0);
  prsDemoExit();
}  
