/*****************************************************************************
 * @file prs_timer_wfe.c
 * @brief PRS demo application, TIMER sending event to the MCU
 * @version 1.0.8
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "main_prs_example.h"

/* Define for WFE */
#define PRS_WFE_CH      5

/***************************************************************************//**
 * @brief
 *   Use PRS to wake up the MCU from WFE.
 *
 * @details
 *   This example demonstrates how to send event to the MCU through the PRS.
 *   The TIMER0 is setup to trigger an event to the MCU periodically
 *   (approximately every one second), every time letting the MCU pass
 *   through a Wait For Event (WFE) instruction in its program.
 *
 * @note
 *   The WFE consumes pulse signals which is the same signal produced by the
 *   TIMER. In this case, there is no edge detection needed, and the PRS leaves
 *   the incoming signal unchanged.
 ******************************************************************************/
void prsTimerWfe(void)
{
  uint16_t wakeupCount = 0;
  
  /* Enable necessary clocks */
  CMU_ClockEnable(cmuClock_PRS, true);
  CMU_ClockEnable(cmuClock_TIMER0, true);

  TIMER_Init_TypeDef timerInit = TIMER_INIT_DEFAULT;

  /* Initialize TIMER0 */
  timerInit.enable = false;                     /* Do not start after init */
  timerInit.prescale = timerPrescale256;        /* Overflow after aprox 1s */
  TIMER_Init(TIMER0, &timerInit);  
  
  /* Select TIMER0 as source and timer overflow as signal */
  PRS_SourceSignalSet(PRS_WFE_CH, PRS_CH_CTRL_SOURCESEL_TIMER0, PRS_CH_CTRL_SIGSEL_TIMER0OF, prsEdgeOff);

  /* Event is generated to the MCU when the selected PRS channel is high */
  PRS->CTRL = PRS_CTRL_SEVONPRSSEL_PRSCH5 + PRS_CTRL_SEVONPRS;
  
  /* Wakeup MCU from WFE, GPIO (BTN1) interrupt in the NVIC is not needed  */
  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_DisableIRQ(GPIO_ODD_IRQn);
  
  /* Set the SEVONPEND bit to wait an event trigger in EM1 */
  SCB->SCR |= SCB_SCR_SEVONPEND_Msk;
  SCB->SCR &= ~SCB_SCR_SLEEPDEEP_Msk;
  
  /* Set event and clear it to put the event register bit in a known state
  (cleared) before going to sleep by waiting for the next event */
  __SEV();
  __WFE();

  /* No wakeup from BTN0, start timer to generate PRS event */
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeDisabled, 0);
  TIMER_Enable(TIMER0, true);

  /* Enter EM1 with WFE to wait TIMER trigger */
  while(1)
  {
    /* Go to sleep while waiting for event (set by IRQ pending bit) */
    /* Since IRQ is not enabled in the NVIC, no ISR will be entered */
    __WFE();

    /* Check wakeup is from BTN1 GPIO or TIMER PRS event */
    bufferTemp = GPIO->IFC;
    if (bufferTemp & (1 << BSP_GPIO_PB1_PIN))
    {
      /* Clear pending GPIO ODD IRQ, reset SEVONPEND bit if from BTN1 */
      NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);	
      SCB->SCR &= ~SCB_SCR_SEVONPEND_Msk;
      progRun = false;
      break;
    }
    else
    {
      /* Write PRS wakeup count on LCD if from TIMER PRS */
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
      printf("\n\n\n\n\n\n\n");
      printf("PRS Wakeup: %05d", ++wakeupCount);
    }
  }
  
  /* Re-enable interrupt on push button BTN1 */
  TIMER_Reset(TIMER0);
  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
  prsDemoExit();
}
