/******************************************************************************
 * @file letimer_common.c
 * @brief LETIMER Demo Application
 * @author Silicon Labs
 * @version  1.09

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 ******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 *****************************************************************************/
#include "em_cmu.h"
#include "em_letimer.h"
#include "em_gpio.h"
#include "config.h"
#if defined( RTCC_COUNT ) && ( RTCC_COUNT == 1 )
#include "em_rtcc.h"
#include "em_prs.h"
#else
#include "em_rtc.h"
#endif
#include "bspconfig.h"

/* GLOBAL DEFINES */
/* Maximum value for letimer compare registers */
#define COMPMAX                 500   /* 32.768k/500 = 65Hz */

/* Maximum value for compare registers */
#define LETIMER_TOP             100   /* oneshot 32.768k/200 = 160Hz */
#define LETIMER_REPEAT          5     /* one-shot 5 pulse */
#define RTC_COMP                5     /* 5 seconds with 32k prescale LFRCO */
#define RTCC_COMP               5     /* 5 seconds with 32k prescale LFRCO */

/* PRS channel used */
#define RTCC_PRS_CHANNEL        0     /* RTCC tigger letimer PRS channel */
#define GPIO_START_PRS_CHANNEL  0     /* GPIO start letimer PRS channel */
#define GPIO_STOP_PRS_CHANNEL   1     /* GPIO stop letimer PRS channel */

/* GLOBAL VARIABLES */
/* COMP1 is changed throughout the code to vary the PWM duty cycle 
   but starts with the maximum value (100% duty cycle) */
uint16_t comp1 = COMPMAX;

/* LETIMER */
LETIMER_Init_TypeDef letimerInit = 
{
  .enable         = true,             // Start counting when init completed.
  .debugRun       = false,            // Counter shall not keep running
                                      // during debug halt.
  .comp0Top       = true,             // Load COMP0 register into CNT
                                      // when counter underflows.
                                      // COMP0 is used as TOP.
  .bufTop         = false,            // Don't load COMP1 into COMP0
                                      // when REP0 reaches 0.
  .out0Pol        = 0,                // Idle value for output 0.
  .out1Pol        = 0,                // Idle value for output 1.
  .ufoa0          = letimerUFOAPwm,   // PWM output on output 0.
  .ufoa1          = letimerUFOAPulse, // Pulse output on output 1.
  .repMode        = letimerRepeatFree // Count until stopped.
};

/******************************************************************************
 * @brief LETIMER0 Interrupt Handler. Clears interrupt flag.
 *        The interrupt table is in assembly startup file startup_efm32.s
 *
 *****************************************************************************/
void LETIMER0_IRQHandler(void)
{
  // Clear LETIMER0 underflow interrupt flag
  LETIMER_IntClear(LETIMER0, LETIMER_IF_UF);

  // If the value of comp1 is over 0, decrement it 
  // or bring comp1 back to COMPMAX (500)
  if (comp1 != 0)
  {
    comp1--;
  }
  else
  {
    comp1 = COMPMAX;
  }
  
  // Write the new compare value to COMP1
  LETIMER_CompareSet(LETIMER0, 1, comp1);
}

#if defined( RTCC_COUNT ) && ( RTCC_COUNT == 1 )
/******************************************************************************
 * @brief  RTCC initialization
 *         Configures and starts the RTCC
 *****************************************************************************/
void setupRtcc(void)
{
  CMU_OscillatorEnable(cmuOsc_LFXO, true, true);
  
  // Enable necessary clocks
  CMU_ClockEnable(cmuClock_CORELE, true);
  CMU_ClockSelectSet(cmuClock_LFE, cmuSelect_LFXO);
  CMU_ClockEnable(cmuClock_RTCC, true);
  
  // Each RTCC tick is 1 second
  RTCC_Init_TypeDef rtccInit = RTCC_INIT_DEFAULT;
  RTCC_CCChConf_TypeDef rtccInitCompareChannel = RTCC_CH_INIT_COMPARE_DEFAULT;
  
  rtccInit.cntWrapOnCCV1 = true;        // Clear counter on compare match
  rtccInit.enable = false;              // Don't Start counting
                                        // when init completed.
  rtccInit.presc = rtccCntPresc_32768;  // 1 second per tick
  
  // Setting up RTCC with compare value.
  // Compare match will occur at 5 seconds
  RTCC_ChannelInit(1, &rtccInitCompareChannel);
  RTCC_ChannelCCVSet(1, RTCC_COMP);
  
  // RTCC initiallization
  RTCC_Init(&rtccInit);
  
  // Enable RTCC and clear counter
  RTCC_Enable(true);
  RTCC_CounterSet(0);
}

#else
/******************************************************************************
 * @brief  RTC initialization.
 *         Configures and starts the RTC
 *****************************************************************************/
void setupRtc(void)
{
  // Enable necessary clocks
  CMU_ClockEnable(cmuClock_RTC, true);
  
  // Each RTC tick is 1 second
  CMU_ClockDivSet(cmuClock_RTC, cmuClkDiv_32768);
  
  // Setting up RTC with compare value.
  // Compare match will occur at 5 seconds
  RTC_CompareSet(0, RTC_COMP);
  
  // Enable RTC
  RTC_Enable(true);
}
#endif

/******************************************************************************
 * @brief  letimer route location initialization.
 *****************************************************************************/
void letimerRoute(void)
{
#if defined( _LETIMER_ROUTEPEN_OUT0PEN_MASK )
  // Route LETIMER to location 0 and enable outputs
  LETIMER0->ROUTEPEN = LETIMER_ROUTEPEN_OUT0PEN | LETIMER_ROUTEPEN_OUT1PEN;
  LETIMER0->ROUTELOC0 =  LETIMER_ROUTELOC0_OUT0LOC_LOC0 |
	                     LETIMER_ROUTELOC0_OUT1LOC_LOC0;
#else
  // Route LETIMER to location 0 and enable outputs
  LETIMER0->ROUTE = LETIMER_ROUTE_OUT0PEN | LETIMER_ROUTE_OUT1PEN | 
                    LETIMER_ROUTE_LOCATION_LOC0;
#endif
}

/******************************************************************************
 * @brief  LETIMER0 initialization.
 *         Clock source is LFXO, prescale = 0, 1KHz/1 = 32.768kHz
 *****************************************************************************/
void setupLetimer(testMode_TypeDef tMode)
{
  // Enable necessary clocks
  CMU_ClockSelectSet(cmuClock_LFA, cmuSelect_LFXO);
  CMU_ClockEnable(cmuClock_CORELE, true);
  CMU_ClockEnable(cmuClock_LETIMER0, true);  
  CMU_ClockEnable(cmuClock_GPIO, true);
  
  // Configure port and pin as push pull so the
  // LETIMER can override them
  GPIO_PinModeSet(LETIMER_OUT0_PORT, LETIMER_OUT0_PIN, gpioModePushPull, 0);
  GPIO_PinModeSet(LETIMER_OUT1_PORT, LETIMER_OUT1_PIN, gpioModePushPull, 0);
  
  if ((tMode == T_PWM) 
#if defined ( _LETIMER_PRSSEL_MASK )
  || (tMode == T_GPIO)
#endif
  )
  {
    // Set initial compare values for COMP0 and COMP1
    // COMP1 keeps it's value and is used as TOP value
    // for the LETIMER.
    // COMP1 gets decremented through the program execution
    // to generate a different PWM duty cycle
    LETIMER_CompareSet(LETIMER0, 0, COMPMAX);
    LETIMER_CompareSet(LETIMER0, 1, COMPMAX);

    // Repetition values must be nonzero so that the outputs
    // return switch between idle and active state
    LETIMER_RepeatSet(LETIMER0, 0, 0x01);
    LETIMER_RepeatSet(LETIMER0, 1, 0x01);
  }
#if defined( RTCC_COUNT ) && ( RTCC_COUNT == 1 )
  else if (tMode == T_RTCC)
#else
  else if (tMode == T_RTC)
#endif
  {
    // Set initial compare values for COMP0
    LETIMER_CompareSet(LETIMER0, 0, LETIMER_TOP);
    // Set repetition values, LETIMER will count 5 times and
    // thus, generate 5 pulses
    LETIMER_RepeatSet(LETIMER0, 0, LETIMER_REPEAT);
  }
  
  // Route output to GPIO
  letimerRoute();
  
  if ((tMode == T_PWM)
#if defined ( _LETIMER_PRSSEL_MASK )
  || (tMode == T_GPIO)
#endif
  )
  {
     letimerInit.enable = true;                // Start counting
                                               //  when init completed.
  }
#if defined( RTCC_COUNT ) && ( RTCC_COUNT == 1 )
  else if (tMode == T_RTCC)
#else
  else if (tMode == T_RTC)
#endif
  {
    letimerInit.enable         = false;        // Don't start counting
                                               // when init completed.
#if defined ( LETIMER_CTRL_RTCC0TEN )
    letimerInit.rtcComp0Enable = true;         // Don't start counting on
                                               // RTC COMP0 match.
    letimerInit.rtcComp1Enable = false;        // Don't start counting on
                                               // RTC COMP1 match.
#endif
    letimerInit.ufoa0  = letimerUFOAPulse;     // Pulse output on output 0
    letimerInit.ufoa1  = letimerUFOANone;      // No output on output 1
    letimerInit.repMode= letimerRepeatOneshot; // Count while REP != 0
  }
  
  // Initialize LETIMER
  LETIMER_Init(LETIMER0, &letimerInit);
}

#if defined( RTCC_COUNT ) && ( RTCC_COUNT == 1 )
/******************************************************************************
 * @brief  RTCC PRS initialization.
 *****************************************************************************/
void prsRtccSetup(void)
{
  // Enable PRS in CMU
  CMU_ClockEnable(cmuClock_PRS, true);

  // Enable PRS source and target
  PRS_SourceAsyncSignalSet(RTCC_PRS_CHANNEL, PRS_CH_CTRL_SOURCESEL_RTCC,
                           PRS_CH_CTRL_SIGSEL_RTCCCCV1);
  LETIMER0->PRSSEL = LETIMER_PRSSEL_PRSSTARTSEL_PRSCH0 | 
                     LETIMER_PRSSEL_PRSSTARTMODE_BOTH;
}

/******************************************************************************
 * @brief  GPIO PRS initialization.
 *****************************************************************************/
#endif

#if defined ( _LETIMER_PRSSEL_MASK )
void prsGpioSetup(void)
{
  // Enable GPIO in CMU
  CMU_ClockEnable(cmuClock_GPIO, true);
  
  // push button configuration to trigger PRS
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeInput, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN,
                BSP_GPIO_PB0_PIN, false, false, false);
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInput, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN,
                BSP_GPIO_PB1_PIN, false, false, false);

  // Enable PRS in CMU
  CMU_ClockEnable(cmuClock_PRS, true);

  // Enable PRS source and target
  PRS_SourceAsyncSignalSet(GPIO_START_PRS_CHANNEL,
                           START_PRS_CH_CRTL_SOURCE_GPIO, BSP_GPIO_PB0_PIN);
  PRS_SourceAsyncSignalSet(GPIO_STOP_PRS_CHANNEL,
                           STOP_PRS_CH_CRTL_SOURCE_GPIO, BSP_GPIO_PB1_PIN);

  LETIMER0->PRSSEL = LETIMER_PRSSEL_PRSSTARTSEL_PRSCH0 | 
                     LETIMER_PRSSEL_PRSSTOPSEL_PRSCH1  | 
                     LETIMER_PRSSEL_PRSSTARTMODE_FALLING | 
                     LETIMER_PRSSEL_PRSSTOPMODE_FALLING;
}
#endif
