/******************************************************************************
 * @file letimer_mode_test.c
 * @brief LETIMER Demo Application
 * @author Silicon Labs
 * @version  1.09

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 ******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 *****************************************************************************/
#include "config.h"
#include "em_letimer.h"

/* Global Variables */

/******************************************************************************
 * @brief  Run letimer different mode test.
 *****************************************************************************/
void modeTest(testMode_TypeDef tMode)
{
  switch (tMode)
  {
    case T_PWM:
      testPWM();
      break;
#if defined( RTCC_COUNT ) && ( RTCC_COUNT == 1 )
    // For series 1 device
    case T_RTCC:
      testRTCC();
      break;
#else
    // For series 0 device
    case T_RTC:
      testRTC();
      break;
#endif
#if defined ( _LETIMER_PRSSEL_MASK )
    case T_GPIO:
      testGPIO();
      break;
#endif
    default:
      break;
  }
}

/******************************************************************************
 * @brief  letimer software start test.

*          The test uses software to start the letimer,
*          it routes PWM and pulse output to 2 pins.
*          You could see continuous PWM (changing duty cycle)
*          and pulse on the 2 pins
 *****************************************************************************/
void testPWM(void)
{
  // Initialize LETIMER
  setupLetimer(tMode);

  //THE INTERRUPT IS SIMPLY TO DECREASE THE VALUE OF COMP1
  //TO VARY THE PWM DUTY CYCLE
  // Enable underflow interrupt
  LETIMER_IntEnable(LETIMER0, LETIMER_IF_UF);
  
  // Enable LETIMER0 interrupt vector in NVIC
  NVIC_EnableIRQ(LETIMER0_IRQn);
}

#if defined( RTCC_COUNT ) && ( RTCC_COUNT == 1 )
/******************************************************************************
 * @brief  letimer RTCC start test.

*          The test uses RTCC to start the letimer through PRS,
*          it routes pulse output to pin.
*          You could see one-shot pulse on the pin
 *****************************************************************************/
void testRTCC(void)
{
  // Initialize LETIMER
  setupLetimer(tMode);

  // Initialize RTCC PRS
  prsRtccSetup();

  // Initialize RTCC
  setupRtcc();
}

#else
/******************************************************************************
 * @brief  letimer RTC start test.

*          The test uses RTC to start the letimer,
*          it routes pulse output to pin.
*          You could see one-shot pulse on the pin
 *****************************************************************************/
void testRTC(void)
{
  // Initialize LETIMER
  setupLetimer(tMode);
  
  // Initialize RTC
  setupRtc();
}
#endif

#if defined ( _LETIMER_PRSSEL_MASK )
/******************************************************************************
 * @brief  letimer GPIO start/stop test.

 *         The test uses GPIOs to start/stop the letimer through PRS,
 *         it routes pulse output to pin.
 *         You could see continuous pulses on the pin
 *         when the start button was pressed.
 *         You could see the pulse disappear on the pin
 *         when the stop button was pressed.
 *****************************************************************************/
void testGPIO(void)
{
  // Initialize LETIMER
  setupLetimer(tMode);

  // Initialize GPIO and PRS
  prsGpioSetup();
}
#endif
