/***************************************************************************//**
 * @file main_opamp.c
 * @brief Examples for EFM32 and EFR32 Series 1 OPAMP.
 * @version  1.00

 *******************************************************************************
 * # License
 * <b>Copyright 2019 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/

#include "em_chip.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "em_device.h"
#include "opamp.h"
#include "opamp_acmp.h"
#include "opamp_adc.h"
#include "opamp_config.h"
#include "retargetserial.h"
#include <stdio.h>

/***************************************************************************//**
 * @brief  Main function
 ******************************************************************************/
int main(void)
{
  bool loop = true;
  int8_t inputChar;

  // Chip errata
  CHIP_Init();

#if defined( _EMU_DCDCCTRL_MASK )  
  // Initialize DCDC regulator if available
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif
  
  // Set HFRCO to 38 MHz
  CMU_HFRCOBandSet(cmuHFRCOFreq_38M0Hz);
  
  // Initialize and retarget USART
  RETARGET_SerialInit();
  RETARGET_SerialCrLf(1);

  // Print menu
  printf("\nOPAMP Examples\n");
  printf("Press a for General OPA mode\n"); 
  printf("Press b for Unity gain voltage follower\n"); 
  printf("Press c for Non-inverting amplifier\n"); 
  printf("Press d for Inverting amplifier\n"); 
  printf("Press e for Cascaded non-inverting amplifier\n"); 
  printf("Press f for Cascaded inverting amplifier\n"); 
  printf("Press g for Two OPA differential amplifier\n"); 
  printf("Press h for OPA as ACMP input front-end\n"); 
  printf("Press i for OPA as ADC input front-end\n");
#if defined(VDAC_STATUS_OPA2ENS)
  printf("Press j for Three OPA differential amplifier\n"); 
#endif  
  
  while (1) {
    // Wait key press to run example
    inputChar = getchar();

    if (inputChar > 0) {
      switch (inputChar) {
        case 'a':
          opaGeneralMode(GM_OPA, GM_POS, GM_NEG, GM_OUT, GM_ALTMASK);
          printf("\nGeneral OPA mode at EM3\n");
          printf("Reset the STK/WSTK to select another example\n");
          RETARGET_SerialFlush();
          EMU_EnterEM3(false);
          break;
        
        case 'b':
          opaUnityGain(UG_OPA, UG_POS, UG_OUT, UG_ALTMASK);
          printf("\nUnity gain voltage follower at EM3\n"); 
          printf("Reset the STK/WSTK to select another example\n");
          RETARGET_SerialFlush();
          EMU_EnterEM3(false);
          break;
        
        case 'c':
          opaNonInvertAmp(NIA_OPA, NIA_POS, NIA_RESSEL, NIA_OUT, NIA_ALTMASK);
          printf("\nNon-inverting amplifier at EM3\n"); 
          printf("Reset the STK/WSTK to select another example\n");
          RETARGET_SerialFlush();
          EMU_EnterEM3(false);
          break;
        
        case 'd':
          opaInvertAmp(IA_OPA, IA_RESIN, IA_POS, IA_RESSEL, IA_OUT, IA_ALTMASK);
          printf("\nInverting amplifier at EM3\n"); 
          printf("Reset the STK/WSTK to select another example\n");
          RETARGET_SerialFlush();
          EMU_EnterEM3(false);
          break;
        
        case 'e':
          opaCascadeTwoNonInvertAmp(CNIA_FIRST, CNIA_POS, CNIA_RESSEL0,
                                    CNIA_RESSEL1, CNIA_OUT, CNIA_ALTMASK);
          printf("\nCascaded two non-inverting amplifier at EM3\n"); 
          printf("Reset the STK/WSTK to select another example\n");
          RETARGET_SerialFlush();
          EMU_EnterEM3(false);
          break;
        
        case 'f':
          opaCascadeTwoInvertAmp(CIA_FIRST, CIA_RESIN, CIA_POS0, CIA_RESSEL0,
                                 CIA_POS1, CIA_RESSEL1, CIA_OUT, CIA_ALTMASK);
          printf("\nCascaded two inverting amplifier at EM3\n"); 
          printf("Reset the STK/WSTK to select another example\n");
          RETARGET_SerialFlush();
          EMU_EnterEM3(false);
          break;
        
        case 'g':
          opaTwoDiffAmp(DIFF2_OPA, DIFF2_POS0, DIFF2_POS1, DIFF2_RESSEL,
                        DIFF2_OUT, DIFF2_ALTMASK);
          printf("\nTwo OPA differential amplifier at EM3\n"); 
          printf("Reset the STK/WSTK to select another example\n");
          RETARGET_SerialFlush();
          EMU_EnterEM3(false);
          break;

        case 'h':
          opaNonInvertAmp(ACMP_POS_OPA, ACMP_POS_POS, ACMP_POS_RESSEL,
                          ACMP_POS_OUT, 0);
          opaNonInvertAmp(ACMP_NEG_OPA, ACMP_NEG_POS, ACMP_NEG_RESSEL,
                          ACMP_NEG_OUT, 0);
          initAcmp();
          printf("\nOPA as ACMP input front-end at EM3\n");
          while (loop) {
            printf("Reset the STK/WSTK to select another example\n");
            RETARGET_SerialFlush();
            EMU_EnterEM3(false);
            printf("\nDevice wakes up by ACMP falling edge interrupt\n");
          }
          break;

        case 'i':
          opaNonInvertAmp(ADC_OPA, ADC_POS, ADC_RESSEL, ADC_OUT, 0);
#if (USE_OPA_PRS == 1)
          initOpaPrs(ADC_OPA, ADC_PRS_MODE, ADC_PRS_IN_CH, ADC_PRS_OUT,
                     ADC_PRS_OUT_CH);
#endif
          initAdc();
          initLdma();
          initLetimer();
          printf("\nOPA as ADC input front-end at EM2\n");
          while (loop) {
            printf("Reset the STK/WSTK to select another example\n");
            RETARGET_SerialFlush();
            EMU_EnterEM2(false);
            printVoltage();
          }
          break;
     
#if defined(VDAC_STATUS_OPA2ENS)
        case 'j':
          opaThreeDiffAmp(DIFF3_POS0, DIFF3_POS1, DIFF3_RESSEL0, DIFF3_RESSEL1,
                          DIFF3_OUT, DIFF3_ALTMASK);
          printf("\nThree OPA differential amplifier at EM3\n"); 
          printf("Reset the STK/WSTK to select another example\n");
          RETARGET_SerialFlush();
          EMU_EnterEM3(false);
          break;
#endif

        default:
          break;
      }
    }
  }
}
