/***************************************************************************//**
 * @file opamp.c
 * @brief Functions for different OPAMP configurations.
 * @version  1.00

 *******************************************************************************
 * # License
 * <b>Copyright 2019 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/

#include "em_cmu.h"
#include "em_opamp.h"
#include "em_prs.h"
#include "opamp.h"

/***************************************************************************//**
 * @brief
 *   Disable OVT on OPA positive input.
 *
 * @param[in] opaSelect
 *   Select an OPA, valid values are OPA0, OPA1, OPA2, and OPA3.
 *
 * @param[in] posInput
 *   OPA positive input source.
 ******************************************************************************/
void posSelOvtDisable(OPAMP_TypeDef opaSelect, OPAMP_PosSel_TypeDef posInput)
{
  uint32_t posValue;
  GPIO_Port_TypeDef port = gpioPortA;
  uint16_t pin = 0;
  
  // Enable GPIO clock
  CMU_ClockEnable(cmuClock_GPIO, true);
  
  posValue = (uint32_t)posInput >> _VDAC_OPA_MUX_POSSEL_SHIFT;

  // Disable OVT if positive input is POSPAD or APORT
  if (posValue == _VDAC_OPA_MUX_POSSEL_POSPAD) {
    switch (opaSelect) {
    case OPA0:
      port = POS0PORT;
      pin = POS0PIN;
      break;
      
    case OPA1:
      port = POS1PORT;
      pin = POS1PIN;
      break;

#if defined(VDAC_STATUS_OPA2ENS)
      // Only valid for device with OPA2
    case OPA2:
      port = POS2PORT;
      pin = POS2PIN;
      break;
#endif
      
#if defined(VDAC_STATUS_OPA3ENS)
      // Only valid for device with OPA3
    case OPA3:
      port = POS3PORT;
      pin = POS3PIN;
      break;
#endif
      
    default:
      EFM_ASSERT(false);
      break;
    }
  }
  else if (posValue < 40) {
    port = APORT1XPORTLOW;
    pin = (posValue - 32) << 1;
  }
  else if (posValue < 48) {
    port = APORT1XPORTHIGH;
    pin = (posValue - 40) << 1;
  }
  else if (posValue < 72) {
    port = APORT2XPORTLOW;
    pin = ((posValue - 64) << 1) + 1;
  }
  else if (posValue < 80) {
    port = APORT2XPORTHIGH;
    pin = ((posValue - 72) << 1) + 1;
  }
#if defined(VDAC_STATUS_OPA3ENS)
  else if (posValue < 104) {
    port = APORT3XPORTLOW;
    pin = (posValue - 96) << 1;
  }
  else if (posValue < 112) {
    port = APORT3XPORTHIGH;
    pin = (posValue - 104) << 1;
  }
  else if (posValue < 136) {
    port = APORT4XPORTLOW;
    pin = ((posValue - 128) << 1) + 1;
  }
#else
  else if (posValue < 100) {
    port = APORT3XPORTLOW0;
    pin = ((posValue - 96) << 1) + 8;
  }
  else if (posValue < 104) {
    port = APORT3XPORTLOW1;
    pin = (posValue - 100) << 1;
  }
  else if (posValue < 112) {
    port = APORT3XPORTHIGH;
    pin = (posValue - 104) << 1;
  }
  else if (posValue < 132) {
    port = APORT4XPORTLOW0;
    pin = ((posValue - 128) << 1) + 9;
  }
  else if (posValue < 136) {
    port = APORT4XPORTLOW1;
    pin = ((posValue - 132) << 1) + 1;
  }
#endif  
  else if (posValue < 144) {
    port = APORT4XPORTHIGH;
    pin = ((posValue - 136) << 1) + 1;
  }
  else {
    return;
  }
  // Disable over-voltage tolerance on selected pin         
  GPIO->P[port].OVTDIS |= (1 << pin);   
}           

/***************************************************************************//**
 * @brief
 *   Disable OVT on OPA negative input.
 *
 * @param[in] opaSelect
 *   Select an OPA, valid values are OPA0, OPA1, OPA2, and OPA3.
 *
 * @param[in] negInput
 *   OPA negative input source.
 ******************************************************************************/
void negSelOvtDisable(OPAMP_TypeDef opaSelect, OPAMP_NegSel_TypeDef negInput)
{
  uint32_t negValue;
  GPIO_Port_TypeDef port = gpioPortA;
  uint16_t pin = 0;
  
  // Enable GPIO clock
  CMU_ClockEnable(cmuClock_GPIO, true);
  
  negValue = (uint32_t)negInput >> _VDAC_OPA_MUX_NEGSEL_SHIFT;

  // Disable OVT if negative input is NEGPAD or APORT
  if (negValue == _VDAC_OPA_MUX_NEGSEL_NEGPAD) {
    switch (opaSelect) {
      case OPA0:
        port = NEG0PORT;
        pin = NEG0PIN;
        break;
      
      case OPA1:
        port = NEG1PORT;
        pin = NEG1PIN;
        break;

#if defined(VDAC_STATUS_OPA2ENS)
      // Only valid for device with OPA2
      case OPA2:
        port = NEG2PORT;
        pin = NEG2PIN;
        break;
#endif
      
#if defined(VDAC_STATUS_OPA3ENS)
      // Only valid for device with OPA3
      case OPA3:
        port = NEG3PORT;
        pin = NEG3PIN;
        break;
#endif
      
      default:
        EFM_ASSERT(false);
        break;
    }
  }
  else if (negValue < 56) {
    port = APORT1YPORTLOW;
    pin = ((negValue - 48) << 1) + 1;
  }
  else if (negValue < 64) {
    port = APORT1YPORTHIGH;
    pin = ((negValue - 56) << 1) + 1;
  }
  else if (negValue < 88) {
    port = APORT2YPORTLOW;
    pin = (negValue - 80) << 1;
  }
  else if (negValue < 96) {
    port = APORT2YPORTHIGH;
    pin = (negValue - 88) << 1;
  }
#if defined(VDAC_STATUS_OPA3ENS)
  else if (negValue < 120) {
    port = APORT3YPORTLOW;
    pin = ((negValue - 112) << 1) + 1;
  }
  else if (negValue < 128) {
    port = APORT3YPORTHIGH;
    pin = ((negValue - 120) << 1) + 1;
  }
  else if (negValue < 152) {
    port = APORT4YPORTLOW;
    pin = (negValue - 144) << 1;
  }
#else
  else if (negValue < 116) {
    port = APORT3YPORTLOW0;
    pin = ((negValue - 112) << 1) + 9;
  }
  else if (negValue < 120) {
    port = APORT3YPORTLOW1;
    pin = ((negValue - 116) << 1) + 1;
  }
  else if (negValue < 128) {
    port = APORT3YPORTHIGH;
    pin = ((negValue - 120) << 1) + 1;
  }
  else if (negValue < 148) {
    port = APORT4YPORTLOW0;
    pin = ((negValue - 144) << 1) + 8;
  }
  else if (negValue < 152) {
    port = APORT4YPORTLOW1;
    pin = (negValue - 148) << 1;
  }
#endif  
  else if (negValue < 160) {
    port = APORT4YPORTHIGH;
    pin = (negValue - 152) << 1;
  }
  else {
    return;
  }
  // Disable over-voltage tolerance on selected pin         
  GPIO->P[port].OVTDIS |= (1 << pin);   
}           

/***************************************************************************//**
 * @brief
 *   Disable OVT on OPA resistor ladder input.
 *
 * @param[in] opaSelect
 *   Select an OPA, valid values are OPA0, OPA1, OPA2, and OPA3.
 *
 * @param[in] resInInput
 *   Resistor ladder input source.
 ******************************************************************************/
void resInOvtDisable(OPAMP_TypeDef opaSelect, OPAMP_ResInMux_TypeDef resInInput)
{
  uint32_t resInValue;
  GPIO_Port_TypeDef port = gpioPortA;
  uint16_t pin = 0;
  
  // Enable GPIO clock
  CMU_ClockEnable(cmuClock_GPIO, true);
  
  resInValue = (uint32_t)resInInput >> _VDAC_OPA_MUX_RESINMUX_SHIFT;

  // Disable OVT if resistor ladder input is NEGPAD, POSPAD, or COMPAD 
  if (resInValue == _VDAC_OPA_MUX_RESINMUX_NEGPAD) {
    switch (opaSelect) {
      case OPA0:
        port = NEG0PORT;
        pin = NEG0PIN;
        break;
      
      case OPA1:
        port = NEG1PORT;
        pin = NEG1PIN;
        break;

#if defined(VDAC_STATUS_OPA2ENS)
      // Only valid for device with OPA2
      case OPA2:
        port = NEG2PORT;
        pin = NEG2PIN;
        break;
#endif
      
#if defined(VDAC_STATUS_OPA3ENS)
      // Only valid for device with OPA3
      case OPA3:
        port = NEG3PORT;
        pin = NEG3PIN;
        break;
#endif
      
      default:
        EFM_ASSERT(false);
        break;
    }
  }
  else if (resInValue == _VDAC_OPA_MUX_RESINMUX_POSPAD) {
    switch (opaSelect) {
    case OPA0:
      port = POS0PORT;
      pin = POS0PIN;
      break;
      
    case OPA1:
      port = POS1PORT;
      pin = POS1PIN;
      break;

#if defined(VDAC_STATUS_OPA2ENS)
      // Only valid for device with OPA2
    case OPA2:
      port = POS2PORT;
      pin = POS2PIN;
      break;
#endif
      
#if defined(VDAC_STATUS_OPA3ENS)
      // Only valid for device with OPA3
    case OPA3:
      port = POS3PORT;
      pin = POS3PIN;
      break;
#endif
      
    default:
      EFM_ASSERT(false);
      break;
    }
  }
  else if (resInValue == _VDAC_OPA_MUX_RESINMUX_COMPAD) {
    port = NEG0PORT;
    pin = NEG0PIN;
  }
  else {
    return;
  }
  // Disable over-voltage tolerance on selected pin         
  GPIO->P[port].OVTDIS |= (1 << pin);   
}

/***************************************************************************//**
 * @brief
 *   Configure and enable an OPA in general mode.
 *
 * @param[in] opaSelect
 *   Select an OPA, valid values are OPA0, OPA1, OPA2, and OPA3.
 *
 * @param[in] posInput
 *   OPA positive input source.
 *
 * @param[in] negInput
 *   OPA negative input source.
 *
 * @param[in] opaOutput
 *   OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaGeneralMode(OPAMP_TypeDef opaSelect,
                    OPAMP_PosSel_TypeDef posInput,
                    OPAMP_NegSel_TypeDef negInput,
                    OPAMP_OutMode_TypeDef opaOutput,
                    uint32_t altOutMask)
{
  OPAMP_Init_TypeDef initOpa = OPA_INIT_UNITY_GAIN;

  // Disable input OVT
  posSelOvtDisable(opaSelect, posInput);
  negSelOvtDisable(opaSelect, negInput);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure OPA
  initOpa.posSel = posInput;       
  initOpa.negSel = negInput;
  initOpa.outMode = opaOutput;
  initOpa.outPen = altOutMask; 
       
  // Enable OPA
  OPAMP_Enable(VDAC0, opaSelect, &initOpa);
}              

/***************************************************************************//**
 * @brief
 *   Configure and enable an OPA in unity gain voltage follower mode.
 *
 * @param[in] opaSelect
 *   Select an OPA, valid values are OPA0, OPA1, OPA2, and OPA3.
 *
 * @param[in] posInput
 *   OPA positive input source.
 *
 * @param[in] opaOutput
 *   OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaUnityGain(OPAMP_TypeDef opaSelect,
                  OPAMP_PosSel_TypeDef posInput,
                  OPAMP_OutMode_TypeDef opaOutput,
                  uint32_t altOutMask)
{
  OPAMP_Init_TypeDef initOpa = OPA_INIT_UNITY_GAIN;

  // Disable input OVT
  posSelOvtDisable(opaSelect, posInput);
  
  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure OPA
  initOpa.posSel = posInput;       
  initOpa.outMode = opaOutput;
  initOpa.outPen = altOutMask; 
  
  // Enable OPA
  OPAMP_Enable(VDAC0, opaSelect, &initOpa);
}              

/***************************************************************************//**
 * @brief
 *   Configure and enable an OPA in non-inverting amplifier mode.
 *
 * @param[in] opaSelect
 *   Select an OPA, valid values are OPA0, OPA1, OPA2, and OPA3.
 *
 * @param[in] posInput
 *   OPA positive input source.
 *
 * @param[in] resLadderRatio
 *   Internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaNonInvertAmp(OPAMP_TypeDef opaSelect,
                     OPAMP_PosSel_TypeDef posInput,
                     OPAMP_ResSel_TypeDef resLadderRatio,
                     OPAMP_OutMode_TypeDef opaOutput,
                     uint32_t altOutMask)
{
  OPAMP_Init_TypeDef initOpa = OPA_INIT_NON_INVERTING;

  // Disable input OVT
  posSelOvtDisable(opaSelect, posInput);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure OPA
  initOpa.posSel = posInput;       
  initOpa.resSel = resLadderRatio;
  initOpa.resInMux = opaResInMuxVss;
  initOpa.outMode = opaOutput;     
  initOpa.outPen = altOutMask; 

  // Enable OPA
  OPAMP_Enable(VDAC0, opaSelect, &initOpa);
}              

/***************************************************************************//**
 * @brief
 *   Configure and enable an OPA in inverting amplifier mode.
 *
 * @param[in] opaSelect
 *   Select an OPA, valid values are OPA0, OPA1, OPA2, and OPA3.
 *
 * @param[in] resInInput
 *   Resistor ladder input source.
 *
 * @param[in] posInput
 *   OPA positive input source.
 *
 * @param[in] resLadderRatio
 *   Internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaInvertAmp(OPAMP_TypeDef opaSelect,
                  OPAMP_ResInMux_TypeDef resInInput,
                  OPAMP_PosSel_TypeDef posInput,
                  OPAMP_ResSel_TypeDef resLadderRatio,
                  OPAMP_OutMode_TypeDef opaOutput,
                  uint32_t altOutMask)
{
  OPAMP_Init_TypeDef initOpa = OPA_INIT_INVERTING;

  // Disable inputs OVT
  resInOvtDisable(opaSelect, resInInput);
  posSelOvtDisable(opaSelect, posInput);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure OPA
  initOpa.resInMux = resInInput;
  initOpa.posSel = posInput;       
  initOpa.resSel = resLadderRatio;
  initOpa.outMode = opaOutput;     
  initOpa.outPen = altOutMask; 

  // Enable OPA
  OPAMP_Enable(VDAC0, opaSelect, &initOpa);
}              

/***************************************************************************//**
 * @brief
 *   Configure and enable two OPAs in cascaded non-inverting amplifier mode.
 *
 * @param[in] firstStage
 *   Select an OPA for first stage, valid values are OPA0, OPA1, and OPA2.
 *
 * @param[in] posInput0
 *   First stage OPA positive input source.
 *
 * @param[in] resSel0
 *   First stage internal resistor ladder ratio.
 *
 * @param[in] resSel1
 *   Second stage internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   Second stage OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaCascadeTwoNonInvertAmp(OPAMP_TypeDef firstStage,
                               OPAMP_PosSel_TypeDef posInput0,
                               OPAMP_ResSel_TypeDef resSel0,
                               OPAMP_ResSel_TypeDef resSel1,
                               OPAMP_OutMode_TypeDef opaOutput,             
                               uint32_t altOutMask)
{
  OPAMP_Init_TypeDef initFirst = OPA_INIT_CASCADED_NON_INVERTING_OPA0;
  OPAMP_Init_TypeDef initSecond = OPA_INIT_CASCADED_NON_INVERTING_OPA1;

  // Disable input OVT
  posSelOvtDisable(firstStage, posInput0);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure first stage OPA
  initFirst.posSel = posInput0;
  initFirst.resSel = resSel0;
  initFirst.resInMux = opaResInMuxVss;
  initFirst.outMode = opaOutModeDisable;

  // Configure second stage OPA
  initSecond.resSel = resSel1;
  initSecond.resInMux = opaResInMuxVss;
  initSecond.outMode = opaOutput;
  initSecond.outPen = altOutMask; 
  
  // Check which OPA is first stage  
  switch (firstStage) {
    case OPA0:
      OPAMP_Enable(VDAC0, OPA0, &initFirst);
      OPAMP_Enable(VDAC0, OPA1, &initSecond);
      break;

#if defined(VDAC_STATUS_OPA2ENS)
    // Only valid for device with OPA2
    case OPA1:
      OPAMP_Enable(VDAC0, OPA1, &initFirst);
      OPAMP_Enable(VDAC0, OPA2, &initSecond);
      break;
#endif
      
#if defined(VDAC_STATUS_OPA3ENS)
    // Only valid for device with OPA3
    case OPA2:
      OPAMP_Enable(VDAC0, OPA2, &initFirst);
      OPAMP_Enable(VDAC0, OPA3, &initSecond);
      break;
#endif
      
    default:
      EFM_ASSERT(false);
      break;
  }
}              

#if defined(VDAC_STATUS_OPA2ENS)
/***************************************************************************//**
 * @brief
 *   Configure and enable three OPAs in cascaded non-inverting amplifier mode.
 *
 * @param[in] firstStage
 *   Select an OPA for first stage, valid values are OPA0, and OPA1.
 *
 * @param[in] posInput0
 *   First stage OPA positive input source.
 *
 * @param[in] resSel0
 *   First stage internal resistor ladder ratio.
 *
 * @param[in] resSel1
 *   Second stage internal resistor ladder ratio.
 *
 * @param[in] resSel2
 *   Third stage internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   Third stage OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaCascadeThreeNonInvertAmp(OPAMP_TypeDef firstStage,
                                 OPAMP_PosSel_TypeDef posInput0,
                                 OPAMP_ResSel_TypeDef resSel0,
                                 OPAMP_ResSel_TypeDef resSel1,
                                 OPAMP_ResSel_TypeDef resSel2,
                                 OPAMP_OutMode_TypeDef opaOutput,             
                                 uint32_t altOutMask)
{
  OPAMP_Init_TypeDef initTemp = OPA_INIT_CASCADED_NON_INVERTING_OPA0;

  // Disable input OVT
  posSelOvtDisable(firstStage, posInput0);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure first stage OPA
  initTemp.posSel = posInput0;
  initTemp.resSel = resSel0;
  initTemp.resInMux = opaResInMuxVss;
  initTemp.outMode = opaOutModeDisable;

  // Check which OPA is first stage  
  switch (firstStage) {
    case OPA0:
      OPAMP_Enable(VDAC0, OPA0, &initTemp);

      // Configure second stage OPA
      // Following stage positive input from previous output
      initTemp.posSel = opaPosSelOpaIn;
      initTemp.resSel = resSel1;
      OPAMP_Enable(VDAC0, OPA1, &initTemp);

      // Configure third stage OPA
      initTemp.resSel = resSel2;
      initTemp.outMode = opaOutput;
      initTemp.outPen = altOutMask; 
      OPAMP_Enable(VDAC0, OPA2, &initTemp);
      break;

#if defined(VDAC_STATUS_OPA3ENS)
    // Only valid for device with OPA3
    case OPA1:
      OPAMP_Enable(VDAC0, OPA1, &initTemp);

      // Configure second stage OPA
      // Following stage positive input from previous output
      initTemp.posSel = opaPosSelOpaIn;
      initTemp.resSel = resSel1;
      OPAMP_Enable(VDAC0, OPA2, &initTemp);

      // Configure third stage OPA
      initTemp.resSel = resSel2;
      initTemp.outMode = opaOutput;
      initTemp.outPen = altOutMask; 
      OPAMP_Enable(VDAC0, OPA3, &initTemp);
      break;
#endif
      
    default:
      EFM_ASSERT(false);
      break;
  }
}              
#endif

#if defined(VDAC_STATUS_OPA3ENS)
/***************************************************************************//**
 * @brief
 *   Configure and enable four OPAs in cascaded non-inverting amplifier mode.
 *
 * @param[in] posInput0
 *   First stage OPA positive input source.
 *
 * @param[in] resSel0
 *   First stage internal resistor ladder ratio.
 *
 * @param[in] resSel1
 *   Second stage internal resistor ladder ratio.
 *
 * @param[in] resSel2
 *   Third stage internal resistor ladder ratio.
 *
 * @param[in] resSel3
 *   Fourth stage internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   Fourth stage OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaCascadeFourNonInvertAmp(OPAMP_PosSel_TypeDef posInput0,
                                OPAMP_ResSel_TypeDef resSel0,
                                OPAMP_ResSel_TypeDef resSel1,
                                OPAMP_ResSel_TypeDef resSel2,
                                OPAMP_ResSel_TypeDef resSel3,
                                OPAMP_OutMode_TypeDef opaOutput,             
                                uint32_t altOutMask)
{
  OPAMP_Init_TypeDef initTemp = OPA_INIT_CASCADED_NON_INVERTING_OPA0;

  // Disable input OVT
  posSelOvtDisable(OPA0, posInput0);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure first stage OPA
  initTemp.posSel = posInput0;
  initTemp.resSel = resSel0;
  initTemp.resInMux = opaResInMuxVss;
  initTemp.outMode = opaOutModeDisable;
  OPAMP_Enable(VDAC0, OPA0, &initTemp);

  // Configure second stage OPA
  // Following stage positive input from previous output
  initTemp.posSel = opaPosSelOpaIn;
  initTemp.resSel = resSel1;
  OPAMP_Enable(VDAC0, OPA1, &initTemp);

  // Configure third stage OPA
  initTemp.resSel = resSel2;
  OPAMP_Enable(VDAC0, OPA2, &initTemp);

  // Configure fourth stage OPA
  initTemp.resSel = resSel3;
  initTemp.outMode = opaOutput;
  initTemp.outPen = altOutMask; 
  OPAMP_Enable(VDAC0, OPA3, &initTemp);
}              
#endif

/***************************************************************************//**
 * @brief
 *   Configure and enable two OPAs in cascaded inverting amplifier mode.
 *
 * @param[in] firstStage
 *   Select an OPA for first stage, valid values are OPA0, OPA1, and OPA2.
 *
 * @param[in] resInInput
 *   First stage resistor ladder input source.
 *
 * @param[in] posInput0
 *   First stage OPA positive input source.
 *
 * @param[in] resSel0
 *   First stage internal resistor ladder ratio.
 *
 * @param[in] posInput1
 *   Second stage OPA positive input source.
 *
 * @param[in] resSel1
 *   Second stage internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   Second stage OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaCascadeTwoInvertAmp(OPAMP_TypeDef firstStage,
                            OPAMP_ResInMux_TypeDef resInInput,
                            OPAMP_PosSel_TypeDef posInput0,
                            OPAMP_ResSel_TypeDef resSel0,
                            OPAMP_PosSel_TypeDef posInput1,
                            OPAMP_ResSel_TypeDef resSel1,
                            OPAMP_OutMode_TypeDef opaOutput,
                            uint32_t altOutMask)              
{
  OPAMP_Init_TypeDef initFirst = OPA_INIT_CASCADED_INVERTING_OPA0;
  OPAMP_Init_TypeDef initSecond = OPA_INIT_CASCADED_INVERTING_OPA1;

  // Disable inputs OVT
  resInOvtDisable(firstStage, resInInput);
  posSelOvtDisable(firstStage, posInput0);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure first stage OPA
  initFirst.resInMux = resInInput;
  initFirst.posSel = posInput0;
  initFirst.resSel = resSel0;
  initFirst.outMode = opaOutModeDisable;

  // Configure second stage OPA
  initSecond.posSel = posInput1;
  initSecond.resSel = resSel1;
  initSecond.outMode = opaOutput;
  initSecond.outPen = altOutMask; 
  
  // Check which OPA is first stage  
  switch (firstStage) {
    case OPA0:
      posSelOvtDisable(OPA1, posInput1);
      OPAMP_Enable(VDAC0, OPA0, &initFirst);
      OPAMP_Enable(VDAC0, OPA1, &initSecond);
      break;
      
#if defined(VDAC_STATUS_OPA2ENS)
    // Only valid for device with OPA2
    case OPA1:
      posSelOvtDisable(OPA2, posInput1);
      OPAMP_Enable(VDAC0, OPA1, &initFirst);
      OPAMP_Enable(VDAC0, OPA2, &initSecond);
      break;
#endif
      
#if defined(VDAC_STATUS_OPA3ENS)
    // Only valid for device with OPA3
    case OPA2:
      posSelOvtDisable(OPA3, posInput1);
      OPAMP_Enable(VDAC0, OPA2, &initFirst);
      OPAMP_Enable(VDAC0, OPA3, &initSecond);
      break;
#endif
      
    default:
      EFM_ASSERT(false);
      break;
  }
}              

#if defined(VDAC_STATUS_OPA2ENS)
/***************************************************************************//**
 * @brief
 *   Configure and enable three OPAs in cascaded inverting amplifier mode.
 *
 * @param[in] firstStage
 *   Select an OPA for first stage, valid values are OPA0, and OPA1.
 *
 * @param[in] resInInput
 *   First stage resistor ladder input source.
 *
 * @param[in] posInput0
 *   First stage OPA positive input source.
 *
 * @param[in] resSel0
 *   First stage internal resistor ladder ratio.
 *
 * @param[in] posInput1
 *   Second stage OPA positive input source.
 *
 * @param[in] resSel1
 *   Second stage internal resistor ladder ratio.
 *
 * @param[in] posInput2
 *   Third stage OPA positive input source.
 *
 * @param[in] resSel2
 *   Third stage internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   Third stage OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaCascadeThreeInvertAmp(OPAMP_TypeDef firstStage,
                              OPAMP_ResInMux_TypeDef resInInput,
                              OPAMP_PosSel_TypeDef posInput0,
                              OPAMP_ResSel_TypeDef resSel0,
                              OPAMP_PosSel_TypeDef posInput1,
                              OPAMP_ResSel_TypeDef resSel1,
                              OPAMP_PosSel_TypeDef posInput2,
                              OPAMP_ResSel_TypeDef resSel2,
                              OPAMP_OutMode_TypeDef opaOutput,
                              uint32_t altOutMask)              
{
  OPAMP_Init_TypeDef initTemp = OPA_INIT_CASCADED_INVERTING_OPA0;

  // Disable inputs OVT
  resInOvtDisable(firstStage, resInInput);
  posSelOvtDisable(firstStage, posInput0);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure first stage OPA
  initTemp.resInMux = resInInput;
  initTemp.posSel = posInput0;
  initTemp.resSel = resSel0;
  initTemp.outMode = opaOutModeDisable;

  // Check which OPA is first stage  
  switch (firstStage) {
    case OPA0:
      posSelOvtDisable(OPA1, posInput1);
      posSelOvtDisable(OPA2, posInput2);
      OPAMP_Enable(VDAC0, OPA0, &initTemp);

      // Configure second stage OPA
      // Following stage resistor ladder input from previous output
      initTemp.resInMux = opaResInMuxOpaIn;
      initTemp.posSel = posInput1;
      initTemp.resSel = resSel1;
      OPAMP_Enable(VDAC0, OPA1, &initTemp);

      // Configure third stage OPA
      initTemp.posSel = posInput2;
      initTemp.resSel = resSel2;
      initTemp.outMode = opaOutput;
      initTemp.outPen = altOutMask; 
      OPAMP_Enable(VDAC0, OPA2, &initTemp);
      break;
      
#if defined(VDAC_STATUS_OPA3ENS)
    // Only valid for device with OPA3
    case OPA1:
      posSelOvtDisable(OPA2, posInput1);
      posSelOvtDisable(OPA3, posInput2);
      OPAMP_Enable(VDAC0, OPA1, &initTemp);

      // Configure second stage OPA
      // Following stage resistor ladder input from previous output
      initTemp.resInMux = opaResInMuxOpaIn;
      initTemp.posSel = posInput1;
      initTemp.resSel = resSel1;
      OPAMP_Enable(VDAC0, OPA2, &initTemp);

      // Configure third stage OPA
      initTemp.posSel = posInput2;
      initTemp.resSel = resSel2;
      initTemp.outMode = opaOutput;
      initTemp.outPen = altOutMask; 
      OPAMP_Enable(VDAC0, OPA3, &initTemp);
      break;
#endif
      
    default:
      EFM_ASSERT(false);
      break;
  }
}              
#endif

#if defined(VDAC_STATUS_OPA3ENS)
/***************************************************************************//**
 * @brief
 *   Configure and enable four OPAs in cascaded inverting amplifier mode.
 *
 * @param[in] resInInput
 *   First stage resistor ladder input source.
 *
 * @param[in] posInput0
 *   First stage OPA positive input source.
 *
 * @param[in] resSel0
 *   First stage internal resistor ladder ratio.
 *
 * @param[in] posInput1
 *   Second stage OPA positive input source.
 *
 * @param[in] resSel1
 *   Second stage internal resistor ladder ratio.
 *
 * @param[in] posInput2
 *   Third stage OPA positive input source.
 *
 * @param[in] resSel2
 *   Third stage internal resistor ladder ratio.
 *
 * @param[in] posInput3
 *   Fourth stage OPA positive input source.
 *
 * @param[in] resSel3
 *   Fourth stage internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   Fourth stage OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaCascadeFourInvertAmp(OPAMP_ResInMux_TypeDef resInInput,
                             OPAMP_PosSel_TypeDef posInput0,
                             OPAMP_ResSel_TypeDef resSel0,
                             OPAMP_PosSel_TypeDef posInput1,
                             OPAMP_ResSel_TypeDef resSel1,
                             OPAMP_PosSel_TypeDef posInput2,
                             OPAMP_ResSel_TypeDef resSel2,
                             OPAMP_PosSel_TypeDef posInput3,
                             OPAMP_ResSel_TypeDef resSel3,
                             OPAMP_OutMode_TypeDef opaOutput,
                             uint32_t altOutMask)              
{
  OPAMP_Init_TypeDef initTemp = OPA_INIT_CASCADED_INVERTING_OPA0;

  // Disable inputs OVT
  resInOvtDisable(OPA0, resInInput);
  posSelOvtDisable(OPA0, posInput0);
  posSelOvtDisable(OPA1, posInput1);
  posSelOvtDisable(OPA2, posInput2);
  posSelOvtDisable(OPA3, posInput3);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure first stage OPA
  initTemp.resInMux = resInInput;
  initTemp.posSel = posInput0;
  initTemp.resSel = resSel0;
  initTemp.outMode = opaOutModeDisable;
  OPAMP_Enable(VDAC0, OPA0, &initTemp);
  
  // Configure second stage OPA
  // Following stage resistor ladder input from previous output
  initTemp.resInMux = opaResInMuxOpaIn;
  initTemp.posSel = posInput1;
  initTemp.resSel = resSel1;
  OPAMP_Enable(VDAC0, OPA1, &initTemp);
  
  // Configure third stage OPA
  initTemp.posSel = posInput2;
  initTemp.resSel = resSel2;
  OPAMP_Enable(VDAC0, OPA2, &initTemp);
  
  // Configure fourth stage OPA
  initTemp.posSel = posInput3;
  initTemp.resSel = resSel3;
  initTemp.outMode = opaOutput;
  initTemp.outPen = altOutMask; 
  OPAMP_Enable(VDAC0, OPA3, &initTemp);
}              
#endif

/***************************************************************************//**
 * @brief
 *   Configure and enable two OPAs in differential amplifier mode.
 *
 * @param[in] firstStage
 *   Select an OPA for first stage, valid values are OPA0, OPA1, and OPA2.
 *
 * @param[in] posInput0
 *   First stage OPA positive input source.
 *
 * @param[in] posInput1
 *   Second stage OPA positive input source.
 *
 * @param[in] resSel1
 *   Second stage internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   Second stage OPA output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaTwoDiffAmp(OPAMP_TypeDef firstStage,
                   OPAMP_PosSel_TypeDef posInput0,
                   OPAMP_PosSel_TypeDef posInput1,
                   OPAMP_ResSel_TypeDef resSel1,
                   OPAMP_OutMode_TypeDef opaOutput,             
                   uint32_t altOutMask)              
{
  OPAMP_Init_TypeDef initFirst = OPA_INIT_DIFF_DRIVER_OPA0;
  OPAMP_Init_TypeDef initSecond = OPA_INIT_DIFF_DRIVER_OPA1;

  // Disable input OVT
  posSelOvtDisable(firstStage, posInput0);
  
  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure first OPA
  initFirst.posSel = posInput0;
  initFirst.outMode = opaOutModeDisable;
  
  // Configure second OPA
  initSecond.posSel = posInput1;
  initSecond.resSel = resSel1;
  initSecond.outMode = opaOutput;
  initSecond.outPen = altOutMask; 

  // Check which OPA is first  
  switch (firstStage) {
    case OPA0:
      posSelOvtDisable(OPA1, posInput1);
      OPAMP_Enable(VDAC0, OPA0, &initFirst);
      OPAMP_Enable(VDAC0, OPA1, &initSecond);
      break;

#if defined(VDAC_STATUS_OPA2ENS)
    // Only valid for device with OPA2
  case OPA1:
    posSelOvtDisable(OPA2, posInput1);
    OPAMP_Enable(VDAC0, OPA1, &initFirst);
    OPAMP_Enable(VDAC0, OPA2, &initSecond);
    break;
#endif
    
#if defined(VDAC_STATUS_OPA3ENS)
    // Only valid for device with OPA3
  case OPA2:
    posSelOvtDisable(OPA3, posInput1);
    OPAMP_Enable(VDAC0, OPA2, &initFirst);
    OPAMP_Enable(VDAC0, OPA3, &initSecond);
    break;
#endif
    
  default:
    EFM_ASSERT(false);
    break;
  }
}              

#if defined(VDAC_STATUS_OPA2ENS)
/***************************************************************************//**
 * @brief
 *   Configure and enable OPA0, 1, and 2 in three differential amplifier mode.
 *
 * @param[in] posInput0
 *   OPA0 positive input source.
 *
 * @param[in] posInput1
 *   OPA1 positive input source.
 *
 * @param[in] resSel0
 *   OPA0 internal resistor ladder ratio.
 *
 * @param[in] resSel2
 *   OPA2 internal resistor ladder ratio.
 *
 * @param[in] opaOutput
 *   OPA2 output source.
 *
 * @param[in] altOutMask
 *   Bit mask to enable alternate output(s) if opaOutput = opaOutModeAlt.
 ******************************************************************************/
void opaThreeDiffAmp(OPAMP_PosSel_TypeDef posInput0,
                     OPAMP_PosSel_TypeDef posInput1,
                     OPAMP_ResSel_TypeDef resSel0,
                     OPAMP_ResSel_TypeDef resSel2,
                     OPAMP_OutMode_TypeDef opaOutput,             
                     uint32_t altOutMask)              
{
  OPAMP_Init_TypeDef initOpa0 = OPA_INIT_DIFF_RECEIVER_OPA0;
  OPAMP_Init_TypeDef initOpa1 = OPA_INIT_DIFF_RECEIVER_OPA1;
  OPAMP_Init_TypeDef initOpa2 = OPA_INIT_DIFF_RECEIVER_OPA2;

  // Disable inputs OVT
  posSelOvtDisable(OPA0, posInput0);
  posSelOvtDisable(OPA1, posInput1);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure OPA0
  initOpa0.posSel   = posInput0; 
  initOpa0.resSel   = resSel0;    
  initOpa0.resInMux = opaResInMuxVss;
  initOpa0.outMode = opaOutModeDisable;
  
  // Configure OPA1
  initOpa1.posSel  = posInput1;  
  initOpa1.outMode = opaOutModeDisable;

  // Configure OPA2
  initOpa2.resSel  = resSel2;     
  initOpa2.outMode = opaOutput; 
  initOpa2.outPen = altOutMask; 

  OPAMP_Enable(VDAC0, OPA0, &initOpa0);
  OPAMP_Enable(VDAC0, OPA1, &initOpa1);
  OPAMP_Enable(VDAC0, OPA2, &initOpa2);
}
#endif

/***************************************************************************//**
 * @brief
 *   Configure and enable two OPAs in instrumentation amplifier mode.
 *
 * @param[in] firstStage
 *   Select an OPA0 or OPA2 for first stage.
 *
 * @param[in] posInput0
 *   OPA0 or OPA2 positive input source.
 *
 * @param[in] posInput1
 *   OPA1 or OPA3 positive input source.
 *
 * @param[in] resSel0
 *   OPA0 or OPA2 internal resistor ladder ratio.
 *
 * @param[in] resSel1
 *   OPA1 or OPA3 internal resistor ladder ratio.
 *
 * @param[in] opaOutput0
 *   OPA0 or OPA2 output source.
 *
 * @param[in] opaOutput1
 *   OPA1 or OPA3 output source.
 *
 * @param[in] altOutMask0
 *   OPA0 or OPA2 alternate o/p enable bit mask if opaOutput0 = opaOutModeAlt.
 *
 * @param[in] altOutMask1
 *   OPA1 or OPA3 alternate o/p enable bit mask if opaOutput1 = opaOutModeAlt.
 ******************************************************************************/
void opaInstrAmp(OPAMP_TypeDef firstStage,
                 OPAMP_PosSel_TypeDef posInput0,
                 OPAMP_PosSel_TypeDef posInput1,
                 OPAMP_ResSel_TypeDef resSel0,
                 OPAMP_ResSel_TypeDef resSel1,
                 OPAMP_OutMode_TypeDef opaOutput0,             
                 OPAMP_OutMode_TypeDef opaOutput1,             
                 uint32_t altOutMask0,
                 uint32_t altOutMask1)              
{
  OPAMP_Init_TypeDef initFirst = OPA_INIT_INSTR_AMP_OPA0;
  OPAMP_Init_TypeDef initSecond = OPA_INIT_INSTR_AMP_OPA0;

  // Disable input OVT
  posSelOvtDisable(firstStage, posInput0);

  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Configure first OPA
  initFirst.posSel = posInput0;
  initFirst.resSel = resSel0;
  initFirst.outMode = opaOutput0;
  initFirst.outPen = altOutMask0; 
  
  // Configure second OPA
  initSecond.posSel = posInput1;
  initSecond.resSel = resSel1;
  initSecond.outMode = opaOutput1;
  initSecond.outPen = altOutMask1; 

  // Check which OPA is first  
  switch (firstStage) {
    case OPA0:
      posSelOvtDisable(OPA1, posInput1);
      OPAMP_Enable(VDAC0, OPA0, &initFirst);
      OPAMP_Enable(VDAC0, OPA1, &initSecond);
      break;
    
#if defined(VDAC_STATUS_OPA3ENS)
    // Only valid for device with OPA3
    case OPA2:
      posSelOvtDisable(OPA3, posInput1);
      OPAMP_Enable(VDAC0, OPA2, &initFirst);
      OPAMP_Enable(VDAC0, OPA3, &initSecond);
      break;
#endif
    
    default:
      EFM_ASSERT(false);
      break;
  }
}              

/***************************************************************************//**
 * @brief
 *   Initialize OPA with PRS trigger.
 *
 * @param[in] opaSelect
 *   Select an OPA, valid values are OPA0, OPA1, OPA2, and OPA3.
 *
 * @param[in] opaPrsMode
 *   Select PLUSED or TIMED trigger.
 *
 * @param[in] opaPrsInCh
 *   PRS input channel.
 *
 * @param[in] opaPrsOut
 *   Select WARM or OUTVALID output to PRS.
 *
 * @param[in] opaPrsOutCh
 *   PRS output channel.
 ******************************************************************************/
void initOpaPrs(OPAMP_TypeDef opaSelect,
                OPAMP_PrsMode_TypeDef opaPrsMode,
                OPAMP_PrsSel_TypeDef opaPrsInCh,
                OPAMP_PrsOut_TypeDef opaPrsOut,
                uint32_t opaPrsOutCh)
{
  // Enable VDAC clock
  CMU_ClockEnable(cmuClock_VDAC0, true);

  // Disable OPAMP before writing to CTRL register.
  OPAMP_Disable(VDAC0, opaSelect);  

  // Setup PRS to trigger OPA and OPA PRS output mode
  VDAC0->OPA[opaSelect].CTRL |= VDAC_OPA_CTRL_PRSEN 
                                | (uint32_t)opaPrsMode
                                | (uint32_t)opaPrsInCh
                                | (uint32_t)opaPrsOut;
 
  // Setup OPA output as PRS source
  CMU_ClockEnable(cmuClock_PRS, true);
  switch (opaSelect) {
    case OPA0:
      PRS_SourceAsyncSignalSet(opaPrsOutCh, PRS_CH_CTRL_SOURCESEL_VDAC0, 
                               PRS_CH_CTRL_SIGSEL_VDAC0OPA0);
      
      VDAC0->CMD = VDAC_CMD_OPA0EN;
      break;

    case OPA1:
      PRS_SourceAsyncSignalSet(opaPrsOutCh, PRS_CH_CTRL_SOURCESEL_VDAC0, 
                               PRS_CH_CTRL_SIGSEL_VDAC0OPA1);
      VDAC0->CMD = VDAC_CMD_OPA1EN;
      break;
      
#if defined(VDAC_STATUS_OPA2ENS)
      // Only valid for device with OPA2
    case OPA2:
      PRS_SourceAsyncSignalSet(opaPrsOutCh, PRS_CH_CTRL_SOURCESEL_VDAC0, 
                               PRS_CH_CTRL_SIGSEL_VDAC0OPA2);
      VDAC0->CMD = VDAC_CMD_OPA2EN;
      break;
#endif
      
#if defined(VDAC_STATUS_OPA3ENS)
    // Only valid for device with OPA3
    case OPA3:
      PRS_SourceAsyncSignalSet(opaPrsOutCh, PRS_CH_CTRL_SOURCESEL_VDAC0, 
                               PRS_CH_CTRL_SIGSEL_VDAC0OPA3);
      VDAC0->CMD = VDAC_CMD_OPA3EN;
      break;
#endif
      
    default:
      EFM_ASSERT(false);
      break;
  }
}              
