/***************************************************************************//**
 * @file opamp_adc.c
 * @brief Functions for OPA as ADC input front-end example
 * @version  1.00

 *******************************************************************************
 * # License
 * <b>Copyright 2019 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/

#include "em_adc.h"
#include "em_cmu.h"
#include "em_ldma.h"
#include "em_letimer.h"
#include "em_prs.h"
#include "opamp_adc.h"
#include "opamp_config.h"
#include <stdio.h>

static LDMA_Descriptor_t descr;                 // LDMA descriptor
static LDMA_TransferCfg_t trans;                // Transfer configuration struct
static uint32_t adcBuffer[ADC_BUFFER_SIZE];     // Buffer for ADC conversion

/**************************************************************************//**
 * @brief LDMA Handler
 *****************************************************************************/
void LDMA_IRQHandler(void)
{
  uint32_t pending;

  // Read and clear interrupt source
  pending = LDMA->IF;
  LDMA_IntClear(pending);

  // Check for LDMA error
  if ( pending & LDMA_IF_ERROR )
  {
    // Loop here to enable the debugger to see what has happened
    while (1)
      ;
  }
}

/**************************************************************************//**
 * @brief LETIMER initialization
 *****************************************************************************/
void initLetimer(void)
{
  LETIMER_Init_TypeDef letimerInit = LETIMER_INIT_DEFAULT;

  // Enable clock to the interface of the low energy modules
  CMU_ClockEnable(cmuClock_HFLE, true);

  // Route the LF clock to LFA
  CMU_ClockSelectSet(cmuClock_LFA, LETIMER_CLKSRC);

  // Enable clock for LETIMER0
  CMU_ClockEnable(cmuClock_LETIMER0, true);

  // Reload COMP0 on underflow, pulse output, and run in repeat mode
  letimerInit.comp0Top  = true;
  letimerInit.ufoa0     = letimerUFOAPulse;

  // Initialize LETIMER
  LETIMER_Init(LETIMER0, &letimerInit);

  // Need REP0 != 0 to pulse on underflow
  LETIMER_RepeatSet(LETIMER0, 0, 1);

  // Compare on wake-up interval count
  LETIMER_CompareSet(LETIMER0, 0, LETIMER_COMPARE);

  // Use LETIMER0 as async PRS to trigger OPA or ADC in EM2
  CMU_ClockEnable(cmuClock_PRS, true);
#if (USE_OPA_PRS == 0)
  PRS_SourceAsyncSignalSet(PRS_ADC_CHANNEL, PRS_CH_CTRL_SOURCESEL_LETIMER0,
                           PRS_CH_CTRL_SIGSEL_LETIMER0CH0);
#else
  PRS_SourceAsyncSignalSet(PRS_OPA_CHANNEL, PRS_CH_CTRL_SOURCESEL_LETIMER0,
                           PRS_CH_CTRL_SIGSEL_LETIMER0CH0);
#endif
}

/**************************************************************************//**
 * @brief LDMA initialization
 *****************************************************************************/
void initLdma(void)
{
  // Basic LDMA configuration
  LDMA_Init_t ldmaInit = LDMA_INIT_DEFAULT;

  // Initialize LDMA and enable interrupt
  LDMA_Init(&ldmaInit);

  // ADC single conversion complete as DMA trigger source
  trans = (LDMA_TransferCfg_t)
    LDMA_TRANSFER_CFG_PERIPHERAL(ldmaPeripheralSignal_ADC0_SINGLE);

  // Initialize LDMA descriptor
  descr = (LDMA_Descriptor_t)
    LDMA_DESCRIPTOR_LINKREL_P2M_WORD(&(ADC0->SINGLEDATA),       // Source
                                     adcBuffer,                 // Destination
                                     ADC_BUFFER_SIZE,           // Size
                                     0);                        // Link to self

  descr.xfer.ignoreSrec = 1;    // Ignores single requests to save energy
  descr.xfer.blockSize = ADC_SINGLE_DVL - 1;

  // Initialize transfer
  LDMA_StartTransfer(ADC_LDMA_CHANNEL, &trans, &descr);
}

/**************************************************************************//**
 * @brief ADC initialization
 *****************************************************************************/
void initAdc(void)
{
  // Declare initialization structs
  ADC_Init_TypeDef init = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef initSingle = ADC_INITSINGLE_DEFAULT;

  // Select AUXHFRCO for ADC ASYNC mode so it can run in EM2
  CMU->ADCCTRL = CMU_ADCCTRL_ADC0CLKSEL_AUXHFRCO;

  // Enable ADC clock
  CMU_ClockEnable(cmuClock_ADC0, true);

  // Set AUXHFRCO frequency and use it to initialize the ADC
  CMU_AUXHFRCOFreqSet(ADC_ASYNC_CLOCK);
  init.timebase = ADC_TimebaseCalc(CMU_AUXHFRCOBandGet());
  init.prescale = ADC_PrescaleCalc(ADC_FREQ, CMU_AUXHFRCOBandGet());
  ADC_Init(ADC0, &init);

  // Initialize single conversion
  initSingle.posSel = ADC_INPUT;        // Select input
  initSingle.reference = ADC_REF;       // Select reference
  initSingle.acqTime = ADC_AT;          // Set acquisition time
  initSingle.prsEnable = true;          // Enable PRS trigger
  initSingle.prsSel = ADC_PRS_CHANNEL;  // Set PRS channel      
  initSingle.singleDmaEm2Wu = true;     // DMA is available in EM2
  ADC_InitSingle(ADC0, &initSingle);    // ADC single initialization
  
  // Set single data valid level (DVL) to trigger
  ADC0->SINGLECTRLX = (ADC0->SINGLECTRLX & ~_ADC_SINGLECTRLX_DVL_MASK)
    | (ADC_SINGLE_DVL - 1) << _ADC_SINGLECTRLX_DVL_SHIFT;
  
  // Clear the Single FIFO
  ADC0->SINGLEFIFOCLEAR = ADC_SINGLEFIFOCLEAR_SINGLEFIFOCLEAR;

  // Switch the ADCCLKMODE to ASYNC at the end of initialization
  BUS_RegMaskedWrite(&ADC0->CTRL,
                     _ADC_CTRL_ADCCLKMODE_MASK | _ADC_CTRL_ASYNCCLKEN_MASK,
                     adcEm2ClockOnDemand);
}

/**************************************************************************//**
 * @brief Print voltage in adcBuffer[15]
 *****************************************************************************/
void printVoltage(void)
{
  printf("\nADC input voltage: %lu mV\n", (3300 * adcBuffer[15])/4096);
}
