/**************************************************************************//**
 * @file em_usbdep.c
 * @brief USB protocol stack library, USB device endpoint handlers.
 * @version 1.22.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2014 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "em_device.h"
#if defined( USB_PRESENT ) && ( USB_COUNT == 1 )
#include "em_usb.h"
#if defined( USB_DEVICE )

#include "em_usbtypes.h"
#include "em_usbhal.h"
#include "em_usbd.h"

/** @cond DO_NOT_INCLUDE_WITH_DOXYGEN */

/*
 * USBDEP_Ep0Handler() is called each time a packet has been transmitted
 * or recieved on the default endpoint.
 * A state machine navigate us through the phases of a control transfer
 * according to "chapter 9" in the USB spec.
 */
#if !defined( USB_DOEP0INT_STUPPKTRCVD )
__ramfunc void USBDEP_Ep0Handler( USBD_Device_TypeDef *device )
{
  int status;
  USBD_Ep_TypeDef *ep;
  static bool statusIn;
  static uint32_t xferred;
  static USB_XferCompleteCb_TypeDef callback;

  ep = &device->ep[ 0 ];
  switch ( ep->state )
  {
    case D_EP_IDLE:
      ep->remaining = 0;
      ep->zlp = 0;
      callback = NULL;
      statusIn = false;

      status = USBDCH9_SetupCmd( device );

      if ( status == USB_STATUS_REQ_ERR )
      {
        ep->in = true;
        USBDHAL_StallEp( ep );              /* Stall Ep0 IN                 */
        ep->in = false;                     /* OUT for next SETUP           */
        USBDHAL_StallEp( ep );              /* Stall Ep0 OUT                */
        USBDHAL_ReenableEp0Setup( device ); /* Prepare for next SETUP packet*/
      }
      else /* ( Status == USB_STATUS_OK ) */
      {
        if ( (ep->state == D_EP_RECEIVING) || (ep->state == D_EP_TRANSMITTING) )
        {
          callback = ep->xferCompleteCb;
        }

        if ( ep->state != D_EP_RECEIVING )
        {
          if ( ep->remaining )
          {
            /* Data will be sent to host, check if a ZLP must be appended */
            if ( ( ep->remaining < device->setup->wLength ) &&
                 ( ep->remaining % ep->packetSize == 0    )    )
            {
              ep->zlp = 1;
            }
          }
          else
          {
            /* Prepare for next SETUP packet*/
            USBDHAL_ReenableEp0Setup( device );

            /* No data stage, a ZLP may have been sent. If not, send one */

            xferred = 0;
            if ( ep->zlp == 0 )
            {
              USBD_Write( 0, NULL, 0, NULL );             /* ACK to host */
              ep->state = D_EP0_IN_STATUS;
            }
            else
            {
              ep->state = D_EP_IDLE;
              ep->in = false;                      /* OUT for next SETUP */
            }
          }
        }
      }
      break;

    case D_EP_RECEIVING:
      if ( ep->remaining )
      {
        /* There is more data to receive */
        USBD_ReArmEp0( ep );
      }
      else
      {
        status = USB_STATUS_OK;
        if ( callback != NULL )
        {
#pragma diag_suppress=Ta022
          status = callback( USB_STATUS_OK, ep->xferred, 0 );
#pragma diag_default=Ta022
          callback = NULL;
        }

        if ( status != USB_STATUS_OK )
        {
          ep->in = true;
          USBDHAL_StallEp( ep );              /* Stall Ep0 IN                */
          ep->in = false;                     /* OUT for next SETUP           */
          USBDHAL_StallEp( ep );              /* Stall Ep0 OUT                */
          USBDHAL_ReenableEp0Setup( device ); /* Prepare for next SETUP pkt. */
          ep->state = D_EP_IDLE;
        }
        else /* Everything OK, send a ZLP (ACK) to host */
        {
          USBDHAL_ReenableEp0Setup( device );/* Prepare for next SETUP packet*/

          ep->state = D_EP_IDLE;              /* USBD_Write() sets state back*/
                                              /* to EP_TRANSMITTING          */
          USBD_Write( 0, NULL, 0, NULL );
          ep->state = D_EP0_IN_STATUS;
        }
      }
      break;

    case D_EP_TRANSMITTING:
      if ( ep->remaining )
      {
        /* There is more data to transmit */
        USBD_ReArmEp0( ep );
      }
      else
      {
        /* All data transferred, is a ZLP packet needed ? */
        if ( ep->zlp == 1 )
        {
          xferred   = ep->xferred;
          ep->state = D_EP_IDLE;          /* USBD_Write() sets state back */
                                          /* to EP_TRANSMITTING           */
          USBD_Write( 0, NULL, 0, NULL ); /* Send ZLP                     */
          ep->zlp = 2;
        }
        else
        {
          if ( ep->zlp == 0 )
          {
            xferred = ep->xferred;
          }

          ep->state = D_EP_IDLE;
          USBD_Read( 0, NULL, 0, NULL );  /* Get ZLP packet (ACK) from host */
          statusIn = true;
          ep->state = D_EP0_OUT_STATUS;
        }
      }
      break;

    case D_EP0_IN_STATUS:
    case D_EP0_OUT_STATUS:
      if ( statusIn )
      {
        USBDHAL_ReenableEp0Setup( device );
      }

      if ( callback != NULL )
      {
#pragma diag_suppress=Ta022
        callback( USB_STATUS_OK, xferred, 0 );
#pragma diag_default=Ta022
      }

      ep->state = D_EP_IDLE;
      ep->in = false;                     /* OUT for next SETUP */
      break;

    default:
      EFM_ASSERT( false );
      break;
  }
}
#endif

#if defined( USB_DOEP0INT_STUPPKTRCVD )
__ramfunc void USBDEP_Ep0Handler( USBD_Device_TypeDef *device )
{
  int status;
  USBD_Ep_TypeDef *ep;
  static uint32_t xferred;
  static USB_XferCompleteCb_TypeDef callback;

  ep = &device->ep[ 0 ];

  switch ( ep->state )
  {
    case D_EP_IDLE:
      ep->zlp       = 0;
      ep->remaining = 0;
      callback      = NULL;

      status = USBDCH9_SetupCmd( device );

      if ( status == USB_STATUS_REQ_ERR )
      {
        ep->in = true;
        USBDHAL_StallEp( ep );              /* Stall Ep0 IN                 */
        ep->in = false;                     /* OUT for next SETUP           */
        USBDHAL_StallEp( ep );              /* Stall Ep0 OUT                */
        USBDHAL_StartEp0Setup( dev );       /* Prepare for next SETUP packet*/
      }
      else /* ( Status == USB_STATUS_OK ) */
      {
        if ( (ep->state == D_EP_RECEIVING) || (ep->state == D_EP_TRANSMITTING) )
        {
          callback = ep->xferCompleteCb;
        }

        if ( ep->state != D_EP_RECEIVING )
        {
          if ( ep->remaining )
          {
            /* Data will be sent to host, check if a ZLP must be appended */
#if defined( NO_RAMFUNCS )
            if ( ( ep->remaining < device->setup->wLength ) &&
                 ( ep->remaining % ep->packetSize == 0    )    )
#else
            // Dangerous optimization, OK when EP size is 64
#if ( USB_FS_CTRL_EP_MAXSIZE != 64 )
#error "Illegal optimization."
#else
            if ( ( ep->remaining < device->setup->wLength ) &&
                 ( ep->remaining & 0x3F == 0              )    )
#endif
#endif
            {
              ep->zlp = 1;
            }
          }
          else
          {
            /* No data stage, a ZLP may have been sent. If not, send one */
            xferred = 0;
            if ( ep->zlp == 0 )
            {
              ep->state = D_EP_IDLE;
              USBD_Write( 0, NULL, 0, NULL );             /* ACK to host */
              ep->state = D_EP0_IN_STATUS;
            }
          }
        }
      }
      break;

    case D_EP_RECEIVING:
      if ( ep->remaining )
      {
        ep->in = false;
        USBD_ReArmEp0( ep );
      }
      else
      {
        status = USB_STATUS_OK;
        if ( callback != NULL )
        {
#pragma diag_suppress=Ta022
          status = callback( USB_STATUS_OK, ep->xferred, 0 );
#pragma diag_default=Ta022
          callback = NULL;
        }

        if ( status != USB_STATUS_OK )
        {
          ep->in = true;
          USBDHAL_StallEp( ep );              /* Stall Ep0 IN                */
          ep->in = false;                     /* OUT for next SETUP          */
          USBDHAL_StallEp( ep );              /* Stall Ep0 OUT               */
          USBDHAL_StartEp0Setup( dev );       /* Prepare for next SETUP pkt. */
          ep->state = D_EP_IDLE;
        }
        else
        {

          USBDHAL_StartEp0Setup( dev );      /* Prepare for next SETUP packet*/
          ep->state = D_EP_IDLE;             /* USBD_Write() sets state back */
                                             /* to EP_TRANSMITTING           */
          USBD_Write( 0, NULL, 0, NULL );
          ep->state = D_EP0_IN_STATUS;
        }
      }
      break;

    case D_EP_TRANSMITTING:
      if ( ep->remaining )
      {
        ep->in = true;
        USBD_ReArmEp0( ep );
      }
      else
      {
        if ( ep->zlp == 1 )
        {
          xferred   = ep->xferred;
          ep->state = D_EP_IDLE;          /* USBD_Write() sets state back */
                                          /* to EP_TRANSMITTING           */
          USBD_Write( 0, NULL, 0, NULL ); /* Send ZLP                     */
          ep->zlp = 2;
        }
        else
        {
          if ( ep->zlp == 0 )
          {
            xferred = ep->xferred;
          }

          ep->state = D_EP_IDLE;
          USBD_Read( 0, NULL, 0, NULL );  /* Get ZLP packet (ACK) from host */
          ep->state = D_EP0_OUT_STATUS;
        }
      }
      break;

    case D_EP0_IN_STATUS:
      if ( ( USB->DOEP0CTL & USB_DOEP0CTL_EPENA ) == 0 )
      {
        /* Prepare for more SETUP Packets */
        USBDHAL_StartEp0Setup( dev );
      }
      if ( callback != NULL )
      {
#pragma diag_suppress=Ta022
        callback( USB_STATUS_OK, xferred, 0 );
#pragma diag_default=Ta022
      }
      ep->state = D_EP_IDLE;
      ep->in = false;                     /* OUT for next SETUP */
      break;

    case D_EP0_OUT_STATUS:
      USBDHAL_StartEp0Setup( dev );       /* Prepare for more SETUP Packets */
      if ( callback != NULL )
      {
#pragma diag_suppress=Ta022
        callback( USB_STATUS_OK, xferred, 0 );
#pragma diag_default=Ta022
      }
      ep->state = D_EP_IDLE;
      ep->in = false;                     /* OUT for next SETUP */
      break;
  }
}
#endif

/*
 * USBDEP_EpHandler() is called each time a packet has been transmitted
 * or recieved on an endpoint other than the default endpoint.
 */
void USBDEP_EpHandler( uint8_t epAddr )
{
  USB_XferCompleteCb_TypeDef callback;
  USBD_Ep_TypeDef *ep = USBD_GetEpFromAddr( epAddr );

  if ( ( ep->state == D_EP_TRANSMITTING ) || ( ep->state == D_EP_RECEIVING ) )
  {
    ep->state = D_EP_IDLE;
    if ( ep->xferCompleteCb )
    {
      callback = ep->xferCompleteCb;
      ep->xferCompleteCb = NULL;
#pragma diag_suppress=Ta022
      callback( USB_STATUS_OK, ep->xferred, ep->remaining );
#pragma diag_default=Ta022
    }
  }
  else
  {
    EFM_ASSERT( false );
  }
}

/** @endcond */

#endif /* defined( USB_DEVICE ) */
#endif /* defined( USB_PRESENT ) && ( USB_COUNT == 1 ) */
