/**************************************************************************//**
 * @file
 * @brief Flash erasing and writing functions for Speex record
 * @author Energy Micro AS
 * @version 1.04
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2013 Energy Micro AS, http://www.energymicro.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 * 4. The source and compiled code may only be used on Energy Micro "EFM32"
 *    microcontrollers and "EFR4" radios.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Energy Micro AS has no
 * obligation to support this Software. Energy Micro AS is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Energy Micro AS will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 *****************************************************************************/
#include "em_device.h"
#include "em_emu.h"
#include "em_msc.h"

/**************************************************************************//**
 * @brief MSC Interrupt handler
 *****************************************************************************/
void MSC_IRQHandler(void)
{
  MSC_IntClear(MSC_IFC_ERASE);
}
    
/***************************************************************************//**
* @brief
*   Initializes write for Flash > 512kB (read-while-write)
*******************************************************************************/
void flashInit512(void)
{
  /* Write MSC unlock code to enable interface */
  MSC->LOCK = MSC_UNLOCK_CODE;
  /* Enable AIDIS for flash write/erase, PREFETCH for encode */
  MSC->READCTRL |= MSC_READCTRL_AIDIS + MSC_READCTRL_PREFETCH;
  /* Enable memory controller with read-while-write */
  MSC->WRITECTRL = MSC_WRITECTRL_RWWEN + MSC_WRITECTRL_WREN + MSC_WRITECTRL_WDOUBLE;
}

/***************************************************************************//**
 * @brief
 *   Disables the flash controller for writing.
 ******************************************************************************/
void flashDeinit(void)
{
  /* Disable AIDIS and PREFETCH */
  MSC->READCTRL &= ~(MSC_READCTRL_AIDIS + MSC_READCTRL_PREFETCH);
  /* Disable writing to the MSC */
  MSC->WRITECTRL = 0;
  /* Lock the MSC */
  MSC->LOCK = 0;
}

/**************************************************************************//**
 * Mass erase a flash block.
 * This function will mass erase a 512K block on a Giant device.
 * This function will not return until the block has been erased.
 *
 * @param eraseCmd The mass erase command to write to MSC WRITECMD register.
 *                 Only ERASEMAIN1 can use in this application note.
 *                 Otherwise the code on lower 512 KB flash will be erased.     
 *                 Make sure there is no valid code or data on upper 512KB.
 *****************************************************************************/
void flashMassErase(uint32_t eraseCmd)
{
  /* Unlock mass erase */
  MSC->MASSLOCK = MSC_MASSLOCK_LOCKKEY_UNLOCK;

  /* Enable flash erase done interrupt */
  MSC_IntEnable(MSC_IF_ERASE);
  NVIC_ClearPendingIRQ(MSC_IRQn);
  NVIC_EnableIRQ(MSC_IRQn);
  
  /* Send Mass erase command */
  MSC->WRITECMD = eraseCmd;

  /* Waiting for erase to complete in EM1 */
  while((MSC->STATUS & MSC_STATUS_BUSY))
  {
    EMU_EnterEM1();
  }
  NVIC_DisableIRQ(MSC_IRQn);
  
  /* Lock mass erase */
  MSC->MASSLOCK = MSC_MASSLOCK_LOCKKEY_LOCK;
}
