/**************************************************************************//**
 * @file
 * @brief Setup Timer for this application note
 * @author Energy Micro AS
 * @version 1.04
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2013 Energy Micro AS, http://www.energymicro.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 * 4. The source and compiled code may only be used on Energy Micro "EFM32"
 *    microcontrollers and "EFR4" radios.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Energy Micro AS has no
 * obligation to support this Software. Energy Micro AS is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Energy Micro AS will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 *****************************************************************************/
#include "em_device.h"
#include "em_prs.h"
#include "config.h"

/***************************************************************************//**
 * @brief
 *   Configure Timer for this application.
 *******************************************************************************/
void initTimer(void)
{
  /* Use default setting of CTRL register, configure PRESC with different */
  /* BASE_FREQ and system core clock frequency, the BASE_FREQ depends on */
  /* maximum sampling frequency of ADC and DAC */
  TIMER_USED->CTRL |= ((31 - __CLZ(SystemCoreClock/BASE_FREQ)) << _TIMER_CTRL_PRESC_SHIFT);
#if (SPEEX_BAND < WIDEBAND || SPEEX_BAND == NB8KCODEC)
  TIMER_USED->TOP = SAMPLE_8K;
#endif
}

/***************************************************************************//**
 * @brief
 *   Enable PRS and configure TIMER TOP value for different sampling frequency.
 * @param bufferSize - Buffer size to setup sampling frequency  
 *******************************************************************************/
void setTopValue(uint16_t bufferSize)
{
  /* Select TIMER as source and Timer overflow as signal */
  PRS_SourceSignalSet(DAC_PRS_CH, TIMER_PRS, TIMER_SRC, prsEdgeOff);

#if SPEEX_BAND == WIDEBAND  
  /* Setup sampling frequency for WB to play NB or WB */
  if (bufferSize < SAMPLE_SIZE)
  {
    TIMER_USED->TOP = SAMPLE_8K;
  }
  else
  {
    TIMER_USED->TOP = SAMPLE_16K;
  }
#elif SPEEX_BAND == ULTRAWIDEBAND
  /* Setup sampling frequency for UWB to play NB or WB or UWB */
  if (bufferSize < SAMPLE_SIZE/2)
  {
    TIMER_USED->TOP = SAMPLE_8K;
  }
  else if (bufferSize < SAMPLE_SIZE)
  {
    TIMER_USED->TOP = SAMPLE_16K;
  }
  else
  {
    TIMER_USED->TOP = SAMPLE_32K;
  }    
#endif
  /* Sampling frequency is fixed on codec, NB8K and NB */  
  TIMER_USED->CMD = TIMER_CMD_START;
}
