/******************************************************************************
 * @file lcd_hello_world_s1.c
 * @brief LCD Hello World Example
 * @author Silicon Labs
 * @version  1.05

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "em_device.h"
#include "em_lcd.h"
#include "segmentlcd.h"
#include "em_chip.h"
#include "em_emu.h"
#include "em_cmu.h"
#include "bspconfig.h"

#define NUMBER_OF_DEMOS   4
#define LCD_STR_LEN       8

volatile uint8_t lcdDemo = 0;
volatile uint8_t lcdSelected = false;

static const char demoString[NUMBER_OF_DEMOS][LCD_STR_LEN] =
{
  "CHRGDST1",
  "CHRGDST2",
  "CHRGDST3",
  "CHRGDST4"
};

/**************************************************************************//**
 * @brief  Sets up LCD animation.
 *****************************************************************************/
void setupAnimation(void)
{
  // Configuration structure for LCD animation.
  static const LCD_AnimInit_TypeDef animInit =
  {
    .enable      = true,                  // Enable animation.
    .AReg        = 0x06,                  // Set up animation start data.
    .AShift      = lcdAnimShiftLeft,      // Register A Shift direction.
    .BReg        = 0x03,                  // Set up animation start data.
    .BShift      = lcdAnimShiftLeft,      // Register B Shift direction.
    .animLogic   = lcdAnimLogicOr,        // Logic function.

    // Adapt animation segments to the different kits.
    // Note that the 8 segments for animation of EFM32TG11 can be either
    // segments 0 to 7 or 8 to 15.
    // Demo the animation with SEG0-SEG7 on SLSTK3301A board.
#if defined(LCD_BACTRL_ALOC)
    .startSeg    = 0
#endif
  };

  // Configuration structure for frame counter.
  static const LCD_FrameCountInit_TypeDef fcInit =
  {
    .enable      = true,                  // Enable frame counter.
    .top         = 0x3,                   // Frame counter period.
    .prescale    = lcdFCPrescDiv8         // Set frame counter prescaler.
  };

  // Initialize Animation.
  LCD_AnimInit(&animInit);

  // Configure and start the frame counter which clocks
  // the animation state machine.
  LCD_FrameCountInit(&fcInit);
}

/***************************************************************************//**
 * @brief Common IRQ handler for the GPIO interrupts (pushbuttons)
 *        PB0 Starts selected demo mode.
 *        PB1 Cycles through the available demo modes.
 *****************************************************************************/
void GPIO_Unified_IRQ(void)
{
  // Get and clear all pending GPIO interrupts.
  uint32_t interruptMask;
  interruptMask = GPIO_IntGet();
  GPIO_IntClear(interruptMask);

  if ((interruptMask & (1 << BSP_GPIO_PB0_PIN)) || (interruptMask & (1 << BSP_GPIO_PB1_PIN)))
  {
    // Act on interrupts.
    if (interruptMask & (1 << BSP_GPIO_PB0_PIN))
    {
      lcdSelected = true;
    }
    if (interruptMask & (1 << BSP_GPIO_PB1_PIN))
    {
      lcdDemo = (lcdDemo + 1) % NUMBER_OF_DEMOS;
    }
  }
}

/***************************************************************************//**
 * @brief GPIO Interrupt handler for odd pins
 ******************************************************************************/
void GPIO_ODD_IRQHandler(void)
{
  GPIO_Unified_IRQ();
}

/***************************************************************************//**
 * @brief GPIO Interrupt handler for even pins
 ******************************************************************************/
void GPIO_EVEN_IRQHandler(void)
{
  GPIO_Unified_IRQ();
}

/**************************************************************************//**
 * @brief Configure two pushbuttons and enable interrupt
 *****************************************************************************/
void configButtonGPIO(void)
{
  // Enable GPIO clock.
  CMU_ClockEnable(cmuClock_GPIO, true);

  // Configure PB0 as input and enable interrupt.
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeInputPull, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, BSP_GPIO_PB0_PIN, false, true, true);

  // Configure PB1 as input and enable interrupt.
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInputPull, 1);
  GPIO_ExtIntConfig(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, BSP_GPIO_PB1_PIN, false, true, true);

  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
}

/**************************************************************************//**
 * @brief Selecting different demo modes to show the effect of charge
 *        redistribution and waveform setting.
 *        mode 1: normal waveform and charge redistribution Off
 *        mode 2: normal waveform and charge redistribution On
 *        mode 3: low power waveform and charge redistribution Off
 *        mode 4: low power waveform and charge redistribution On
 *****************************************************************************/
void selectLcdDemo(void)
{
  static uint32_t prevDemo = NUMBER_OF_DEMOS - 1;

  // Setup GPIO for pushbuttons.
  configButtonGPIO();

  while (lcdSelected == false)
  {
    if (prevDemo != lcdDemo)
    {
      prevDemo = lcdDemo;
      SegmentLCD_Write(demoString[lcdDemo]);
    }
  }

  // Disable GPIO interrupt after selecting demo mode done.
  NVIC_DisableIRQ(GPIO_ODD_IRQn);
  NVIC_DisableIRQ(GPIO_EVEN_IRQn);
  GPIO_PinModeSet(BSP_GPIO_PB0_PORT, BSP_GPIO_PB0_PIN, gpioModeDisabled, 1);
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeDisabled, 1);
}

/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main(void)
{
  LCD_Init_TypeDef lcdConf = LCD_INIT_DEF;

  // Initialize chip.
  CHIP_Init();

  // Initialize LCD controller with current source mode.

  // LCD Controller Prescaler (divide LFACLK / 8)
  // LFACLK_LCDpre = 32768/8 => 4096 Hz
  // Set FRDIV = 4, means LFACLKlcd = 4096/(1+4) => 819.2 Hz
  // With Octaplex multiplexing, frame Rate = 819.2/16 => 51.2 Hz
  SegmentLCD_Init(false);

  // Write something to text field on LCD display.
  SegmentLCD_Write("HELLO");

  // Turn on Gecko symbol.
  SegmentLCD_Symbol(LCD_SYMBOL_GECKO, 1);

  // Configure Animation, revolving door.
  setupAnimation();

  // Enable segment blinking.
  LCD_BlinkEnable(false);                 // Set to true to see blink feature.

  // Play around with settings below to adjust contrast.
  // These settings affect the current consumption of LCD.
  // Set contrast to suitable level.
  LCD_ContrastSet(0x12);

  // Selecting different demo modes.
  selectLcdDemo();

  switch (lcdDemo)
  {
    case 0:
      lcdConf.wave = lcdWaveNormal;
      lcdConf.frameRateDivider = 63;
      lcdConf.chargeRedistribution = lcdChargeRedistributionDisable;
      break;
    case 1:
      lcdConf.wave = lcdWaveNormal;
      lcdConf.frameRateDivider = 63;
      lcdConf.chargeRedistribution = lcdChargeRedistributionEnable;
      break;
    case 2:
      lcdConf.wave = lcdWaveLowPower;
      lcdConf.frameRateDivider = 63;
      lcdConf.chargeRedistribution = lcdChargeRedistributionDisable;
      break;
    case 3:
      lcdConf.wave = lcdWaveLowPower;
      lcdConf.frameRateDivider = 63;
      lcdConf.chargeRedistribution = lcdChargeRedistributionEnable;
      break;
    default:
      break;
  }

  // Disable Animation and Frame Counter before turning on all of the segments.
  LCD_AnimEnable(false);
  LCD_FrameCountEnable(false);

  // Charge redistribution cycle percentage should be 5% or less to prevent
  // reduction in LCD pad RMS voltage.
  // LCD Controller Prescaler (divide LFACLK / 1)
  // LFACLK_LCDpre = 32768 Hz
  // Set FRDIV = 63, means LFACLKlcd = 32768/(63+1) => 512 Hz
  // With Octaplex multiplexing, frame Rate = 512/16 => 32 Hz
  // Charge redistribution cycle percentage is 1/(63+1) = 1.6% by default.
  CMU_ClockDivSet(cmuClock_LCDpre, cmuClkDiv_1);

  // Initialize and enable LCD controller.
  LCD_Init(&lcdConf);

  // Turn on all of the segments.
  LCD_ALL_SEGMENTS_ON();

  // Stay in this loop forever at end of program, the LCD controller
  // continues to output LCD waveforms according to the data that is
  // currently synchronized to the LCD Driver logic.
  while (1)
  {
    EMU_EnterEM2(false);
  }
}
