/***************************************************************************//**
 * @file main_uart_s0.c
 * @brief UART Flow Control Example
 * @author Silicon Labs
 * @version  1.04

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 *******************************************************************************/
#include <stdint.h>
#include <string.h>
#include <stdbool.h>
#include "em_device.h"
#include "em_usart.h"
#include "em_chip.h"
#include "em_cmu.h"
#include "em_gpio.h"
#include "em_emu.h"
#include "em_rtc.h"
#include "uart_flow_control_s0.h"

// This is the actual size of the receive buffer
#define REC_BUFFER_SIZE 30

// This is the number of elements before we activate flow control
#define REC_BUFFER_THRESHOLD 20

// A message to send
char *message = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

// The buffer used to store incoming data
uint8_t recBuffer[REC_BUFFER_SIZE];

// The current index of the receive buffer
volatile uint8_t bufferIndex = 0;

// Flag to see if it is time to send a new message
volatile bool sendEvent = false;

// The flow control configuration
static UART_FC_Config_TypeDef fcConfig;

// Select the flow control type
volatile uint8_t fcDemo = DEMO_NONE;
volatile uint8_t fcSelected = false;

#if defined(TFT_LCD)
DISPLAY_Device_t displayDevice;    // Display device handle.
#endif


static const char demoString[NUMBER_OF_DEMOS][LCD_STR_LEN] =
{
#if defined(SEGMENT_LCD)
  // Hardware flow control
  "HW FC ",
  // Software flow control
  "SW FC ",
  // Legacy hardware flow control Master
  "LeHW M",
  // Legacy hardware flow control Slave
  "LeHW S"
#elif defined(TFT_LCD)
  // Hardware flow control
  "HW Flow  \n  Control",
  // Software flow control
  "SW Flow  \n  Control",
  // Legacy hardware flow control Master
  "Legacy HW\n  Master ",
  // Legacy hardware flow control Slave
  "Legacy HW\n  Slave  "
#endif
};

/***************************************************************************//**
 * @brief Callback when a byte has been received
 ******************************************************************************/
void rxHandler(uint8_t receivedByte)
{
  // Store character in buffer
  recBuffer[bufferIndex++] = receivedByte;

  if ((bufferIndex >= REC_BUFFER_SIZE) && (fcConfig.mode == modeLegacyHW_Master))
  {
    // Simply wrap around buffer when the end is reached
    // The slave in legacy hardware flow control is always allowed to transmit,
    // and not be controlled by legacy master
    bufferIndex = 0;
  }
  else if (bufferIndex > REC_BUFFER_SIZE)
  {
    // We actually ran out of space in our buffer.
    // Here we should do some fault handling. This code
    // simply halts in a while loop so it is easy for a
    // debugger to identify the problem.
    while (1);
  }
}

/**************************************************************************//**
 * @brief Configures flow control parameters
 *****************************************************************************/
void initFlowControl(void)
{
  // Select the RTS pin
  fcConfig.rtsPin.port = USART_RTS_PORT;
  fcConfig.rtsPin.pin = USART_RTS_PIN;
  fcConfig.rtsPolarity = activeLow;

  // Select the CTS pin
  fcConfig.ctsPin.port = USART_CTS_PORT;
  fcConfig.ctsPin.pin = USART_CTS_PIN;
  fcConfig.ctsPolarity = activeLow;

  // Set the flow control mode
  fcConfig.mode = (UART_FC_Mode)fcDemo;

  // Register callback functions
  fcConfig.txCallback = NULL;
  fcConfig.rxCallback = rxHandler;

  // Initialize Flow Control
  uartFcInit(&fcConfig);
}

/***************************************************************************//**
 * @brief Configures USART and flow control parameters
 ******************************************************************************/
void initUART(USART_TypeDef *usart)
{
  // Enable the required clocks
  CMU_ClockEnable(USART_CLOCK, true);
  CMU_ClockEnable(cmuClock_GPIO, true);

  // Initialize the UART
  USART_InitAsync_TypeDef uartInit = USART_INITASYNC_DEFAULT;
  // RX/TX enable is done after initialization done.
  uartInit.enable = usartDisable;
  USART_InitAsync(usart, &uartInit);

  // Enable TX and RX pin
  GPIO_PinModeSet(USART_RX_PORT, USART_RX_PIN, gpioModeInput, 0);
  GPIO_PinModeSet(USART_TX_PORT, USART_TX_PIN, gpioModePushPull, 1);
  usart->ROUTE |= USART_ROUTE_RXPEN | USART_ROUTE_TXPEN | USART_LOCATION;

  // Configures flow control parameters
  initFlowControl();

  // Enable interrupt when character arrives
  USART_IntEnable(usart, USART_IEN_RXDATAV);
  NVIC_EnableIRQ(USART_RX_IRQn);

  // Enable USART receiver and transmitter.
  USART_Enable(usart, usartEnable);
}

/**************************************************************************//**
 * @brief Configures the RTC. The RTC in this example is only used to
 *        send messages periodically.
 *****************************************************************************/
void initRTC(void)
{
  // The CORELE clock must be enabled before the CPU
  // can access the LF peripherals
  CMU_ClockEnable(cmuClock_CORELE, true);

  // Enable the LFXO for RTC
  CMU_ClockSelectSet(cmuClock_LFA, cmuSelect_LFXO);

  // Enable the clock for the RTC
  CMU_ClockEnable(cmuClock_RTC, true);

  // Prescale the RTC
  CMU_ClockDivSet(cmuClock_RTC, cmuClkDiv_512);

  // Configure RTC to wrap around on COMP0
  RTC_Init_TypeDef rtcInit = RTC_INIT_DEFAULT;
  rtcInit.comp0Top = true;
  rtcInit.enable = false;
  RTC_Init(&rtcInit);

  // Overflow every second
  RTC_CompareSet(0, (CMU_ClockFreqGet(cmuClock_RTC) - 1));

  // Enable interrupt
  RTC_IntEnable(RTC_IEN_COMP0);
  NVIC_EnableIRQ(RTC_IRQn);

  // Start the RTC
  RTC_Enable(true);
}

void RTC_IRQHandler(void)
{
  // Clear the flag
  RTC_IntClear(RTC_IF_COMP0);

  // Request a new message to be sent
  sendEvent = true;
}

/**************************************************************************//**
 * @brief Process the received data.
 *
 * @details
 *   Dummy function to pretend that some processing is done.
 *   This only included to demonstrate the flow control.
 *****************************************************************************/
void processBuffer(void)
{
  uint32_t i;

  // Disable reception by deasserting RTS signal.
  // Note that the actual RX interrupt is never disabled.
  // We assume the other end will respect the signal.
  uartFcDisableRx();

  // Pretend that we have to do a large processing
  for (i = 0; i < 1000; i++);

  // Empty the buffer
  bufferIndex = 0;

  // Enable reception again
  uartFcEnableRx();
}

/**************************************************************************//**
 * @brief Configure two pushbuttons and enable interrupt
 *****************************************************************************/
void configButtonGPIO(void)
{
  // Enable GPIO clock.
  CMU_ClockEnable(cmuClock_GPIO, true);

  // Configure PB0 as input and enable interrupt.
  GPIO_PinModeSet(PB0_PORT, PB0_PIN, gpioModeInputPull, 1);
  GPIO_IntConfig(PB0_PORT, PB0_PIN, false, true, true);

  // Configure PB1 as input and enable interrupt.
  GPIO_PinModeSet(PB1_PORT, PB1_PIN, gpioModeInputPull, 1);
  GPIO_IntConfig(PB1_PORT, PB1_PIN, false, true, true);

  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
}

/**************************************************************************//**
 * @brief Selecting flow control demos.
 *****************************************************************************/
void selectFcDemo(void)
{
  static uint32_t prevDemo = NUMBER_OF_DEMOS - 1;

  // Setup GPIO for pushbuttons.
  configButtonGPIO();

  while (fcSelected == false)
  {
    if (prevDemo != fcDemo)
    {
      prevDemo = fcDemo;
      displayString(demoString[prevDemo]);
    }
  }

  // Disable GPIO interrupt after selecting demo mode done.
  NVIC_DisableIRQ(GPIO_ODD_IRQn);
  NVIC_DisableIRQ(GPIO_EVEN_IRQn);
  GPIO_PinModeSet(PB0_PORT, PB0_PIN, gpioModeDisabled, 1);
  GPIO_PinModeSet(PB1_PORT, PB1_PIN, gpioModeDisabled, 1);
}

/**************************************************************************//**
 * @brief Main function
 * The main loop will stay in EM1 and wait for interrupts to occur.
 * This will be either:
 *  - An RTC interrupt indicating that it is time to send a new message
 *  - An RX interrupt indicating that a new byte has been received
 *  - A TXBL interrupt indicating that a new byte should be sent
 *    (handled by the flow control routine)
 *  - A TXC interrupt indicating that the full message has been sent
 *****************************************************************************/
int main(void)
{
  // Initialize chip - handle erratas
  CHIP_Init();

  // A crystal is usually needed for UART communication
  CMU_ClockSelectSet(cmuClock_HF, cmuSelect_HFXO);

  // Initialize display module
  displayInit();

  // Selecting flow control demos
  selectFcDemo();

  initUART(USART_DEV);
  initRTC();

  while (1)
  {
    // Stay in sleep until something happens
    EMU_EnterEM1();

    // It is time to send a message
    if (sendEvent)
    {
      uartFcStartTx((uint8_t*)message, strlen(message));
      sendEvent = false;
    }

    // The slave in legacy hardware flow control is always allowed to transmit,
    // and not be controlled by Legacy master, master don't need check the threshold
    if (fcConfig.mode != modeLegacyHW_Master)
    {
      // The buffer is almost filled up
      if (bufferIndex > REC_BUFFER_THRESHOLD)
      {
        // Do some processing. This will temporarily halt transmission.
        processBuffer();
      }
    }
  }
}

