/**************************************************************************//**
 * @file main_uartdrv.c
 * @brief UART Flow Control Example
 * @author Silicon Labs
 * @version  1.04

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include <stdint.h>
#include <string.h>
#include <stdbool.h>
#include "em_device.h"
#include "em_chip.h"
#include "em_cmu.h"
#include "em_gpio.h"
#include "em_emu.h"
#include "em_rtc.h"
#include "em_rtcc.h"
#include "gpiointerrupt.h"
#include "em_usart.h"
#include "uartdrv_flow_control.h"

// This is the actual size of the receive buffer
#define REC_BUFFER_SIZE 30

// This is the number of elements before we activate flow control
#define REC_BUFFER_THRESHOLD 20

// The buffer used to store incoming data
uint8_t recBuffer[REC_BUFFER_SIZE];

#define STRING_SIZE 52
uint8_t sendBuffer[STRING_SIZE] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

// Flag to see if it is time to send a new message
volatile bool sendEvent = false;

// Select the flow control type
volatile uint8_t fcDemo = DEMO_NONE;
volatile uint8_t fcSelected = false;

#if defined(TFT_LCD)
DISPLAY_Device_t displayDevice;    // Display device handle.
#endif

// Define receive/transmit operation queues
DEFINE_BUF_QUEUE(EMDRV_UARTDRV_MAX_CONCURRENT_RX_BUFS, rxBufferQueue);
DEFINE_BUF_QUEUE(EMDRV_UARTDRV_MAX_CONCURRENT_TX_BUFS, txBufferQueue);

UARTDRV_HandleData_t handleData;
UARTDRV_Handle_t handle = &handleData;

// Configuration for USART
UARTDRV_InitUart_t initData =
{
  USART_DEV,
  USART_BAUDRATE,
#if defined( _USART_ROUTELOC0_MASK )
  USART_TX_LOCATION,
  USART_RX_LOCATION,
#else
  USART_LOCATION,
#endif
  usartStopbits1,
  usartNoParity,
  usartOVS16,
#if defined(USART_CTRL_MVDIS)
  false,
#endif
  // Newer devices, such as EFR32MG1 and EFM32PG1, natively support CTS/RTS
  // in the USART peripheral hardware
  uartdrvFlowControlNone,
  USART_CTS_PORT,
  USART_CTS_PIN,
  USART_RTS_PORT,
  USART_RTS_PIN,
  (UARTDRV_Buffer_FifoQueue_t *)&rxBufferQueue,
  (UARTDRV_Buffer_FifoQueue_t *)&txBufferQueue,
#if defined( _USART_ROUTELOC1_MASK )
  USART_CTS_LOCATION,
  USART_RTS_LOCATION
#endif
};

// Demo SW flow control and natively supported hardware flow control for newer
// devices, such as EFR32MG1 and EFM32PG1
// Demo SW flow control and hardware flow control implemented by firmware for
// all of serial 0 devices.
#if defined(USART_ROUTEPEN_RTSPEN) && defined(USART_ROUTEPEN_CTSPEN)
int32_t demoIterm[2] = {uartdrvFlowControlSw, uartdrvFlowControlHwUart};
#else
int32_t demoIterm[2] = {uartdrvFlowControlSw, uartdrvFlowControlHw};
#endif

static const char demoString[NUMBER_OF_DEMOS][LCD_STR_LEN] =
{
#if defined(SEGMENT_LCD)
  "SW FC",
  "HW FC"
#elif defined(TFT_LCD)
  "SW Flow  \n  Control",
  "HW Flow  \n  Control"
#endif
};

#if defined(RTCC_COUNT) && (RTCC_COUNT == 1)

void RTCC_IRQHandler(void)
{
  // Clear the flag
  RTCC_IntClear(RTCC_IFC_CC1);

  // Request a new message to be sent
  sendEvent = true;
}

/***************************************************************************//**
 * @brief Configures the RTCC. The RTCC in this example is only used to
 *        send messages periodically
 *****************************************************************************/
void initRTCC(void)
{
  RTCC_Init_TypeDef rtccInit = RTCC_INIT_DEFAULT;
  rtccInit.presc = rtccCntPresc_1;

  // Enable LE domain registers
  CMU_ClockEnable(cmuClock_CORELE, true);

  // Enable LF(A|E)CLK in CMU.
  CMU_ClockSelectSet(cmuClock_LFE, cmuSelect_LFXO);

  // Enable RTCC clock
  CMU_ClockEnable(cmuClock_RTCC, true);

  // Do not start RTC until initialization is complete.
  rtccInit.enable   = false;

  // Halt RTC when debugging.
  rtccInit.debugRun = false;
  // Wrap around on CCV1 match.
  rtccInit.cntWrapOnCCV1 = true;
  RTCC_Init(&rtccInit);

  // Interrupt at given frequency.
  RTCC_CCChConf_TypeDef ccchConf = RTCC_CH_INIT_COMPARE_DEFAULT;
  ccchConf.compMatchOutAction = rtccCompMatchOutActionToggle;
  RTCC_ChannelInit(1, &ccchConf);
  RTCC_ChannelCCVSet(1, CMU_ClockFreqGet(cmuClock_RTCC) - 1);

  // Enable interrupt
  NVIC_EnableIRQ(RTCC_IRQn);
  RTCC_IntEnable(RTCC_IEN_CC1);

  RTCC_CounterSet(_RTCC_CNT_RESETVALUE);
  // Start Counter
  RTCC_Enable(true);
}

#else

void RTC_IRQHandler(void)
{
  // Clear the flag
  RTC_IntClear(RTC_IF_COMP0);

  // Request a new message to be sent
  sendEvent = true;
}

/***************************************************************************//**
 * @brief Configures the RTC. The RTC in this example is only used to
 *        send messages periodically
 *****************************************************************************/
void initRTC(void)
{
  // The CORELE clock must be enabled before the CPU
  // can access the LF peripherals
  CMU_ClockEnable(cmuClock_CORELE, true);

  // Enable the LFXO for RTC
  CMU_ClockSelectSet(cmuClock_LFA, cmuSelect_LFXO);

  // Prescale the RTC
  CMU_ClockDivSet(cmuClock_RTC, cmuClkDiv_512);

  // Enable the clock for the RTC
  CMU_ClockEnable(cmuClock_RTC, true);

  // Configure RTC to wrap around on COMP0
  RTC_Init_TypeDef rtcInit = RTC_INIT_DEFAULT;
  rtcInit.comp0Top = true;
  rtcInit.enable = false;
  RTC_Init(&rtcInit);

  // Overflow every second
  RTC_CompareSet(0, (CMU_ClockFreqGet(cmuClock_RTC) - 1));

  // Enable interrupt
  RTC_IntEnable(RTC_IEN_COMP0);
  NVIC_EnableIRQ(RTC_IRQn);

  // Start the RTC
  RTC_Enable(true);
}
#endif

/***************************************************************************//**
 * @brief  Process the received data.
 *
 * @details
 *   Dummy function to pretend that some processing is done.
 *   This only included to demonstrate the flow control.
 *****************************************************************************/
static void processRecBuffer(UARTDRV_Handle_t handle, Ecode_t transferStatus, uint8_t *data, UARTDRV_Count_t transferCount)
{
  uint32_t i;

  // XOFF XON checking for software flow control
  if ((handle->fcType == uartdrvFlowControlSw)
      && (transferStatus == ECODE_EMDRV_UARTDRV_OK)
      && (transferCount == 1))
  {
    if (data[0] == UARTDRV_FC_SW_XOFF)
    {
      // pause the transmitter
      UARTDRV_PauseTransmit(handle);
    }
    else if (data[0] == UARTDRV_FC_SW_XON)
    {
      // resume the transmitter
      UARTDRV_ResumeTransmit(handle);
    }
    // Re-start to monitor buffers for XON XOFF checking during transmitting.
    UARTDRV_Receive(handle, recBuffer, 1, processRecBuffer);
  }
  else
  {
    // Pretend that we have to do a large processing
    for (i = 0; i < 1000; i++);
  }
}

/***************************************************************************//**
 * @brief  Callback when a transfer completion.
 ******************************************************************************/
static void processTxDone(UARTDRV_Handle_t handle,
                          Ecode_t transferStatus,
                          uint8_t *data,
                          UARTDRV_Count_t transferCount)
{
  (void)handle;
  (void)transferStatus;
  (void)data;
  (void)transferCount;
}

/**************************************************************************//**
 * @brief  Gpio callback
 * @param  pin - pin which triggered interrupt
 *****************************************************************************/
void gpioCallBack(uint8_t pin)
{
  if (pin == PB0_PIN)
  {
    fcSelected = true;
  }
  else if (pin == PB1_PIN)
  {
    fcDemo = (fcDemo + 1) % NUMBER_OF_DEMOS;
  }
}

/**************************************************************************//**
 * @brief  Configure two buttons and enable interrupt
 *****************************************************************************/
void configButtonGPIO(void)
{
  // Enable clock for GPIO module, initialize GPIOINT
  CMU_ClockEnable(cmuClock_GPIO, true);

  // Configure PB0 as input and enable interrupt.
  GPIO_PinModeSet(PB0_PORT, PB0_PIN, gpioModeInputPull, 1);
  GPIO_IntConfig(PB0_PORT, PB0_PIN, false, true, true);

  // Configure PB1 as input and enable interrupt.
  GPIO_PinModeSet(PB1_PORT, PB1_PIN, gpioModeInputPull, 1);
  GPIO_IntConfig(PB1_PORT, PB1_PIN, false, true, true);

  GPIOINT_Init();

  // Register callback functions and enable interrupts
  GPIOINT_CallbackRegister(PB0_PIN, gpioCallBack);
  GPIOINT_CallbackRegister(PB1_PIN, gpioCallBack);
}


/**************************************************************************//**
 * @brief Selecting flow control demos.
 *****************************************************************************/
void selectFcDemo(void)
{
  static uint32_t prevDemo = NUMBER_OF_DEMOS - 1;

  // Setup GPIO for pushbuttons.
  configButtonGPIO();

  while (fcSelected == false)
  {
    if (prevDemo != fcDemo)
    {
      prevDemo = fcDemo;
      displayString(demoString[prevDemo]);
    }
  }

  // Unregisters user callback after selecting demo mode done.
  GPIOINT_CallbackUnRegister(PB0_PIN);
  GPIOINT_CallbackUnRegister(PB1_PIN);
}

/**************************************************************************//**
 * @brief Main function
 * The main loop will stay in EM1 and wait for interrupts to occur.
 * This will be either:
 *  - An RTC interrupt indicating that it is time to send a new message
 *  - An RX interrupt indicating that a new byte has been received
 *  - A TXBL interrupt indicating that a new byte should be sent
 *    (handled by the flow control routine)
 *  - A TXC interrupt indicating that the full message has been sent
 *****************************************************************************/
int main(void)
{
  // Initialize chip - handle erratas
  CHIP_Init();

#if defined( _EMU_DCDCCTRL_MASK )
  // Init DCDC regulator
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif

  // A crystal is usually needed for UART communication
  CMU_ClockSelectSet(cmuClock_HF, cmuSelect_HFXO);

  // Initialize display module
  displayInit();

  // Selecting flow control demos
  selectFcDemo();

  initData.fcType = (UARTDRV_FlowControlType_t)(demoIterm[fcDemo]);
  // Initialize driver handle
  UARTDRV_InitUart(handle, &initData);

#if defined(RTCC_COUNT) && (RTCC_COUNT == 1)
  initRTCC();
#else
  initRTC();
#endif

  while (1)
  {
    // UART hardware flow control (CTS/RTS) is fully supported by the driver.
    // UART software flow control (XON/XOFF) is partially supported by the driver.
    // The application must monitor buffers for XON XOFF checking during transmitting.
    if (sendEvent == true)
    {
      if (initData.fcType == uartdrvFlowControlSw)
      {
        UARTDRV_Receive(handle, recBuffer, 1, processRecBuffer);
      }
      // Note: Before starting the UART DMA transmit operation, UARTDRV should
      // check the CTS status. The mechanism should be included in the latest UARTDRV.
      UARTDRV_Transmit(handle, sendBuffer, STRING_SIZE, processTxDone);
      while (handle->txDmaActive)
      {
        EMU_EnterEM1();
      }
      sendEvent = false;
    }

    // Abort the ongoing XON XOFF checking.
    UARTDRV_Abort(handle, uartdrvAbortReceive);

    // Receive data using a non-blocking receive function
    UARTDRV_Receive(handle, recBuffer, REC_BUFFER_THRESHOLD, processRecBuffer);
    while (handle->rxDmaActive)
    {
      EMU_EnterEM1();
    }
  }
}
