/**************************************************************************//**
 * @file uart_flow_control_s0.h
 * @brief UART Flow Control Header File
 * @author Silicon Labs
 * @version  1.04

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#ifndef __UART_FLOW_CONTROL_S0_H__
#define __UART_FLOW_CONTROL_S0_H__

#include "em_gpio.h"
#include "bspconfig.h"

// UART flow control demo, includes hardware flow control, software flow control, legacy hardware master, legacy hardware slave

#if (defined EFM32GG_STK3700) || (defined EFM32_Gxxx_STK) \
    || (defined EFM32LG_STK3600) || (defined EFM32WG_STK3800)
#define PB0_PORT   gpioPortB               // Pushbutton 0 port.
#define PB0_PIN    9                       // Pushbutton 0 pin.
#define PB1_PORT   gpioPortB               // Pushbutton 1 port.
#define PB1_PIN    10                      // Pushbutton 1 pin.

#elif (defined SLSTK3400A_EFM32HG) || (defined EFM32ZG_STK3200)
#define PB0_PORT   BSP_GPIO_PB0_PORT       // Pushbutton 0 port.
#define PB0_PIN    BSP_GPIO_PB0_PIN        // Pushbutton 0 pin.
#define PB1_PORT   BSP_GPIO_PB1_PORT       // Pushbutton 1 port.
#define PB1_PIN    BSP_GPIO_PB1_PIN        // Pushbutton 1 pin.

#elif defined (EFM32TG_STK3300)
#define PB0_PORT   gpioPortD               // Pushbutton 0 port.
#define PB0_PIN    8                       // Pushbutton 0 pin.
#define PB1_PORT   gpioPortB               // Pushbutton 1 port.
#define PB1_PIN    11                      // Pushbutton 1 pin.
#endif

extern volatile uint8_t fcDemo;
extern volatile uint8_t fcSelected;

#if defined ( SLSTK3400A_EFM32HG )
#define USART_DEV       USART0
#define USART_CLOCK     cmuClock_USART0
#define USART_TX_IRQn   USART0_TX_IRQn
#define USART_RX_IRQn   USART0_RX_IRQn
#define USART_TX_IRQHANDLER  USART0_TX_IRQHandler
#define USART_RX_IRQHANDLER  USART0_RX_IRQHandler
#else
#define USART_DEV       USART1
#define USART_CLOCK     cmuClock_USART1
#define USART_TX_IRQn   USART1_TX_IRQn
#define USART_RX_IRQn   USART1_RX_IRQn
#define USART_TX_IRQHANDLER  USART1_TX_IRQHandler
#define USART_RX_IRQHANDLER  USART1_RX_IRQHandler
#endif

#if defined ( SLSTK3400A_EFM32HG )
#define USART_TX_PORT   gpioPortE
#define USART_TX_PIN    10
#define USART_RX_PORT   gpioPortE
#define USART_RX_PIN    11
#define USART_LOCATION  USART_ROUTE_LOCATION_LOC0
#define USART_RTS_PORT  gpioPortE
#define USART_RTS_PIN   12
#define USART_CTS_PORT  gpioPortE
#define USART_CTS_PIN   13
#elif defined ( EFM32ZG_STK3200 )
#define USART_TX_PORT   gpioPortD
#define USART_TX_PIN    7
#define USART_RX_PORT   gpioPortD
#define USART_RX_PIN    6
#define USART_LOCATION  USART_ROUTE_LOCATION_LOC3
#define USART_RTS_PORT  gpioPortC
#define USART_RTS_PIN   15
#define USART_CTS_PORT  gpioPortC
#define USART_CTS_PIN   14
#else
#define USART_TX_PORT   gpioPortD
#define USART_TX_PIN    0
#define USART_RX_PORT   gpioPortD
#define USART_RX_PIN    1
#define USART_LOCATION  USART_ROUTE_LOCATION_LOC1
#define USART_RTS_PORT  gpioPortD
#define USART_RTS_PIN   2
#define USART_CTS_PORT  gpioPortD
#define USART_CTS_PIN   3
#endif


#define DEMO_NONE  0
#define NUMBER_OF_DEMOS   4

#if defined(SEGMENT_LCD)
#include "segmentlcd.h"
#define displayInit()   SegmentLCD_Init(false)
#define displayString(x)  SegmentLCD_Write(x)
#define displayOff()    SegmentLCD_Disable()
#define LCD_STR_LEN       7
#elif defined(TFT_LCD)
#include "display.h"
#include "textdisplay.h"
#include "retargettextdisplay.h"
#include <stdio.h>
extern DISPLAY_Device_t displayDevice;

#define displayInit()   DISPLAY_Init(); RETARGET_TextDisplayInit()
#define displayString(x)  printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100); \
  printf("\n  Flow Control" \
         "\n  Demo" \
         "\n\n\n  Push BTN1 to\n" \
         "  cycle modes\n" \
         "\n  Push BTN0 to\n" \
         "  start test \n\n\n\n"); \
  printf("  %s",x)
#define displayOff() DISPLAY_DeviceGet( 0, &displayDevice ); \
  displayDevice.pDisplayPowerOn(&displayDevice, false)
#define LCD_STR_LEN       20
#endif

__STATIC_INLINE void enableUsartTxblInt(void)
{
  // Enable TXBL interrupt
  USART_DEV->IEN |= USART_IEN_TXBL;
}

__STATIC_INLINE void disableUsartTxblInt(void)
{
  // Disable TXBL interrupt
  USART_DEV->IEN &= ~USART_IEN_TXBL;
}

typedef enum
{
  // Signal is low when asserted
  activeLow = 0,

  // Signal is high when asserted
  activeHigh = 1

} UART_FC_Polarity;

typedef enum
{
  modeHW,
  modeSW,
  modeLegacyHW_Master,
  modeLegacyHW_Slave
} UART_FC_Mode;

typedef struct _USART_FC_Pin
{
  GPIO_Port_TypeDef port;
  unsigned int      pin;
} UART_FC_Pin;

typedef struct
{
  // The location of the RTS pin
  UART_FC_Pin rtsPin;

  // Active polarity of RTS pin
  UART_FC_Polarity rtsPolarity;

  // The location of the CTS pin
  UART_FC_Pin ctsPin;

  // Active polarity of CTS pin
  UART_FC_Polarity ctsPolarity;

  // Selects the type of flow control
  UART_FC_Mode mode;

  // Callback function which is called when
  // the entire message has been sent.
  // If a callback is not used, this field MUST
  // be set to NULL.
  void (*txCallback)(void);

  // Callback function which is called for every byte received.
  // This MUST be defined
  void (*rxCallback)(uint8_t);

} UART_FC_Config_TypeDef;


void uartFcInit(UART_FC_Config_TypeDef *);

bool uartFcStartTx(uint8_t *data, uint32_t numBytes);

void uartFcEnableRx(void);

void uartFcDisableRx(void);

#endif /* __UART_FLOW_CONTROL_S0_H__ */

