/**************************************************************************//**
 * @file main_uart_s1.c
 * @brief UART Flow Control Example
 * @author Silicon Labs
 * @version  1.04

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include <stdint.h>
#include <string.h>
#include <stdbool.h>
#include <stdio.h>

#include "em_device.h"
#include "em_usart.h"
#include "em_chip.h"
#include "em_cmu.h"
#include "em_gpio.h"
#include "em_emu.h"
#include "em_core.h"
#include "em_rtcc.h"
#include "display.h"
#include "textdisplay.h"
#include "retargettextdisplay.h"
#include "uart_flow_control_s1.h"

// This is the actual size of the receiving buffer
#define REC_BUFFER_SIZE 30

// This is the number of elements before we activate flow control
#define REC_BUFFER_THRESHOLD 20

#define XON  0x11
#define XOFF 0x13

// Pointer to the next character to be sent
static uint8_t *nextByte = 0;

// A message to send
char *message = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

// The buffer used to store incoming data
uint8_t recBuffer[REC_BUFFER_SIZE];

// The current index of the receive buffer
volatile uint8_t bufferIndex = 0;

// Flag to see if it is time to send a new message
volatile bool sendEvent = false;

// The flow control configuration
static UART_FC_Config_TypeDef fcConfig;

// Select the flow control type
volatile uint8_t fcDemo = DEMO_NONE;
volatile uint8_t fcSelected = false;

// Flag indicating the status of the flow control.
// If this is true we are allowed to send more data
static volatile bool txEnabled = false;

// Number of bytes left of the message. Will be zero when no message
// is pending
static volatile uint32_t remainingBytes = 0;

// Display device handle.
static DISPLAY_Device_t displayDevice;

static const char demoString[NUMBER_OF_DEMOS][LCD_STR_LEN] =
{
  "HW Flow  \n  Control",
  "SW Flow  \n  Control"
};

/**************************************************************************//**
 * @brief Callback when a byte has been received
 *****************************************************************************/
void rxHandler(uint8_t receivedByte)
{
  // Store character in buffer
  recBuffer[bufferIndex++] = receivedByte;

  if (bufferIndex > REC_BUFFER_SIZE)
  {
    // We actually ran out of space in our buffer.
    // Here we should do some fault handling. This code
    // simply halts in a while loop so it is easy for a
    // debugger to identify the problem.
    while (1);
  }
}


/**************************************************************************//**
 * @brief Setting up USART
 * Configures USART module, and flow control, and enables the RX/TX interrupt
 *****************************************************************************/
void initUART(USART_TypeDef *usart)
{
  // Set the mode to selected Flow Control mode
  fcConfig.mode = (UART_FC_Mode)fcDemo;

  // Register callback functions
  fcConfig.txCallback = NULL;
  fcConfig.rxCallback = rxHandler;

  // Enable clocks
  CMU_ClockEnable(cmuClock_HFPER, true);
  CMU_ClockEnable(cmuClock_GPIO, true);
  CMU_ClockEnable(USART_CLOCK, true);

  // Initialize the UART
  USART_InitAsync_TypeDef uartInit = USART_INITASYNC_DEFAULT;
  uartInit.enable = usartDisable;
  USART_InitAsync(usart, &uartInit);

  // Set up RX pin
  usart->ROUTELOC0 = (usart->ROUTELOC0 & (~_USART_ROUTELOC0_RXLOC_MASK))
                     | USART_RX_LOCATION;
  usart->ROUTEPEN = usart->ROUTEPEN | USART_ROUTEPEN_RXPEN;

  // Set up TX pin
  usart->ROUTELOC0 = (usart->ROUTELOC0 & (~_USART_ROUTELOC0_TXLOC_MASK))
                     | USART_TX_LOCATION;
  usart->ROUTEPEN = usart->ROUTEPEN | USART_ROUTEPEN_TXPEN;

  // Enable TX RX pin
  GPIO_PinModeSet(USART_RX_PORT, USART_RX_PIN, gpioModeInput, 0);
  GPIO_PinModeSet(USART_TX_PORT, USART_TX_PIN, gpioModePushPull, 1);

  if (fcConfig.mode == modeHWUART)
  {
    // Enable CTS and RTS pin
    GPIO_PinModeSet(USART_CTS_PORT, USART_CTS_PIN, gpioModeInput, 0);
    GPIO_PinModeSet(USART_RTS_PORT, USART_RTS_PIN, gpioModePushPull, 0);

    // Set up CTS pin
    usart->ROUTELOC1 = (usart->ROUTELOC1 & (~_USART_ROUTELOC1_CTSLOC_MASK))
                       | USART_CTS_LOCATION;
    usart->ROUTEPEN = usart->ROUTEPEN | USART_ROUTEPEN_CTSPEN;

    // CTS Function enabled
    usart->CTRLX |= USART_CTRLX_CTSEN;

    // Set up RTS pin
    usart->ROUTELOC1 = (usart->ROUTELOC1 & (~_USART_ROUTELOC1_RTSLOC_MASK))
                       | USART_RTS_LOCATION;
    usart->ROUTEPEN = usart->ROUTEPEN | USART_ROUTEPEN_RTSPEN;
  }

  txEnabled = true;

  // Enable interrupt when character arrives
  USART_IntEnable(usart, USART_IEN_RXDATAV);

  NVIC_EnableIRQ(USART_RX_IRQn);

  // Enable TX interrupt in NVIC. Note that the TXBL interrupt in USART0 is not
  // enabled here. The TXBL interrupt will be enabled and disabled based on
  // the CTS signal
  NVIC_EnableIRQ(USART_TX_IRQn);

  USART_Enable(usart, usartEnable);
}

void RTCC_IRQHandler(void)
{
  // Clear the flag
  RTCC_IntClear(RTCC_IFC_CC1);

  // Request a new message to be sent
  sendEvent = true;
}

/**************************************************************************//**
 * @brief Configures the RTCC. The RTCC in this example is only used to
 *        send messages periodically
 *****************************************************************************/
void initRTCC(void)
{
  RTCC_Init_TypeDef rtccInit = RTCC_INIT_DEFAULT;
  rtccInit.presc = rtccCntPresc_1;

  // Enable LE domain registers
  CMU_ClockEnable(cmuClock_CORELE, true);

  // Enable LF(A|E)CLK in CMU.
  CMU_ClockSelectSet(cmuClock_LFE, cmuSelect_LFXO);

  // Enable RTCC clock
  CMU_ClockEnable(cmuClock_RTCC, true);

  // Do not start RTC until initialization is complete.
  rtccInit.enable   = false;

  // Halt RTC when debugging.
  rtccInit.debugRun = false;
  // Wrap around on CCV1 match.
  rtccInit.cntWrapOnCCV1 = true;
  RTCC_Init(&rtccInit);

  // Interrupt at given frequency.
  RTCC_CCChConf_TypeDef ccchConf = RTCC_CH_INIT_COMPARE_DEFAULT;
  ccchConf.compMatchOutAction = rtccCompMatchOutActionToggle;
  RTCC_ChannelInit(1, &ccchConf);
  RTCC_ChannelCCVSet(1, CMU_ClockFreqGet(cmuClock_RTCC) - 1);

  // Enable interrupt
  NVIC_EnableIRQ(RTCC_IRQn);
  RTCC_IntEnable(RTCC_IEN_CC1);

  RTCC_CounterSet(_RTCC_CNT_RESETVALUE);
  // Start Counter
  RTCC_Enable(true);
}

/**************************************************************************//**
 * @brief Sends the software flow control signal XON
 *****************************************************************************/
static void sendXON(USART_TypeDef *usart)
{
  CORE_DECLARE_IRQ_STATE;
  // Disable TXBL interrupt in case there already is an
  // ongoing transmission.
  USART_IntDisable(usart, USART_IEN_TXBL);

  // Wait until TX buffer is ready
  while (!(usart->STATUS & USART_STATUS_TXBL));

  // Send signal
  USART_Tx(usart, XON);

  // Enable TXBL interrupt again
  CORE_ENTER_ATOMIC();
  if (txEnabled && remainingBytes > 0)
  {
    USART_IntEnable(usart, USART_IEN_TXBL);
  }
  CORE_EXIT_ATOMIC();
}

/**************************************************************************//**
 * @brief Sends the software flow control signal XOFF
 *****************************************************************************/
static void sendXOFF(USART_TypeDef *usart)
{
  CORE_DECLARE_IRQ_STATE;
  // Disable TXBL interrupt in case there already is an
  // ongoing transmission.
  USART_IntDisable(usart, USART_IEN_TXBL);

  // Wait until TX buffer is ready
  while (!(usart->STATUS & USART_STATUS_TXBL));

  // Send signal
  USART_Tx(usart, XOFF);

  // Enable TXBL interrupt again
  CORE_ENTER_ATOMIC();
  if (txEnabled && remainingBytes > 0)
  {
    USART_IntEnable(usart, USART_IEN_TXBL);
  }
  CORE_EXIT_ATOMIC();
}

/**************************************************************************//**
 * @brief Indicate that we are ready to receive data.
 *****************************************************************************/
void uartFcEnableRx(void)
{
  if (fcConfig.mode == modeSW)
  {
    sendXON(USART_DEV);
  }
}

/**************************************************************************//**
 * @brief Indicate that we are NOT ready to receive more data.
 *****************************************************************************/
void uartFcDisableRx(void)
{
  if (fcConfig.mode == modeSW)
  {
    sendXOFF(USART_DEV);
  }
}

/**************************************************************************//**
 * @brief Process the received data.
 *
 * @details
 *   Dummy function to pretend that some processing is done.
 *   This only included to demonstrate the flow control.
 *****************************************************************************/
void processBuffer(void)
{
  uint32_t i;

  // For SW flow control, send XOFF to disable reception.
  // For HWUART flow control, USART module will deassert RTS on RX buffer full,
  // space is still available in the receive shift register for one more frame.

  uartFcDisableRx();

  // Pretend that we have to do a large processing
  for (i = 0; i < 1000; i++);

  // Empty the buffer
  bufferIndex = 0;

  // For SW flow control, send XON to Enable reception again.
  uartFcEnableRx();
}

/**************************************************************************//**
 * @brief Handler for the TXBL interrupt.
 *****************************************************************************/
void USART_TX_IRQHANDLER(void)
{
  uint32_t intFlags;

  // Clear the flag(s)
  intFlags = USART_IntGet(USART_DEV);
  USART_IntClear(USART_DEV, intFlags);

  if ((intFlags & USART_IF_TXC) && (remainingBytes == 0))
  {
    // Disable TXC interrupt
    USART_IntDisable(USART_DEV, USART_IEN_TXC);
    NVIC_ClearPendingIRQ(USART_TX_IRQn);

    // Execute callback function
    if (fcConfig.txCallback != NULL)
    {
      fcConfig.txCallback();
    }
  }
  else if (intFlags & USART_IF_TXBL)
  {
    if (remainingBytes > 0)
    {
      // Send next byte and increment pointer
      USART_DEV->TXDATA = *nextByte++;

      if (--remainingBytes == 0)
      {
        // Disable TXBL and enable TXC
        USART_IntDisable(USART_DEV, USART_IEN_TXBL);
        NVIC_ClearPendingIRQ(USART_TX_IRQn);
        USART_IntEnable(USART_DEV, USART_IEN_TXC);

      }
    }
  }
}

/**************************************************************************//**
 * @brief Handler for the RX interrupt.
 *****************************************************************************/
void USART_RX_IRQHANDLER(void)
{
  uint8_t rxData;
  uint32_t intFlags;

  // Clear the flag(s)
  intFlags = USART_IntGet(USART_DEV);
  USART_IntClear(USART_DEV, intFlags);

  rxData = USART_DEV->RXDATA;

  // If the mode is SW, check first if the byte was one of the
  // flow control characters
  if (fcConfig.mode == modeSW && rxData == XON)
  {
    // XON received, enable TX
    txEnabled = true;
    if (remainingBytes > 0)
    {
      USART_IntEnable(USART_DEV, USART_IEN_TXBL);
    }
  }
  else if (fcConfig.mode == modeSW && rxData == XOFF)
  {
    // XOFF received, disable TX
    txEnabled = false;
    USART_IntDisable(USART_DEV, USART_IEN_TXBL);
  }
  else
  {
    // Call the callback function with the received byte
    if (fcConfig.rxCallback != NULL)
    {
      fcConfig.rxCallback(rxData);
    }
  }
}

/**************************************************************************//**
 * @brief Start sending a message with flow control
 *
 * @details
 *   Will start sending a message with flow control. The method returns immediately
 *   and uses interrupts to send the actual message. If a previous transmission is
 *   incomplete this method will not start a new transmission and the method will
 *   return false.
 *
 * @returns True if the transmission was started.
 *****************************************************************************/
bool uartFcStartTx(uint8_t *data, uint32_t numBytes)
{
  CORE_DECLARE_IRQ_STATE;
  // Wait until previous transfer is complete
  if (remainingBytes > 0)
  {
    return false;
  }

  // Store location and size of the message
  nextByte = data;
  remainingBytes = numBytes;

  // Enable TXBL interrupt if allowed by flow control
  CORE_ENTER_ATOMIC();
  // When using HWUART flow control,  transmission always be controlled by
  // the USART module
  if ((fcConfig.mode == modeHWUART) || ((fcConfig.mode == modeSW) && txEnabled))
  {
    // Enable TXBL interrupt
    enableUsartTxblInt();
  }
  CORE_EXIT_ATOMIC();

  return true;
}

/***************************************************************************//**
 * @brief Common IRQ handler for the GPIO interrupts (pushbuttons)
 *        PB0 Starts selected flow control test.
 *        PB1 Cycles through the available tests.
 *****************************************************************************/
void GPIO_COMMON_IRQHandler(void)
{
  // Get and clear all pending GPIO interrupts
  uint32_t interruptMask;
  interruptMask = GPIO_IntGet();
  GPIO_IntClear(interruptMask);

  if ((interruptMask & (1 << PB0_PIN)) || (interruptMask & (1 << PB1_PIN)))
  {
    // Act on interrupts
    if (interruptMask & (1 << PB0_PIN))
    {
      fcSelected = true;
    }
    if (interruptMask & (1 << PB1_PIN))
    {
      fcDemo = (fcDemo + 1) % NUMBER_OF_DEMOS;
    }
  }
}

/***************************************************************************//**
 * @brief GPIO Interrupt handler for odd pins
 ******************************************************************************/
void GPIO_ODD_IRQHandler(void)
{
  GPIO_COMMON_IRQHandler();
}

/***************************************************************************//**
 * @brief GPIO Interrupt handler for even pins
 ******************************************************************************/
void GPIO_EVEN_IRQHandler(void)
{
  GPIO_COMMON_IRQHandler();
}

/**************************************************************************//**
 * @brief Configure two pushbuttons and enable interrupt
 *****************************************************************************/
void configButtonGPIO(void)
{
  // Enable GPIO clock.
  CMU_ClockEnable(cmuClock_GPIO, true);

  // Configure PB0 as input and enable interrupt.
  GPIO_PinModeSet(PB0_PORT, PB0_PIN, gpioModeInputPull, 1);
  GPIO_IntConfig(PB0_PORT, PB0_PIN, false, true, true);

  // Configure PB1 as input and enable interrupt.
  GPIO_PinModeSet(PB1_PORT, PB1_PIN, gpioModeInputPull, 1);
  GPIO_IntConfig(PB1_PORT, PB1_PIN, false, true, true);

  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
}

/**************************************************************************//**
 * @brief Selecting flow control demos. The HWUART and SW flow control.
 *****************************************************************************/
void selectFcDemo(void)
{
  static uint32_t prevDemo = NUMBER_OF_DEMOS - 1;

  // Setup GPIO for pushbuttons.
  configButtonGPIO();

  printf("\n  Flow Control"
         "\n  Demo"
         "\n\n\n  Push BTN1 to\n"
         "  cycle modes\n"
         "\n  Push BTN0 to\n"
         "  start test \n\n");

  while (fcSelected == false)
  {
    if (prevDemo != fcDemo)
    {
      prevDemo = fcDemo;
      printf("\n\n  ");
      printf(demoString[prevDemo]);
      printf(TEXTDISPLAY_ESC_SEQ_CURSOR_HOME_VT100);
      printf("\n\n\n\n\n\n\n\n\n\n\n");
    }
  }

  // Disable GPIO interrupt after selecting demo mode done.
  NVIC_DisableIRQ(GPIO_ODD_IRQn);
  NVIC_DisableIRQ(GPIO_EVEN_IRQn);
  GPIO_PinModeSet(PB0_PORT, PB0_PIN, gpioModeDisabled, 1);
  GPIO_PinModeSet(PB1_PORT, PB1_PIN, gpioModeDisabled, 1);
}

/***************************************************************************//**
 * @brief Main function
 * The main loop will stay in EM1 and wait for interrupts to occur.
 * This will be either:
 *  - An RTCC interrupt indicating that it is time to send a new message
 *  - An RX interrupt indicating that a new byte has been received
 *  - A TXBL interrupt indicating that a new byte should be sent
 *    (handled by the flow control routine)
 *  - A TXC interrupt indicating that the full message has been sent
 ******************************************************************************/
int main(void)
{
  // Initialize chip - handle erratas
  CHIP_Init();

#if defined( _EMU_DCDCCTRL_MASK )
  // Init DCDC regulator
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif

  // A crystal is usually needed for UART communication
  CMU_ClockSelectSet(cmuClock_HF, cmuSelect_HFXO);

  // Initialize display module
  DISPLAY_Init();
  // Retrieve the properties of the display.
  if (DISPLAY_DeviceGet(0, &displayDevice) != DISPLAY_EMSTATUS_OK)
  {
    // Unable to get display handle.
    EFM_ASSERT(false);
  }
  // Retarget stdio to the display.
  if (TEXTDISPLAY_EMSTATUS_OK != RETARGET_TextDisplayInit())
  {
    // Text display initialization failed.
    EFM_ASSERT(false);
  }

  // Selecting flow control demos
  selectFcDemo();

  initUART(USART_DEV);
  initRTCC();

  while (1)
  {

    // Stay in sleep until something happens
    EMU_EnterEM1();

    // It is time to send a message
    if (sendEvent)
    {
      uartFcStartTx((uint8_t*)message, strlen(message));
      sendEvent = false;
    }

    // The receiving buffer is almost filled up
    if (bufferIndex > REC_BUFFER_THRESHOLD)
    {
      // Do some processing. This will temporarily halt the transmission
      processBuffer();
    }
  }
}
