/**************************************************************************//**
 * @file fortuna_adc.c
 * @brief Interface to ADC in order to collect entropy from ADC.
 * @author Energy Micro AS
 * @version 1.02
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2013 Energy Micro AS, http://www.energymicro.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 * 4. The source and compiled code may only be used on Energy Micro "EFM32"
 *    microcontrollers and "EFR4" radios.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Energy Micro AS has no
 * obligation to support this Software. Energy Micro AS is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Energy Micro AS will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 *****************************************************************************/
#include <stdint.h>
#include <stdbool.h>

/* EMLIB headers */
#include "em_device.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "em_adc.h"

/* Fortuna headers */
#include "fortuna_adc.h"

/* Global variables for ADC */
static volatile bool  adcFinished;


/**************************************************************************//**
 * @brief   ADC interrupt Handler. Clears interrupt and then indicates data is
 *          ready by setting adcFinished to true.
 *****************************************************************************/
void ADC0_IRQHandler(void)
{
  /* Acknowledge interrupt. */
  ADC_IntClear(ADC0, ADC_IF_SINGLE);

  /*Indicate data is ready. */
  adcFinished = true;
}


/**************************************************************************//**
 * @brief   Configure ADC for sampling the temperature sensor.
 *****************************************************************************/
void AdcRandomInitialize(void)
{
  ADC_Init_TypeDef       init       = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  /* Enable ADC clock. */
  CMU_ClockEnable(cmuClock_ADC0, true);

  init.timebase = ADC_TimebaseCalc(0);

  /* Set ADC clock to 4 MHz, use default HFPERCLK */
  init.prescale = ADC_PrescaleCalc(4000000, 0);

  ADC_Init(ADC0, &init);

  /* Set input to temperature sensor. Reference must be 1.25V. */
  singleInit.reference = adcRef1V25;
  singleInit.input     = adcSingleInpTemp;
  singleInit.resolution = adcRes12Bit;
  singleInit.acqTime = adcAcqTime1;
  ADC_InitSingle(ADC0, &singleInit);

  /* Generate interrupt when data conversion is completed. */
  ADC_IntClear(ADC0, ADC_IFC_SINGLE);
  ADC_IntEnable(ADC0, ADC_IEN_SINGLE);
  NVIC_EnableIRQ(ADC0_IRQn);
}


/**************************************************************************//**
 * @brief Get a sample from the ADC.
 *****************************************************************************/
uint32_t AdcSampleGet(void)
{
  uint32_t  adcData;          /* Sample from ADC. */
  
  /* Clear flag which indicates whether the ADC irq handler has been called
     and ADC data is ready. */
  adcFinished = false;
  
  /* Trigger ADC conversion*/
  ADC_Start(ADC0, adcStartSingle);
  
  /* Stay in EM1 Energy mode until ADC finishes conversion. */
  while (!adcFinished)
  {
    /* Only ADC interrupt wakes-up EFM32 */
    EMU_EnterEM1();
  }
  
  /* ADC has finished conversion if it reached here. */
  adcData = ADC_DataSingleGet(ADC0);

  return adcData;
}
