/**************************************************************************//**
 * @file fortuna.h
 * @brief Fortuna Pseudo Random Number Generator (PRNG) public API for EFM32.
 * @author Energy Micro AS
 * @version 1.02
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2013 Energy Micro AS, http://www.energymicro.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 * 4. The source and compiled code may only be used on Energy Micro "EFM32"
 *    microcontrollers and "EFR4" radios.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Energy Micro AS has no
 * obligation to support this Software. Energy Micro AS is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Energy Micro AS will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 *****************************************************************************/
#ifndef __FORTUNA_H
#define __FORTUNA_H

/*
 *  Fortuna user configuration definitions:
 */

#define NUM_POOLS                         (32)  /* Number of pools used for
                                                   entropy acquisition.
                                                   Max is 32 pools. */
#define MAX_ENTROPY_SOURCES                (1)  /* Maximum number of entropy
                                                   sources allowed to feed
                                                   Fortuna with entropy via
                                                   the FORTUNA_AddRandomEvent
                                                   function. */
#define RESEED_PERIOD_TICKS              (100)  /* Minimum time between
                                                   reseed events in ticks. */
#define RESEED_LIMIT               (1024*1024)  /* Maximum number of pseudo
                                                   random data bytes between
                                                   reseed events. */
#define SEED_FILE_UPDATE_PERIOD_TICKS  (10000)  /* Minimum time between seed
                                                   file updates in ticks. */
#define SEED_FILE_FLASH_ADDRESS (FLASH_BASE+FLASH_SIZE-FLASH_PAGE_SIZE)
                                                /* Flash address where to
                                                   store the see file.
                                                   The user should make sure
                                                   the selected page is not
                                                   used for other purposes
                                                   in the application. */


/*
 *  Fortuna interface definitions:
 */

#define FORTUNA_MAX_DATA_SIZE   (1024*1024)     /* Maximum number of random data
                                                   bytes that can be requested
                                                   when calling the
                                                   FORTUNA_RandomDataGet
                                                   function. */
/*
 * Fortuna error codes:
 */
#define FORTUNA_OK                    (0)  /* Status code that indicates
                                              successfull operation. */
#define FORTUNA_NOT_ENOUGH_ENTROPY   (-1)  /* Fortuna is not ready to produce
                                              random data because it has not
                                              accumulated enough entropy. The
                                              user should make sure that the
                                              entropy sources add entropy via
                                              the FORTUNA_AddRandomEvent
                                              function. */
#define FORTUNA_INVALID_SOURCE_ID    (-2)  /* The entropy source id is invalid.
                                              i.e. outside the range
                                              0..MAX_ENTROPY_SOURCES */
#define FORTUNA_INVALID_DATA_SIZE    (-3)  /* The requested random data size is
                                              higher than FORTUNA_MAX_DATA_SIZE.
                                           */
#define FORTUNA_INVALID_SEED_FILE    (-4)  /* The seed file in flash memory is
                                              invalid. I.e. the checksum does
                                              not match the hash of the content.
                                           */
#define FORTUNA_FLASH_WRITE_ERROR    (-5)  /* Failed to write to flash memory.*/
#define FORTUNA_FLASH_ERASE_ERROR    (-6)  /* Failed to erase flash memory.*/

/*
 *  Fortuna interface functions:
 */

int    FORTUNA_Init (void);

int    FORTUNA_AddRandomEvent (int       sourceId,
                               uint32_t  eventData);

int    FORTUNA_RandomDataGet  (void* randomData,
                               int numBytes);

int    FORTUNA_Rand (uint32_t* pseudoRandomNumber);

void   FORTUNA_Shutdown (void);


/*
 *  Interface to system functions:
 */

/* Return a tick counter which is assumed to wrap around at the maximum value
   MAX_SYS_TICK_VAL. */
#define MAX_SYS_TICK_VAL  (0xFFFFFFFF)
uint32_t SysTickGet(void);


#endif /*__FORTUNA_H*/
