/**************************************************************************//**
 * @file adc.c
 * @brief Functions to initialize and use the ADC to measure back-EMF and motor current
 * @author Silicon Labs
 * @version 0.00 (leave as is with x.xx, Correct version is automatically inserted by auto-generation)
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2014 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include <string.h>
#include "em_device.h"
#include "em_cmu.h"
#include "em_timer.h"
#include "em_prs.h"
#include "em_adc.h"
#include "config.h"
#include "motor.h"
#include "logging.h"
#include "adc.h"

/* Reference voltage */
#define ADC_REF 2.5f

/* Current PWM duty cycle. It is used to calculate the 
 * time-averaged motor current. */
extern int currentPwm;

/* Flag telling if the motor is currently running */
extern bool isRunning; 
   
/* A circiular buffer object to keep track of an 
 * average value */
#define BUF_LEN (6 * 10)
typedef struct _bufAvg
{
  int buf[BUF_LEN];
  int index;
  int sum;
} BufAvg;
BufAvg currentBuf;


/**********************************************************
 * Uses a circular buffer to keep track of an average value.
 * 
 * @param b
 *   Buffer object
 * 
 * @param v
 *   New value to insert into the buffer
 * 
 * @returns
 *   The average of all values in the buffer
 *********************************************************/
int bufAverage(BufAvg *b, int v) 
{
  b->sum = b->sum - b->buf[b->index] + v;
  b->buf[b->index] = v;
  
  b->index = b->index + 1;
  if ( b->index >= BUF_LEN ) {
    b->index = 0;
  }
  
  return b->sum / BUF_LEN;
}

/**********************************************************
 * Stops the motor if the measured current is too high.
 * 
 * @param milliamps
 *   Measured motor current in milliamps
 *********************************************************/
static void adcCurrentMeasured(int milliamps)
{
  /* Stop motor if drawing more than the maximum configured current */
  if ( milliamps > MAX_CURRENT_MA )
  { 
    stopMotor();
  }
  
  LOG_SET_MOTOR_CURRENT((int16_t)milliamps);
}

/**********************************************************
 * ADC IRQ Handler. Called when a measurement is complete.
 *********************************************************/
void ADC0_IRQHandler(void)
{  
  ADC0->IFC = ADC_IFC_SINGLE;
  int result = ADC0->SINGLEDATA;
  
  /* Calculate the average value from the last samples */
  result = bufAverage(&currentBuf, result);
  
  /* Calculate the motor current.  */
  int milliamps = (int)((1000 * result * ADC_REF * currentPwm) / (float)(4096 * CURRENT_RESISTOR * PWM_TOP));
  
  adcCurrentMeasured(milliamps);
}

/**********************************************************
 * Init ADC trigger. The ADC is triggered via PRS
 * to perform a measurement. The PRS signal is 
 * generated by TIMER2 CC2.
 *********************************************************/
void adcInitTrigger(void)
{  
  /* Set channel 2 in compare mode. Trigger ADC measurements. 
   * ADC should trigger in the middle of the PWM on period. */
  TIMER_CompareSet(TIMER2, 2, PWM_DEFAULT_DUTY_CYCLE / 2);

  TIMER_InitCC_TypeDef initCc = TIMER_INITCC_DEFAULT;
  initCc.prsInput = false;
  initCc.mode = timerCCModeCompare;
  TIMER_InitCC(TIMER2, 2, &initCc);
}


/**********************************************************
 * Initialize the ADC to do current measurements. 
 *********************************************************/
void adcInit(void)
{
  /* Clear the average buffer */
  memset(&currentBuf, 0, sizeof(currentBuf));
  
  CMU_ClockEnable(cmuClock_ADC0, true);
  
  ADC_Init_TypeDef adcInit = ADC_INIT_DEFAULT;
  
  /* Keep ADC warm to allow a conversion to start ASAP when triggered. */
  adcInit.warmUpMode = adcWarmupKeepADCWarm;
  
  /* At 48MHz HF clock the ADC can run at 12 MHz with 4x divider. */
  adcInit.timebase = ADC_TimebaseCalc(0);
  adcInit.prescale = ADC_PrescaleCalc(12000000, 0);
 
  ADC_Init(ADC0, &adcInit);
  
  /* Initialize single conversion. Set reference and input channel. */
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;
  singleInit.reference = adcRef2V5;
  singleInit.input = ADC_CHANNEL;
  
  /* Use differential measurement */
  singleInit.diff = true;
  
  /* Trigger conversion by PRS */
  singleInit.prsEnable = true;
  singleInit.prsSel = adcPRSSELCh3;
  
  /* Set 12 bit resolution */
  singleInit.resolution = adcRes12Bit;
  
  ADC_InitSingle(ADC0, &singleInit);
  
  /* Set the reference also for SCAN mode. This is needed to 
   * keep the reference warm */
  ADC0->SCANCTRL = (ADC0->SCANCTRL & ~_ADC_SCANCTRL_REF_MASK)
      | ADC_SCANCTRL_REF_2V5;
  
  /* Enable interrupts */
  ADC0->IFC = ADC_IFC_SINGLE;
  ADC0->IEN = ADC_IEN_SINGLE;
  NVIC_ClearPendingIRQ(ADC0_IRQn);
  NVIC_EnableIRQ(ADC0_IRQn);
  
  adcInitTrigger();
  
  /* Start measurements */
  adcStartMeasurements();
}


/**********************************************************
 * Stops the ADC
 *********************************************************/
void adcStop(void)
{
  adcStopMeasurements();  
  ADC0->IEN = 0;
  NVIC_DisableIRQ(ADC0_IRQn);
  ADC0->CTRL = 0;
  ADC0->SINGLECTRL = 0;
  CMU_ClockEnable(cmuClock_ADC0, false);
}


/**********************************************************
 * Stop ADC measurements by disabling the PRS channel
 *********************************************************/
void adcStopMeasurements(void)
{
  /* Disable PRS triggered ADC conversions */
  PRS_SourceSignalSet(3, 0, 0, prsEdgeOff);  
}


/**********************************************************
 * Start ADC measurements by enabling the PRS channel
 *********************************************************/
void adcStartMeasurements(void)
{
  /* Enable PRS triggered ADC conversions */
  PRS_SourceSignalSet(3, PRS_CH_CTRL_SOURCESEL_TIMER2, PRS_CH_CTRL_SIGSEL_TIMER2CC2, prsEdgePos);
}


/**********************************************************
 * Reset the ADC measurement point. The ADC should 
 * sample in the middle of the PWM period. Should be 
 * called every time the PWM duty cycle changes.
 *********************************************************/
void adcSetMeasurementPoint(void)
{
  TIMER_CompareSet(TIMER2, 2, currentPwm/2);
}
