#ifndef _CONFIG_H_
#define _CONFIG_H_
#include "bsp.h"
#include "em_gpio.h"
#include "em_cmu.h"
#define DEBUG_GPIO
/*Define for different target*/
#if defined (EFM32GG990F1024)
#define acmpVMN acmpChannel0
#define acmpVMC acmpChannel7
#define acmpVMB acmpChannel6
#define acmpVMA acmpChannel1
#define TIMER0_ROUTE_LOC TIMER_ROUTE_LOCATION_LOC0
#define GPIO_PWM_PORT  gpioPortA
#define SWD_ENABLE  1
  #if defined DEBUG_GPIO
  #define DBG_GPIO_PORT1 gpioPortB
  #define DBG_GPIO_PIN1  1
  #define DBG_GPIO_PORT2 gpioPortB
  #define DBG_GPIO_PIN2  2
  #endif
#elif defined (EFM32HG322F64)
#define acmpVMN acmpChannel3
#define acmpVMC acmpChannel2
#define acmpVMB acmpChannel1
#define acmpVMA acmpChannel0

#define TIMER0_ROUTE_LOC TIMER_ROUTE_LOCATION_LOC5
#define GPIO_PWM_PORT  gpioPortF
#define SWD_ENABLE  0
  #if defined DEBUG_GPIO
  #define DBG_GPIO_PORT1 gpioPortD
  #define DBG_GPIO_PIN1  4
  #define DBG_GPIO_PORT2 gpioPortD
  #define DBG_GPIO_PIN2  5
  #endif
#endif

/* Overflow value of TIMERs */
#define TIMER_MAX 65536

/* Enum values for COMMUTATION_METHOD */
#define COMMUTATION_HALL  1
#define COMMUTATION_SENSORLESS 2

#define BLDC_FW_VER 0x01


/**********************************************************
 * 
 * MOTOR CONFIGURATION
 * 
 * Change these values to match the motor setup
 * 
 *********************************************************/

/* Must match the number of motor pole pairs */
#define MOTOR_POLE_PAIRS 2

/* This parameter controls whether Hall or Sensorless 
 * commutation is used */
#define COMMUTATION_METHOD COMMUTATION_SENSORLESS


/**********************************************************
 * 
 * SPEED SETPOINT
 * 
 * Set various properities for the RPM setpoint
 * 
 **********************************************************/

/* Default speed setpoint in RPM */
#define DEFAULT_SETPOINT_RPM 2800
/* Minimum value for the speed setpoint (in RPM) */
#define SETPOINT_MIN_RPM 500

/* Maximum value for the speed setpoint (in RPM) */
#define SETPOINT_MAX_RPM 35000

/* The step size for each speed increment/decrement (in RPM) */
#define SPEED_INCREMENT_RPM 800


/**********************************************************
 * 
 * PULSE WIDTH MODULATION
 * 
 * Control the behavior of the PWM waveform
 * 
 **********************************************************/

/* The PWM period in s. */
#if defined (EFM32GG990F1024)
#define PWM_PERIOD_US 40
#elif defined (EFM32HG322F64)
#define PWM_PERIOD_US 120
#endif
/* The minimum PWM duty cycle in percent */
#define PWM_MIN_PERCENT 1.0

/* The maximum PWM duty cycle in percent */
#define PWM_MAX_PERCENT 90.0



/* The initial PWM duty cycle. If the motor has
 * trouble to begin spinning during sensorless
 * startup, increase this parameter */
#define PWM_DEFAULT_DUTY_CYCLE_PERCENT 8
   

/**********************************************************
 * 
 * DEAD TIME INSERTION
 * 
 * Controls the amount of dead time inserted 
 * 
 **********************************************************/
   
/* Uncomment this to use complementary PWM with Dead Time Insertion */
#define USE_COMPLEMENTARY_PWM
   
/* Set the prescaler for the Dead Time Generator. See the reference
 * manual for possible values. Note that this prescales the HFPERCLK
 * and is independent of the prescaler used for the TIMER itself. */
#define DEAD_TIME_PRESCALER TIMER_DTTIME_DTPRESC_DIV16

/* The number of (prescaled) Dead Time cycles for a rising edge of the PWM. */
#define DEAD_TIME_CYCLES_RISING_EDGE 4
   
/* The number of (prescaled) Dead Time cycles for a falling edge of the PWM. */
#define DEAD_TIME_CYCLES_FALLING_EDGE 4   
   
/**********************************************************
 *
 * TIMING PARAMETERS
 * 
 * Various timing parameters and clock frequencies
 * 
 **********************************************************/

/* The HF clock frequency. If using a different crystal or HFRCO band
 * change this to match the nominal frequency. */
#if defined (EFM32GG990F1024)
#define CORE_FREQUENCY 48000000
#elif defined (EFM32HG322F64)
#define CORE_FREQUENCY 24000000
#endif
/* Prescaler used for TIMER0 (PWM generation). Should not need to change this */
#define PRESCALER_TIMER0 1
   
/* Prescaler used for TIMER1, used for timing commutation. Should normally
 * not need to change this. */
#define PRESCALER_TIMER1 32

/* The PID controller period. The controller algorithm is invoked
 * every PID_PERIOD_MS milliseconds. */
#if defined (EFM32GG990F1024)
#define PID_PERIOD_MS 6
#elif defined (EFM32HG322F64)
#define PID_PERIOD_MS 24
#endif
/* Number of milliseconds without a commutation event that should 
 * trigger a 'Motor Stall' condition and shut off the motor */
#define STALL_TIMEOUT_MS 500

/* Number of triggered commutations to do
 * after manual startup is complete, but before
 * before starting PID regulator. This helps the 
 * measured speed to settle and avoids bumps when
 * switching to closed loop regulation */
#define PID_STARTUP_COMMUTATIONS (30)


/**********************************************************
 *
 * HALL STARTUP
 * 
 **********************************************************/

/* This parameter sets the number of manual commutations
 * before starting the Hall commutations. The startup 
 * commutations are perfomed so that the motor has 
 * some speed before the Hall GPIO interrupts are enabled. */
#define STARTUP_COMMUTATIONS_HALL 100

/**********************************************************
 *
 * SENSORLESS STARTUP
 * 
 * These parameters control the motor startup sequence
 * in sensorless (back-emf) driven mode. See also
 * PWM_DEFAULT_DUTY_CYCLE_PERCENT.
 *
 **********************************************************/

/* The initial commutation delay. The commutation delay will 
 * be decreased by a function that keeps a constant angular
 * acceleration. If the startup speed is too fast for the
 * motor to keep up, this parameter can be increased. */
#define STARTUP_INITIAL_PERIOD_MS 50
   
/* The threshold for when the controller should switch from
 * the 'blindly' driven startup mode to back-emf driven
 * commutation. If the motor is able to follow the startup
 * timing, but stops when switching to back-emf, this parameter
 * can be increased.*/
#define STARTUP_FINAL_SPEED_RPM_SENSORLESS 2800

/**********************************************************
 * 
 * CURRENT/OVERCURRENT MEASUREMENT
 * 
 * These parameters control the current measurement which
 * is used for logging and overcurrent protection. 
 * 
 **********************************************************/ 

/* Uncomment this to enable current measurement */
#define CURRENT_MEASUREMENT_ENABLED

/* The maximum current before overcurrent protection
 * will shut off the motor */
#define MAX_CURRENT_MA 10000
   
/* The size of the current measurement resistor (in ohms) */
#define CURRENT_RESISTOR 0.05f

/* Select differential ADC channel used to measure motor current */
#define ADC_CHANNEL adcSingleInpCh6Ch7


/**********************************************************
 *
 * DATA LOGGING
 * 
 * These parameters control data logging. Real-time information
 * about the motor speed, current and input can be plotted
 * on a PC with the supplied efm32_bldc.exe
 *
 **********************************************************/

/* Uncomment this to enable logging to PC. When this is set
 * the firmware will report real-time data over UART */
#define BLDC_LOGGING_ENABLED

/* Set the UART baud rate. This must match the configuration
 * for efm32_bldc.exe */
#define UART_BAUDRATE 115200

/**********************************************************
 * 
 * CALCULATED VALUES
 * (do not modify)
 * 
 *********************************************************/

#define PWM_TOP ((((CORE_FREQUENCY/PRESCALER_TIMER0) / 1000) * PWM_PERIOD_US) / 1000)

#define PWM_DEFAULT_DUTY_CYCLE ((((CORE_FREQUENCY/PRESCALER_TIMER0) / 1000) * (PWM_PERIOD_US * PWM_DEFAULT_DUTY_CYCLE_PERCENT)) / (100 * 1000))

#define PWM_MIN ((int)(((((CORE_FREQUENCY/PRESCALER_TIMER0) / 1000) * (PWM_PERIOD_US * PWM_MIN_PERCENT)) / (100 * 1000))))

#define PWM_MAX ((int)(((((CORE_FREQUENCY/PRESCALER_TIMER0) / 1000) * (PWM_PERIOD_US * PWM_MAX_PERCENT)) / (100 * 1000))))

#define COUNT_TO_RPM(x) (((60 * (CORE_FREQUENCY/PRESCALER_TIMER1)) / MOTOR_POLE_PAIRS) / (x))

#define RPM_TO_COUNT(x) (((60 * (CORE_FREQUENCY/PRESCALER_TIMER1)) / MOTOR_POLE_PAIRS) / (x))   
   
#define PID_PRESCALER ((PID_PERIOD_MS * 1000) / PWM_PERIOD_US)
   
#define STALL_TIMEOUT_OF ((STALL_TIMEOUT_MS * (CORE_FREQUENCY/1000)) / (TIMER_MAX * PRESCALER_TIMER1))
   
#if defined (EFM32GG990F1024)
__STATIC_INLINE void ButtonInit(void)
{
  ;
}

__STATIC_INLINE int ButtonDetect(void)
{
  return BSP_PushButtonsGet();
}

__STATIC_INLINE void kitInit(void)
{
  BSP_Init(BSP_INIT_DK_SPI);
  BSP_PeripheralAccess(BSP_RS232_UART, true);  
}
#elif defined (EFM32HG322F64)
__STATIC_INLINE void ButtonInit(void)
{
  /*int buttonstatus;*/
  CMU_ClockEnable(cmuClock_GPIO, true);
  /* Configure PC9/PC10 as input. */
  GPIO_PinModeSet(gpioPortC, 9, gpioModeInputPull, 1);
  GPIO_PinModeSet(gpioPortC, 10, gpioModeInputPull, 1);
}

__STATIC_INLINE int ButtonDetect(void)
{
  int buttonstatus;
  buttonstatus = !GPIO_PinInGet(gpioPortC,9);
  buttonstatus += (!GPIO_PinInGet(gpioPortC,10)) << 1;
  return buttonstatus;
}

__STATIC_INLINE void kitInit(void)
{
  ;  
}
#endif

#endif
