/**************************************************************************//**
 * @file hall_motor.c
 * @brief Hall-motor specific functions
 * @author Silicon Labs
 * @version 0.00 (leave as is with x.xx, Correct version is automatically inserted by auto-generation)
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2014 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include "em_device.h"
#include "em_cmu.h"
#include "em_timer.h"
#include "em_gpio.h"
#include "gpiointerrupt.h"
#include "config.h"
#include "motor.h"
#include "pwm.h"
#include "pid.h"
#include "logging.h"
#include "hall_motor.h"


#include "hall_motor.h"

/* This array contains the Hall sensor patterns. 
 * Sensor 1,2,3 is connected to PB0,PB1,PB2. 
 * When the motor is driven counter-clockwise
 * the motor phases are shifted 180 degrees
 * with respect to the hall pattern. 
 */
uint32_t hallStatesCw[] =  {1, 3, 2, 6, 4, 5};
uint32_t hallStatesCcw[] = {6, 4, 5, 1, 3, 2};

/* Flag telling if hall-based commutation is running. This flag
 * is set after the startup is completed. */
volatile bool isHallRunning = false;

/* Wrapping counter. Used to keep track of commutation cycle
 * and log speed */
volatile int commutationCounter = 0;

/* Flag telling if PID controller is enabled */
extern bool pidActive;

/* Flag telling if motor is currently runnning */
extern bool isRunning;

/* Keeps track of overflows on TIMER1. Used to 
 * calculate the current speed and detect stall
 * conditions. */
extern volatile int timer1OverflowCounter;


/**********************************************************
 * Initialize hall sensor driven commutation. This
 * functions sets up the pins required to measure
 * the state of the hall sensors.
 *********************************************************/
void hallInit(void)
{
  CMU_ClockEnable(cmuClock_GPIO, true);
  
  /* Hall sensor inputs */
  GPIO_PinModeSet(gpioPortB, 0, gpioModeInput, 0);
  GPIO_PinModeSet(gpioPortB, 1, gpioModeInput, 0);
  GPIO_PinModeSet(gpioPortB, 2, gpioModeInput, 0);
  
  /* Hall sensor VDD */
  GPIO_PinModeSet(gpioPortB, 3, gpioModePushPull, 1);
  
  /* Enable interrupts */
  GPIO_IntConfig(gpioPortB, 0, true, true, true);
  GPIO_IntConfig(gpioPortB, 1, true, true, true);
  GPIO_IntConfig(gpioPortB, 2, true, true, true);
  
  GPIOINT_Init();

  GPIOINT_CallbackRegister(0, hallCommutate);
  GPIOINT_CallbackRegister(1, hallCommutate);
  GPIOINT_CallbackRegister(2, hallCommutate);
}


/**********************************************************
 * Start motor in hall commutation mode. 
 *********************************************************/
void hallStartup(void)
{
  int i,j;
  
  i = 0;
  uint32_t h = 0;
  
  int period = 0;
  int speed = 0;
  
  uint32_t *hallStates = getDirection() ? hallStatesCw : hallStatesCcw;
    
  /* Reset counter before starting the motor */
  TIMER1->CNT = 0;
  
  isRunning = true;
  
  /* Do a fixed number of commutations before switching
   * to GPIO interrupts. This is so that the motor has 
   * at least some speed and we can be sure the GPIO interrupts
   * will fire. */
  for ( i=0; i<STARTUP_COMMUTATIONS_HALL; i++ ) 
  {
    h = GPIO_PortInGet(gpioPortB) & 0x7;
    for ( j=0; j<6; j++ ) {
      if ( hallStates[j] == h )
      {
        int next = (j + 1) % 6;
        pwmSetState(next);
        break;
      }
    }
    
    /* Wait until the hall sensors change state. */
    int timeout = 1000000;
    while ( isRunning && (h == (GPIO_PortInGet(gpioPortB) & 0x7)) && timeout )
    {
      timeout--;
    }
    
    /* If the busy wait timed out, treat as a stall condition.
     * Stop the motor and abort. */
    if ( !timeout ) {
      stopMotor();
      return;
    }
    
    if ( i > 0 && (i % 6 == 0) )
    {
      /* Calculate speed */
      period = TIMER1->CNT;
      TIMER1->CNT = 0;
      period += timer1OverflowCounter * TIMER_MAX;
      timer1OverflowCounter = 0;
      
      speed = (int)((60 * (CORE_FREQUENCY/PRESCALER_TIMER1)) / (float)(period * MOTOR_POLE_PAIRS));
      
      LOG_SET_SPEED(speed);    
    }
  }
  
  LOG_SEND_SCALAR(PARAM_DEBUG, i);
  
  pidInit();
      
  commutationCounter = 1;
  isHallRunning = true;
  pidActive = true;
}


/**********************************************************
 * Starts a hall-sensor driven motor
 **********************************************************/
void hallStartMotor(void)
{
  hallInit();
  hallStartup();
}

/**********************************************************
 * Commutates the motor. On each transition on the hall
 * sensor pins the motor commutation should change. 
 * This function measures the current hall state and
 * enables the next state. 
 *********************************************************/
void hallCommutate(uint8_t n)
{
  /* Unused parameter */
  (void)n;
  
  GPIO->IFC = GPIO->IF;
  
  if ( !isHallRunning )
  {
    return;
  }
    
  /* Get the current state */
  uint32_t hallState = GPIO_PortInGet(gpioPortB) & 0x7;
  
  /* Get the correct hall state array */
  uint32_t *hallStates = getDirection() ? hallStatesCw : hallStatesCcw;
  
  /* Retrieve and set the next state */
  int nextState = 0;
  int i;
  for ( i=0; i<6; i++ ) 
  {
    if ( hallStates[i] == hallState )
    {
      nextState = (i + 1) % 6;
      pwmSetState(nextState);
      break;
    }
  }
  
  commutationCounter = (commutationCounter + 1) % 6 ;  
  if ( commutationCounter == 0 )
  {
    saveSpeed();
  }
  
}
