/**************************************************************************//**
 * @file pwm.c
 * @brief Configures PWM signals for high-side transitors with optional 
 *        complementary PWM for low-side.
 * @author Silicon Labs
 * @version 0.00 (leave as is with x.xx, Correct version is automatically inserted by auto-generation)
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2014 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
#include "em_device.h"
#include "em_cmu.h"
#include "em_timer.h"
#include "em_gpio.h"
#include "config.h"
#include "adc.h"
#include "logging.h"
#include "motor.h"
#include "pwm.h"

/* This mask is used to change only the low
 * side transitor pins (PA3 - PA5) */
#define LOW_SIDE_PIN_MASK 0xFFC7

/* The current commutation state [0-5] */
volatile int pwmCurState = 0;

/* Contains the values to write to TIMER0_ROUTE and 
 * port GPIO_PORT_A_DOUT register for each commutation step */
PwmState states[] = 
{
  { 0x01, 0x10 },
  { 0x01, 0x20 },
  { 0x02, 0x20 },
  { 0x02, 0x08 },
  { 0x04, 0x08 },
  { 0x04, 0x10 }
};


/**********************************************************
 * Sets a new commutation state. 
 * 
 * @param state
 *   The new state [0-5]
 **********************************************************/
void pwmSetState(int state)
{
  uint32_t dout;
  
  /* Write to the TIMER0_ROUTE register to enable/disable the PWM
   * waveforms. This controls the top side transistors.  */
  TIMER0->ROUTE = TIMER0_ROUTE_LOC | states[state].pwm | (states[state].pwm << 8);
  TIMER0->DTOGEN = states[state].pwm | (states[state].pwm << 3);
  
  /* Do a read-modify-write of DOUT register. This controls the
   * low side transistors. Only change pins PA3-PA5 */
  dout = GPIO->P[GPIO_PWM_PORT].DOUT & LOW_SIDE_PIN_MASK;  
  GPIO->P[GPIO_PWM_PORT].DOUT = dout | states[state].high;
    
  /* Save the current state */
  pwmCurState = state;
}

/**********************************************************
 * Go to the next commutation state. 
 *********************************************************/
void pwmNextState(void)
{
  if ( getDirection() ) {
    pwmSetState((pwmCurState + 1) % 6);
  } else {
    int nextState = (pwmCurState - 1);
    if ( nextState < 0 ) {
      nextState += 6;
    }
    pwmSetState(nextState);
  }
}


/**********************************************************
 * Turn of all the transistors. 
 *********************************************************/
void pwmOff(void)
{
  /* Disable all PWM pins */
  TIMER0->ROUTE = 0;
  TIMER0->DTOGEN = 0;
  
  /* Clear PA3-PA7 */
  GPIO->P[GPIO_PWM_PORT].DOUT = GPIO->P[GPIO_PWM_PORT].DOUT & LOW_SIDE_PIN_MASK;
}


/**********************************************************
 * Initialize the PWM waveforms that drive the 
 * inverter transistors. 
 *********************************************************/
void pwmInit(void)
{
  /* Reset state */
  pwmCurState = 0;
  
  /* Enable the driving pins PA0-PA6 */
  CMU_ClockEnable(cmuClock_GPIO, true);
  GPIO_PinModeSet(GPIO_PWM_PORT, 0, gpioModePushPull, 0);
  GPIO_PinModeSet(GPIO_PWM_PORT, 1, gpioModePushPull, 0);
  GPIO_PinModeSet(GPIO_PWM_PORT, 2, gpioModePushPull, 0);
  GPIO_PinModeSet(GPIO_PWM_PORT, 3, gpioModePushPull, 0);
  GPIO_PinModeSet(GPIO_PWM_PORT, 4, gpioModePushPull, 0);
  GPIO_PinModeSet(GPIO_PWM_PORT, 5, gpioModePushPull, 0);
  
  // For Happy Gecko, Disable SWD since the PWM occupy the pins
  // For Giant Gecko, SWD was still used as debug purpose
  GPIO_DbgSWDIOEnable (SWD_ENABLE);

  /* Enable PWM on all three compare/capture channels */
  TIMER_InitCC_TypeDef initCc = TIMER_INITCC_DEFAULT;
  initCc.mode = timerCCModePWM;
  TIMER_InitCC(TIMER0, 0, &initCc);
  TIMER_InitCC(TIMER0, 1, &initCc);
  TIMER_InitCC(TIMER0, 2, &initCc);
  
  /* Set the PWM period and initial duty cycle */
  TIMER_TopSet(TIMER0, PWM_TOP);
  TIMER0->CC[0].CCV = PWM_DEFAULT_DUTY_CYCLE;
  TIMER0->CC[1].CCV = PWM_DEFAULT_DUTY_CYCLE;
  TIMER0->CC[2].CCV = PWM_DEFAULT_DUTY_CYCLE;
  
#ifdef USE_COMPLEMENTARY_PWM
  pwmEnableComplementaryPwm(true);
#endif
  
  /* Set location */
  TIMER0->ROUTE = TIMER0_ROUTE_LOC;
}


/**********************************************************
 * Enables/disables complemtentary PWM output. 
 *********************************************************/
void pwmEnableComplementaryPwm(bool enable)
{
  if ( enable ) 
  {
    /* Enable complementary output */
    TIMER0->DTCTRL =  TIMER_DTCTRL_DTEN;
  
    /* Enable dead time output generation for all CCx and DTIx pins */
    TIMER0->DTOGEN = TIMER_DTOGEN_DTOGCC0EN
                   | TIMER_DTOGEN_DTOGCC1EN
                   | TIMER_DTOGEN_DTOGCC2EN
                   | TIMER_DTOGEN_DTOGCDTI0EN
                   | TIMER_DTOGEN_DTOGCDTI1EN
                   | TIMER_DTOGEN_DTOGCDTI2EN;
                     
    /* Configure dead time for rising and falling edges of the PWM waveform. */
    TIMER0->DTTIME = ((DEAD_TIME_CYCLES_RISING_EDGE-1) << _TIMER_DTTIME_DTRISET_SHIFT) 
                   | ((DEAD_TIME_CYCLES_FALLING_EDGE-1) << _TIMER_DTTIME_DTFALLT_SHIFT)
                   | DEAD_TIME_PRESCALER;
  }
  else
  {
    TIMER0->DTCTRL = 0;
  }
}



/**********************************************************
 * Sets the a new PWM duty cycle. 
 *********************************************************/
void pwmSetDutyCycle(int pwm)
{
  TIMER0->CC[0].CCVB = pwm;
  TIMER0->CC[1].CCVB = pwm;
  TIMER0->CC[2].CCVB = pwm;
  
/* If current measurement is enabled set the 
 * mesurement point to be in the middle of PWM period */  
#ifdef CURRENT_MEASUREMENT_ENABLED
  adcSetMeasurementPoint();
#endif
  
  /* Log the new pwm value */
  LOG_SET_PWM(pwm);
}
