/**************************************************************************//**
 * @file
 * @brief DC-DC Power Configuration 2 Demo for SLSTK3401A
 * @version 4.1.1
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2015 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdint.h>
#include <stdbool.h>
#include "em_device.h"
#include "em_chip.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "bsp.h"

volatile uint32_t msTicks; /* counts 1ms timeTicks */

void Delay(uint32_t dlyTicks);

/**************************************************************************//**
 * @brief SysTick_Handler
 * Interrupt Service Routine for system tick counter
 *****************************************************************************/
void SysTick_Handler(void)
{
  msTicks++;       /* increment counter necessary in Delay()*/
}

/**************************************************************************//**
 * @brief Delays number of msTick Systicks (typically 1 ms)
 * @param dlyTicks Number of ticks to delay
 *****************************************************************************/
void Delay(uint32_t dlyTicks)
{
  uint32_t curTicks;

  curTicks = msTicks;
  while ((msTicks - curTicks) < dlyTicks) ;
}

/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main(void)
{
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_STK_DEFAULT;
  CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_STK_DEFAULT;

  // Turn on DCDC converter for power configuration 2 - DCDC to DVDD (1.8V)
  dcdcInit.powerConfig = emuPowerConfig_DcdcToDvdd;             // DCDC to DVDD
  dcdcInit.dcdcMode = emuDcdcMode_LowNoise;                     // Low-noise mode in EM0
  dcdcInit.mVout = 1800;                                        // Nominal output voltage for DVDD mode, 1.8V
  dcdcInit.em01LoadCurrent_mA = 5;                              // Nominal EM0/EM1 load current of less than 5mA
  dcdcInit.em234LoadCurrent_uA = 10;                            // Nominal EM2/EM3 load current less than 10uA
  dcdcInit.maxCurrent_mA = 160;                                 // Maximum peak current of 160mA
  dcdcInit.anaPeripheralPower = emuDcdcAnaPeripheralPower_AVDD; // Select AVDD supply (less noise)
  dcdcInit.reverseCurrentControl = emuDcdcLnHighEfficiency;     // Use low-noise high-efficiency mode (ignored if emuDcdcMode_LowPower)
  
  /* Chip errata */
  CHIP_Init();

  /* Init DCDC regulator and HFXO with kit specific parameters */
  EMU_DCDCInit(&dcdcInit);
  CMU_HFXOInit(&hfxoInit);

  /* Switch HFCLK to HFXO and disable HFRCO */
  CMU_ClockSelectSet(cmuClock_HF, cmuSelect_HFXO);
  CMU_OscillatorEnable(cmuOsc_HFRCO, false, false);

  /* Setup SysTick Timer for 1 msec interrupts  */
  if (SysTick_Config(CMU_ClockFreqGet(cmuClock_CORE) / 1000)) while (1) ;

  /* Initialize LED driver */
  BSP_LedsInit();
  BSP_LedSet(0);
  BSP_LedClear(1);

  /* Infinite blink loop */
  while (1)
  {
    BSP_LedToggle(0);
    BSP_LedToggle(1);
    Delay(1000);
  }
}
