/**************************************************************************//**
 * @file
 * @brief AES-256 encryption example program
 * @version 4.1.1
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2014 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

/*
 * This file is a modified version of the aescrypt2.c example program
 * included in mbedtls-2.1.0 (https://tls.mbed.org)
 */

#if !defined(MBEDTLS_CONFIG_FILE)
#include "mbedtls/config.h"
#else
#include MBEDTLS_CONFIG_FILE
#endif

#include "em_device.h"
#include "em_chip.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "em_gpio.h"
#include "em_msc.h"

#include "display.h"
#include "textdisplay.h"
#include "retargettextdisplay.h"
#include "pbconfig.h"
//
#include "retargetserial.h"
#include "mbedtls/aes.h"
#include "mbedtls/md.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define CYCLE_INST

#define USERPAGE    0x0FE00000 /**< Address of the user page */
msc_Return_TypeDef        currentError = mscReturnOk; /** < Latest error encountered */

#define mbedtls_fprintf            fprintf
#define mbedtls_printf              printf

#define MODE_ENCRYPT                   (1)
#define MODE_DECRYPT                   (2)

#define AES_BLOCK_SIZE                (16)
#define IV_SIZE                       (16)
#define TAG_SIZE                      (32)
#define MAX_MESSAGE_SIZE_ENCRYPTION (512)
#define MAX_MESSAGE_SIZE_DECRYPTION                             \
  (2*MAX_MESSAGE_SIZE_ENCRYPTION + 2*IV_SIZE + 2*TAG_SIZE + 1)

#define STDIN_FILENO                   (0)
#define STDOUT_FILENO                  (1)

static void GpioSetup(void);

static char message[MAX_MESSAGE_SIZE_DECRYPTION];

static const char *key256bits = "603DEB10 15CA71BE 2B73AEF0 857D7781"
                                "1F352C07 3B6108D7 2D9810A3 0914DFF4";

static volatile bool      displayEnabled = false; /* Status of LCD display. */
static DISPLAY_Device_t displayDevice;    /* Display device handle.         */

int  mode = 0;
int  enc_sel = 0;

/**************************************************************************//**
 * @brief Convert ascii hexadecimal text into binary
 *****************************************************************************/
static int hextext2bin(uint8_t *binbuf, unsigned int binbuflen, const char *hexstr)
{
    uint32_t ret = 0;
    int      i;
    uint8_t  tmp;
    uint8_t  val;

    while (ret < binbuflen)
    {
        val = 0;
        for (i = 1; i >= 0; i--)
        {
            tmp = *(hexstr++);
            /* Skip spaces */
            while (tmp == ' ')
            {
                tmp = *(hexstr++);
            }
            /* Reached end of string? */
            if (!tmp)
              goto done;

            if (tmp > '9')
            {
                /* Ensure uppercase hex */
                tmp &= ~0x20;

                val |= ((tmp - 'A') + 10) << (4 * i);
            }
            else
            {
                val |= (tmp - '0') << (4 * i);
            }
        }
        *(binbuf++) = val;
        ret++;
    }
 done:
    return ret;
}

/**************************************************************************//**
 * @brief Convert binary data to ascii hexadecimal text string
 *****************************************************************************/
void bin2hextext(char* hexstr, uint8_t* binbuf, unsigned int binbuflen)
{
    uint32_t i;
    uint8_t nibble;

    for (i = 0; i < binbuflen; i++)
    {
      nibble = (binbuf[i] >> 4) & 0xF;
      *hexstr++ = nibble > 9 ? nibble - 10 + 'A' : nibble + '0';
      nibble = (binbuf[i] >> 0) & 0xF;
      *hexstr++ = nibble > 9 ? nibble - 10 + 'A' : nibble + '0';
    }
    /* Null terminate at end of string. */
    *hexstr = 0;
}

/**************************************************************************//**
 * @brief Initialize timestamp
 *****************************************************************************/
static void timestamp_init(void)
{
    /* Enable debug clock AUXHFRCO */
    CMU_OscillatorEnable(cmuOsc_AUXHFRCO, true, true);

    /* Enable trace in core debug */
    CoreDebug->DHCSR |= CoreDebug_DHCSR_C_DEBUGEN_Msk;
    CoreDebug->DEMCR |= CoreDebug_DEMCR_TRCENA_Msk;

    /* Unlock ITM and output data */
    ITM->LAR = 0xC5ACCE55;
    ITM->TCR = 0x10009 | ITM_TCR_DWTENA_Pos;

    /* Enable PC and IRQ sampling output */
    DWT->CTRL = 0x400113FF;
    DWT->CYCCNT = 0;
}

/**************************************************************************//**
 * @brief  Get clock cycle timestamp
 *****************************************************************************/
static inline uint32_t clk_timestamp_get(void)
{
    return DWT->CYCCNT;
}

/**************************************************************************//**
 * @brief Setup GPIO interrupt for pushbuttons.
 *****************************************************************************/
static void GpioSetup(void)
{
  /* Enable GPIO clock. */
  CMU_ClockEnable(cmuClock_GPIO, true);

  /* Configure PF6 as input and enable interrupt. */
  GPIO_PinModeSet(PB0_SW_IRQ_PORT, PB0_SW_IRQ_PIN, gpioModeInputPull, 1);
  GPIO_IntConfig(PB0_SW_IRQ_PORT, PB0_SW_IRQ_PIN, false, true, true);

  /* Configure PF7 as input. */
  GPIO_PinModeSet(PB1_SW_IRQ_PORT, PB1_SW_IRQ_PIN, gpioModeInputPull, 1);
  GPIO_IntConfig(PB1_SW_IRQ_PORT, PB1_SW_IRQ_PIN, false, true, true);

  NVIC_ClearPendingIRQ(GPIO_EVEN_IRQn);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  NVIC_ClearPendingIRQ(GPIO_ODD_IRQn);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);

}

/**************************************************************************//**
* @brief Unified GPIO Interrupt handler (pushbuttons)
*        PB0 Starts selected test
*        PB1 Cycles through the available tests
*****************************************************************************/
void GPIO_Unified_IRQ(void) {
   /* Get and clear all pending GPIO interrupts */
   uint32_t interruptMask = GPIO_IntGet();
   GPIO_IntClear(interruptMask);

   /* Act on interrupts */
   if (interruptMask & (1 << PB0_SW_IRQ_PIN)) {
   }
}

/**************************************************************************//**
* @brief GPIO Interrupt handler for even pins
*****************************************************************************/
void GPIO_EVEN_IRQHandler(void)
{
	 mode = MODE_ENCRYPT;
	 if (enc_sel) {
		 enc_sel = 0;
	 }
	 else enc_sel = 1;

     GPIO_Unified_IRQ();
}

/**************************************************************************//**
* @brief GPIO Interrupt handler for odd pins
*****************************************************************************/
void GPIO_ODD_IRQHandler(void)
{
	 mode = MODE_DECRYPT;
     GPIO_Unified_IRQ();
}

/**************************************************************************//**
 * @brief Read cipher text from flash user data page
 *****************************************************************************/
void ReadCipherText(uint32_t *fldata, uint32_t len)
{
	  /* Copy contents of the userpage (flash) into fldata  */
	  memcpy(fldata, (void *) USERPAGE, len);
}

/**************************************************************************//**
 * @brief Write cipher text to the flash user data page
 *****************************************************************************/
void WriteCipherText(char *fldata, uint32_t len)
{
      msc_Return_TypeDef ret;

	  /* Initialize the MSC for writing */
	  MSC_Init();

	  /* Erase the page */
	  ret = MSC_ErasePage((uint32_t *) USERPAGE);

	  /* Check for errors. If there are errors, set the global error variable and
	   * deinitialize the MSC */
	  if (ret != mscReturnOk)
	  {
	    currentError = ret;
	    MSC_Deinit();
	    return;
	  }

	  /* Write data to the userpage */
	  ret = MSC_WriteWordFast((uint32_t *) USERPAGE, (void *) fldata, len);

	  /* Check for errors. If there are errors, set the global error variable and
	   * deinitialize the MSC */
	  if (ret != mscReturnOk)
	  {
	    currentError = ret;
	    MSC_Deinit();
	    return;
	  }
	  /* Deinitialize the MSC. This disables writing and locks the MSC */
	  MSC_Deinit();

}

/**************************************************************************//**
 * @brief Main function
 *****************************************************************************/
int main( void )
{
    CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_STK_DEFAULT;
    EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_STK_DEFAULT;
    int ret;
    int i, loopcnt=0, data_create_64_256[3] = {64, 128, 256};
    int lastn;
    size_t keylen = 256/8;
    char *p;
    unsigned char IV[AES_BLOCK_SIZE];
    unsigned char key[256/8];
    unsigned char digest[32];
    unsigned char diff;

    char initphrase2[16] = "Gecko";
    char encrypt_status = 0;


    unsigned char buffer[MAX_MESSAGE_SIZE_DECRYPTION];
    char message2decrypt[MAX_MESSAGE_SIZE_DECRYPTION] = {0};
    int decrypt_bit_cnt = 0;

    mbedtls_aes_context aes_ctx;
    mbedtls_md_context_t sha_ctx;

    long message_size;
    static uint32_t cycles[8];

    /* Chip errata */
    CHIP_Init();

    /* Init DCDC regulator and HFXO with kit specific parameters */
    EMU_DCDCInit(&dcdcInit);
    CMU_HFXOInit(&hfxoInit);

    /* Switch HFCLK to HFXO and disable HFRCO */
    CMU_ClockSelectSet(cmuClock_HF, cmuSelect_HFXO);
    CMU_OscillatorEnable(cmuOsc_HFRCO, false, false);
    /* Setup GPIO for pushbuttons. */
    GpioSetup();

    /* Initialize the display module. */
    displayEnabled = true;
    DISPLAY_Init();

    /* Retrieve the properties of the display. */
    if (DISPLAY_DeviceGet(0, &displayDevice ) != DISPLAY_EMSTATUS_OK)
    {
      /* Unable to get display handle. */
      while (1);
    }

    /* Retarget stdio to the display. */
    if (TEXTDISPLAY_EMSTATUS_OK != RETARGET_TextDisplayInit())
    {
      /* Text display initialization failed. */
       while (1);
    }

    printf( "\n  CRYPTO DEMO "
            "\n\n " );
    printf("BTN0-> Encrypt\n\n");
    printf(" BTN1-> Decrypt\n\n");

    /* Enable timestamping */
    timestamp_init();

    while(1) {
        mbedtls_aes_init( &aes_ctx );
        mbedtls_md_init( &sha_ctx );

        ret = mbedtls_md_setup( &sha_ctx,
                                mbedtls_md_info_from_type( MBEDTLS_MD_SHA256 ),
                                1 );
        if( ret != 0 )
        {
            mbedtls_printf( "  ! mbedtls_md_setup() returned -0x%04x\n", -ret );
            goto exit;
        }

    	memset(message, 0, sizeof(message));
        memset(IV, 0, sizeof(IV));
        memset(key, 0, sizeof(key));
        memset(digest, 0, sizeof(digest));
        memset(buffer, 0, sizeof(buffer));

        hextext2bin((uint8_t *) key, 256/8, key256bits);

        if( mode == MODE_ENCRYPT )
        {
    	   if (encrypt_status) {
    	      decrypt_bit_cnt = 0;
    	   }
           if (!loopcnt) loopcnt = 1;
           else if (loopcnt == 1) loopcnt = 2;
           else loopcnt = 0;

           char hexbuf[1025];

    	   printf("\033[H");
    	   printf("\n\n\n\n\n\n\n\n");
    	   printf("Encrypting %d \n", data_create_64_256[loopcnt]);
    	   printf("bytes.          ");
    	   printf("\n                ");
    	   printf("\n                ");
    	   printf("\n                ");
    	   printf("\n                ");
           printf("\033A");
           printf("\033A");
           printf("\033A");
           printf("\033A");

           /*
            * Generate the initialization vector as:
            * IV = SHA-256( message_size || initphrase )[0..15]
            */
           for( i = 0; i < 8; i++ )
              buffer[i] = (unsigned char)( message_size >> ( i << 3 ) );


           mbedtls_md_starts( &sha_ctx );
           mbedtls_md_update( &sha_ctx, buffer, 8 );

           p = initphrase2;
           mbedtls_md_update( &sha_ctx, (unsigned char*)p, strlen(initphrase2) );
           /*
            * Generate the message.  In this case we are just creating a sequential
            * array of 128 values.
            */
           for (i=0;i<data_create_64_256[loopcnt];i++)
           {
        	   message[i] = i;
           }
           message_size = i;
           mbedtls_md_finish( &sha_ctx, digest );

           memcpy( IV, digest, sizeof(IV) );

           /*
            * The last four bits in the IV are actually used
            * to store the file size modulo the AES block size.
            */
           lastn = (int)( message_size & 0x0F );

           IV[15] = (unsigned char) ( ( IV[15] & 0xF0 ) | lastn );

           /*
            * Append the IV at the beginning of the output.
            */
           bin2hextext( hexbuf, IV, sizeof(IV) );

           for (i=0; i < 2*IV_SIZE; i++) {
        	  message2decrypt[decrypt_bit_cnt] = hexbuf[i];
        	  decrypt_bit_cnt++;
           }
           memset( digest, 0,  sizeof(digest) );
           memcpy( digest, IV, IV_SIZE );
           memset( key, 0, sizeof( key ) );
           p = message;
           memset (buffer, 0, message_size);
           memcpy (buffer, p, message_size);

           /*
            * Start the timing here
            *
            */
    	   printf("\rbytes..");
           DWT->CYCCNT = 0x00;
           cycles[0] = clk_timestamp_get();
           cycles[7]=0;

           /*
            * Hash the IV and the secret key together 8192 times
            * using the result to setup the AES context and HMAC.
            */
           for( i = 0; i < 8192; i++ )
           {
               mbedtls_md_starts( &sha_ctx );
               mbedtls_md_update( &sha_ctx, digest, TAG_SIZE );
               mbedtls_md_update( &sha_ctx, key, keylen );
               mbedtls_md_finish( &sha_ctx, digest );
           }
#ifdef CYCLE_INST
           cycles[1] = clk_timestamp_get() - cycles[0];
#endif
    	   printf(".");
#ifdef CYCLE_INST
    	   DWT->CYCCNT = 0x00;
           cycles[0] = clk_timestamp_get();
#endif
           mbedtls_aes_setkey_enc( &aes_ctx, digest, 256 );
#ifdef CYCLE_INST
           cycles[2] = clk_timestamp_get() - cycles[0];
           cycles[7] += cycles[2];
           DWT->CYCCNT = 0x00;
           cycles[0] = clk_timestamp_get();
#endif
           mbedtls_md_hmac_starts( &sha_ctx, digest, TAG_SIZE );
#ifdef CYCLE_INST
           cycles[3] = clk_timestamp_get() - cycles[0];
           cycles[7] += cycles[3];
#endif
    	   printf(".");
#ifdef CYCLE_INST
           DWT->CYCCNT = 0x00;
           cycles[0] = clk_timestamp_get();
#endif
           mbedtls_aes_crypt_cbc( &aes_ctx, MBEDTLS_AES_ENCRYPT, message_size, IV, buffer, buffer );
#ifdef CYCLE_INST
           cycles[4] = clk_timestamp_get() - cycles[0];
           cycles[7] += cycles[4];
           DWT->CYCCNT = 0x00;
           cycles[0] = clk_timestamp_get();
#endif
           mbedtls_md_hmac_update( &sha_ctx, buffer, message_size );
#ifdef CYCLE_INST
           cycles[5] = clk_timestamp_get() - cycles[0];
           cycles[7] += cycles[5];
           DWT->CYCCNT = 0x00;
           cycles[0] = clk_timestamp_get();
#endif
           /*
            * Finally write the HMAC.
            */
           mbedtls_md_hmac_finish( &sha_ctx, digest );
#ifdef CYCLE_INST
           cycles[6] = clk_timestamp_get() - cycles[0];
           cycles[7] += cycles[6];
#endif
           /*
            * End timing measurement
            */
    	   printf(".");
             bin2hextext( hexbuf, buffer, message_size );
           for (i=0; i < 2*message_size; i++)
           {
              message2decrypt[decrypt_bit_cnt] = hexbuf[i];
              decrypt_bit_cnt++;
           }
           printf("\n                ");
           printf("\r\nTotal Encrypt  ");
           printf("\r\ncycles = %d\n", cycles[7]);
           printf("\n                ");

           bin2hextext( hexbuf, digest, TAG_SIZE );
           for (i=0; i < 2*TAG_SIZE; i++) {
              message2decrypt[decrypt_bit_cnt] = hexbuf[i];
        	  decrypt_bit_cnt++;
           }

           WriteCipherText(message2decrypt, (sizeof(message2decrypt)));
           encrypt_status = 1;

       }

       if( mode == MODE_DECRYPT )
       {
           if (encrypt_status)
           {
       	      printf("\033[H");
       	      printf("\n\n\n\n\n\n\n\n");
       	      printf("Decrypting %d \n",data_create_64_256[loopcnt]);
       	      printf("bytes.          ");
       	      printf("\n                ");
       	      printf("\n                ");
       	      printf("\n                ");
              printf("\033A");
              printf("\033A");
              printf("\033A");

       	      message_size = decrypt_bit_cnt;
              memset( message2decrypt, 0, message_size );
              ReadCipherText((uint32_t *)message2decrypt, message_size);

              /*
               *  The encrypted file must be structured as follows:
               *
               *        00 .. 15              Initialization Vector
               *        16 .. 31              AES Encrypted Block #1
               *           ..
               *      N*16 .. (N+1)*16 - 1    AES Encrypted Block #N
               *  (N+1)*16 .. (N+1)*16 + 32   HMAC-SHA-256(ciphertext)
               */
              if( message_size <  (2*IV_SIZE + 2*TAG_SIZE) )
              {
                  mbedtls_printf( "File too short to be encrypted.\n" );
                  goto exit;
              }

              if( ( message_size & 0x0F ) != 0 )
              {
                 mbedtls_printf(  "File size not a multiple of 16.\n" );
                 goto exit;
              }

              /*
               * Subtract the IV + HMAC length.
               */
              message_size -= ( 2*IV_SIZE + 2*TAG_SIZE );

              /*
               * Read the IV and original message_size modulo 16.
               */
              hextext2bin(IV, IV_SIZE, message2decrypt);
              lastn = IV[15] & 0x0F;

              /*
               * Hash the IV and the secret key together 8192 times
               * using the result to setup the AES context and HMAC.
               */
              memset( digest, 0,  TAG_SIZE );
              memcpy( digest, IV, IV_SIZE );
              memset( key, 0, sizeof( key ) );

              p = &message2decrypt[2*IV_SIZE];
              hextext2bin( buffer, message_size/2, p );
              /*
               * Start timing here
               */
              DWT->CYCCNT = 0x00;
              cycles[0] = clk_timestamp_get();
              cycles[7]=0;

             for( i = 0; i < 8192; i++ )
             {
                mbedtls_md_starts( &sha_ctx );
                mbedtls_md_update( &sha_ctx, digest, TAG_SIZE );
                mbedtls_md_update( &sha_ctx, key, keylen );
                mbedtls_md_finish( &sha_ctx, digest );
             }
#ifdef CYCLE_INST
             cycles[1] = clk_timestamp_get() - cycles[0];
#endif
             printf("\rbytes..");

#ifdef CYCLE_INST
             DWT->CYCCNT = 0x00;
             cycles[0] = clk_timestamp_get() - cycles[0];
#endif
             mbedtls_aes_setkey_dec( &aes_ctx, digest, 256 );
#ifdef CYCLE_INST
             cycles[2] = clk_timestamp_get() - cycles[0];
             cycles[7] += cycles[2];
             DWT->CYCCNT = 0x00;
             cycles[0] = clk_timestamp_get();
#endif
             mbedtls_md_hmac_starts( &sha_ctx, digest, TAG_SIZE );
#ifdef CYCLE_INST
             cycles[3] = clk_timestamp_get() - cycles[0];
             cycles[7] += cycles[3];
#endif
             printf(".");
#ifdef CYCLE_INST
             DWT->CYCCNT = 0x00;
             cycles[0] = clk_timestamp_get() - cycles[0];
#endif
             mbedtls_md_hmac_update( &sha_ctx, buffer, message_size/2 );
#ifdef CYCLE_INST
             cycles[4] = clk_timestamp_get() - cycles[0];
             cycles[7] += cycles[4];
             DWT->CYCCNT = 0x00;
             cycles[0] = clk_timestamp_get();
#endif
             mbedtls_aes_crypt_cbc( &aes_ctx, MBEDTLS_AES_DECRYPT, message_size/2, IV, buffer, buffer );
#ifdef CYCLE_INST
             cycles[5] = clk_timestamp_get() - cycles[0];
             cycles[7] += cycles[5];
#endif
             printf(".");
#ifdef CYCLE_INST
             DWT->CYCCNT = 0x00;
             cycles[0] = clk_timestamp_get();
#endif
             /*
              * Verify the message authentication code.
              */
             mbedtls_md_hmac_finish( &sha_ctx, digest );
#ifdef CYCLE_INST
             cycles[6] = clk_timestamp_get() - cycles[0];
             cycles[7] += cycles[6];
#endif
             /*
              * End timing here
              */
             cycles[0] = clk_timestamp_get() - cycles[0];
             printf(".");

             p=&message2decrypt[2*IV_SIZE+message_size];
             hextext2bin( buffer, TAG_SIZE, p );


             /* Use constant-time buffer comparison */
             diff = 0;
             for( i = 0; i < TAG_SIZE; i++ )
                 diff |= digest[i] ^ buffer[i];

             if( diff != 0 )
             {
                mbedtls_printf( "HMAC check failed: wrong key, "
                                "or file corrupted.\n" );
                memset( message2decrypt, 0, sizeof( message2decrypt ) );
                encrypt_status = 0;
                decrypt_bit_cnt = 0;
                goto exit;
             }
             else
                printf("\nMessage OK.");
                printf("\r\n                ");
                printf("\r\nTotal Decrypt   ");
                printf("\r\ncycles = %d", cycles[7]);

             memset( message2decrypt, 0, sizeof( message2decrypt ) );
             encrypt_status = 0;
             decrypt_bit_cnt = 0;
          } // encrypt_status
       }

       ret = 0;
  	   mode = 0;

       exit:
          memset( buffer, 0, sizeof( buffer ) );
          memset( digest, 0, sizeof( digest ) );

          mbedtls_aes_free( &aes_ctx );
          mbedtls_md_free( &sha_ctx );
   }
}
