/**************************************************************************//**
 * @file  lcddisplay.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdio.h>
#include "em_cmu.h"
#include "em_gpio.h"
#include "em_usart.h"
#include "delay.h"
#include "lcddisplay.h"
#include "progconfig.h"

#if (DISP_INTERFACE == DISP_EXT_LCD)
static volatile uint8_t lcdRow, lcdCol;
#if (SHARE_SPI == 1)
static volatile uint32_t spiCtrl, spiClkDiv;
static volatile uint32_t lcdSpiCtrl, lcdSpiClkDiv;
#endif

static void checkBusy(void);
static void writeInsData(bool insDataSelect, uint8_t insData);

/**************************************************************************//**
 * @brief Receive a byte
 *    No input method from the LCD is possible, thus we always
 *    return -1
 *
 * @return -1 on failure
 *****************************************************************************/
int RETARGET_ReadChar(void)
{
  return -1;
}

/**************************************************************************//**
 * @brief Write a single byte to the LCD
 *
 * @param c Character to write
 *
 * @return Printed character if OK.
 *         -1 on failure.
 *****************************************************************************/
int RETARGET_WriteChar(char c)
#if (SHARE_SPI == 1)
{
  uint32_t i;
  
  LCD_SPI_USART->CTRL = lcdSpiCtrl;
  LCD_SPI_USART->CLKDIV = lcdSpiClkDiv;
  
  if (c == '\b')
  {
    writeInsData(true, SET_ROW1);
    lcdRow = 0x01;
    LCD_SPI_USART->CTRL = spiCtrl;
    LCD_SPI_USART->CLKDIV = spiClkDiv;
    return c;
  }

  if (c == '\t')
  {
    writeInsData(true, SET_ROW2);
    lcdRow = 0x02;
    LCD_SPI_USART->CTRL = spiCtrl;
    LCD_SPI_USART->CLKDIV = spiClkDiv;
    return c;
  }

  if (c == '\v')
  {
    writeInsData(true, SET_ROW3);
    lcdRow = 0x03;
    LCD_SPI_USART->CTRL = spiCtrl;
    LCD_SPI_USART->CLKDIV = spiClkDiv;
    return c;
  }

  if (c == '\n')
  {
    /* Line Feed, fill remain column with space */
    lcdCol = COL_MAX - lcdCol;
    for (i=0; i<lcdCol; i++)
    {
      writeInsData(false, DATA_SPACE);
    }
    
    /* Address auto advance to next row, set new row and reset column */
    lcdRow++;
    lcdCol = 0;
    LCD_SPI_USART->CTRL = spiCtrl;
    LCD_SPI_USART->CLKDIV = spiClkDiv;
    return c;
  }
  
  /* Clear display if last row or \a */
  if ((lcdRow == ROW_MAX) || (c == '\a'))
  {
    writeInsData(true, DISP_CLEAR);
    lcdRow = 0;
    if (c == '\a')
    {
      LCD_SPI_USART->CTRL = spiCtrl;
      LCD_SPI_USART->CLKDIV = spiClkDiv;
      return c;
    }
  }

  /* All string must with line feed and <= 20 columns */
  if (lcdCol == COL_MAX)
  {
    LCD_SPI_USART->CTRL = spiCtrl;
    LCD_SPI_USART->CLKDIV = spiClkDiv;
    return -1;
  }

  /* Output character to LCD */
  writeInsData(false, c);
  lcdCol++;

  LCD_SPI_USART->CTRL = spiCtrl;
  LCD_SPI_USART->CLKDIV = spiClkDiv;
  return c;
}
#else
{
  uint32_t i;
  
  if (c == '\b')
  {
    writeInsData(true, SET_ROW1);
    lcdRow = 0x01;
    return c;
  }

  if (c == '\t')
  {
    writeInsData(true, SET_ROW2);
    lcdRow = 0x02;
    return c;
  }

  if (c == '\v')
  {
    writeInsData(true, SET_ROW3);
    lcdRow = 0x03;
    return c;
  }

  if (c == '\n')
  {
    /* Line Feed, fill remain column with space */
    lcdCol = COL_MAX - lcdCol;
    for (i=0; i<lcdCol; i++)
    {
      writeInsData(false, DATA_SPACE);
    }
    
    /* Address auto advance to next row, set new row and reset column */
    lcdRow++;
    lcdCol = 0;
    return c;
  }
  
  /* Clear display if last row or \a */
  if ((lcdRow == ROW_MAX) || (c == '\a'))
  {
    writeInsData(true, DISP_CLEAR);
    lcdRow = 0;
    if (c == '\a')
    {
      return c;
    }
  }

  /* All string must with line feed and <= 20 columns */
  if (lcdCol == COL_MAX)
  {
    return -1;
  }

  /* Output character to LCD */
  writeInsData(false, c);
  lcdCol++;

  return c;
}
#endif

/******************************************************************************
 * @brief  Initialize LCD
 *****************************************************************************/
void initLcdDisp(void)
{
  USART_InitSync_TypeDef init = USART_INITSYNC_DEFAULT;

  LCD_RESET_CLR();
  delayMs(1);
  LCD_RESET_SET();
  delayMs(1);
  
#if (SHARE_SPI == 1)
  /* Save SPI setting for flash */
  spiCtrl = LCD_SPI_USART->CTRL;
  spiClkDiv = LCD_SPI_USART->CLKDIV;
#endif
  
  CMU_ClockEnable(LCD_SPI_CMU, true);

  /* Configure SPI */  
  init.baudrate = LCD_SPI_FREQ;

  /* Configure USART for SPI */
  USART_InitSync(LCD_SPI_USART, &init);
  
#if (SHARE_SPI == 1)
  /* Save SPI setting for LCD */
  lcdSpiCtrl = LCD_SPI_USART->CTRL;
  lcdSpiClkDiv = LCD_SPI_USART->CLKDIV;

  /* Setup USART pins, high drive on TX and CLK if share */
  GPIO_DriveModeSet(LCD_SPI_TXPORT, gpioDriveModeHigh);
  GPIO_DriveModeSet(LCD_SPI_CLKPORT, gpioDriveModeHigh);
  GPIO_PinModeSet(LCD_SPI_TXPORT, LCD_SPI_TXPIN, gpioModePushPullDrive, 0); 
  GPIO_PinModeSet(LCD_SPI_CLKPORT, LCD_SPI_CLKPIN, gpioModePushPullDrive, 0);
#else  
  GPIO_PinModeSet(LCD_SPI_TXPORT, LCD_SPI_TXPIN, gpioModePushPull, 0); 
  GPIO_PinModeSet(LCD_SPI_CLKPORT, LCD_SPI_CLKPIN, gpioModePushPull, 0);
#endif
  GPIO_PinModeSet(LCD_SPI_RXPORT, LCD_SPI_RXPIN, gpioModeInput, 0);
  GPIO_PinModeSet(LCD_SPI_CS_PORT, LCD_SPI_CS_PIN, gpioModePushPull, 1);	

  /* Enabling pins and setting location, use manual control on CS */
  LCD_SPI_USART->ROUTE = USART_ROUTE_TXPEN | USART_ROUTE_RXPEN | USART_ROUTE_CLKPEN | LCD_SPI_LOC;

  writeInsData(true, 0x34);	/* RE=1 */
  writeInsData(true, 0x09);	/* 4 line mode */
  writeInsData(true, 0x30);	/* RE=0 */
  writeInsData(true, 0x0c);	/* Display on */
  writeInsData(true, 0x01);	/* Clear display */
}

/******************************************************************************
 * @brief  Check LCD is busy or not
 *****************************************************************************/
static void checkBusy(void)
{
  uint8_t readData;

  do
  {
    LCD_CS_CLR();
    USART_Tx(LCD_SPI_USART, READ_START);
    USART_Rx(LCD_SPI_USART);
    USART_Tx(LCD_SPI_USART, READ_BUSY);
    readData = USART_Rx(LCD_SPI_USART);
    LCD_CS_SET();
  } while (readData & BUSY_MASK); 
}

/**************************************************************************//**
 * @brief Write instruction or data to the LCD
 *
 * @param insDataSelect 
 *              True for instruction, false for data 
 * @param insDat
 *              Instruction or data to write
 *****************************************************************************/
static void writeInsData(bool insDataSelect, uint8_t insData)
{
  checkBusy();
  LCD_CS_CLR();
  if (insDataSelect)
  {
    USART_Tx(LCD_SPI_USART, INS_START);
  }
  else
  {
    USART_Tx(LCD_SPI_USART, DATA_START);
  }    
  USART_Rx(LCD_SPI_USART);
  USART_Tx(LCD_SPI_USART, (insData & 0x0f));
  USART_Rx(LCD_SPI_USART);
  USART_Tx(LCD_SPI_USART, ((insData >> 4) & 0x0f));
  USART_Rx(LCD_SPI_USART);
  LCD_CS_SET();
}
#endif
