/**************************************************************************//**
 * @file  nandutil.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdio.h>
#include "em_cmu.h"
#include "em_gpio.h"
#include "em_ebi.h"
#include "nandflash.h"
#include "nandutil.h"

static void nandFlashEbiInit(void);

/**************************************************************************//**
 * @brief Initialize the EBI interface for accessing the onboard nandflash.
 *****************************************************************************/
static void nandFlashEbiInit(void)
{
  /* ------------------------------------------ */
  /* NAND Flash, Bank0, Base Address 0x80000000 */
  /* Micron flash NAND256W3A                    */
  /* ------------------------------------------ */

  EBI_Init_TypeDef ebiConfig =
  {   ebiModeD8A8,       /* 8 bit address, 8 bit data */
      ebiActiveLow,      /* ARDY polarity */
      ebiActiveLow,      /* ALE polarity */
      ebiActiveLow,      /* WE polarity */
      ebiActiveLow,      /* RE polarity */
      ebiActiveLow,      /* CS polarity */
      ebiActiveLow,      /* BL polarity */
      false,             /* disble BL */
      true,              /* enable NOIDLE */
      false,             /* disable ARDY */
      true,              /* disable ARDY timeout */
      EBI_BANK0,         /* enable bank 0 */
      0,                 /* no chip select */
      0,                 /* addr setup cycles */
      0,                 /* addr hold cycles */
      false,             /* disable half cycle ALE strobe */
      0,                 /* read setup cycles */
      2,                 /* read strobe cycles */
      1,                 /* read hold cycles */
      false,             /* disable page mode */
      false,             /* disable prefetch */
      false,             /* disable half cycle REn strobe */
      0,                 /* write setup cycles */
      2,                 /* write strobe cycles */
      1,                 /* write hold cycles */
      false,             /* enable the write buffer */
      false,             /* disable half cycle WEn strobe */
      ebiALowA24,        /* ALB - Low bound, address lines */
      ebiAHighA26,       /* APEN - High bound, address lines */
      ebiLocation1,      /* Use Location 1 */
      true,              /* enable EBI */
  };

  /* Enable clocks */
  CMU_ClockEnable(cmuClock_GPIO, true);
  CMU_ClockEnable(cmuClock_EBI, true);

  /* Enable GPIO's */
  /* ALE and CLE */
  GPIO_PinModeSet(gpioPortC, 1, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortC, 2, gpioModePushPull, 0);

  /* WP, CE and R/B */
  GPIO_PinModeSet(gpioPortD, 13, gpioModePushPull, 0);   /* active low write-protect */
  GPIO_PinModeSet(gpioPortD, 14, gpioModePushPull, 1);   /* active low chip-enable */
  GPIO_PinModeSet(gpioPortD, 15, gpioModeInput, 0);      /* ready/busy */

  /* IO pins */
  GPIO_PinModeSet(gpioPortE, 8, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortE, 9, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortE, 10, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortE, 11, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortE, 12, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortE, 13, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortE, 14, gpioModePushPull, 0);
  GPIO_PinModeSet(gpioPortE, 15, gpioModePushPull, 0);

  /* WE and RE */
  GPIO_PinModeSet(gpioPortF, 8, gpioModePushPull, 1);
  GPIO_PinModeSet(gpioPortF, 9, gpioModePushPull, 1);

  /* NAND Power Enable */
  GPIO_PinModeSet(gpioPortB, 15, gpioModePushPull, 1);

  EBI_Init(&ebiConfig);
  EBI->NANDCTRL = (EBI_NANDCTRL_BANKSEL_BANK0 | EBI_NANDCTRL_EN);
}

/**************************************************************************//**
 * @brief Initialize NAND flash.
 *****************************************************************************/
void initNandFlash(void)
{
  /* Setup EBI for NAND Flash. */
  nandFlashEbiInit();
  
  /* Initialize nand flash module, use DMA channel 5. */
  NANDFLASH_Init( 5 );
}

/**************************************************************************//**
 * @brief Erase NAND Flash block.
 *
 * @param[in] blockNum
 *   The block to erase.
 *****************************************************************************/
void nandFlashEraseBlock(uint32_t blockNum)
{
  /* Erase a block */
  int status;
  uint32_t addr;

  addr = BLOCKNUM_2_ADDR( blockNum );

  if ( !NANDFLASH_AddressValid( addr ) )
  {
    printf( "Invalid block %ld\n", blockNum );
  }
  else
  {
    status = NANDFLASH_EraseBlock( addr );
    if ( status != NANDFLASH_STATUS_OK )
    {
      printf( "Block erase error %d\n", status );
    }
  }
}

/**************************************************************************//**
 * @brief Read NAND Flash page into a buffer.
 *
 * @param[in] pageNum
 *   The page to read at.
 * @param[in] *buffer
 *   Buffer to read page.
 *****************************************************************************/
void nandFlashReadPage(uint32_t pageNum, uint8_t *buffer)
{
  /* Read a page */
  int status;
  uint32_t addr;

  addr = PAGENUM_2_ADDR( pageNum );

  if ( !NANDFLASH_AddressValid( addr ) )
  {
    printf( "Invalid page %ld\n", pageNum );
  }
  else
  {
    status = NANDFLASH_ReadPage( addr, buffer );
    if ( (status != NANDFLASH_STATUS_OK) && (status != NANDFLASH_ECC_UNCORRECTABLE) )
    {
      printf( "Page read error %d\n", status );
    }
  }
}

/**************************************************************************//**
 * @brief Write a buffer into NAND Flash page.
 *
 * @param[in] pageNum
 *   The page to write at.
 * @param[in] *buffer
 *   Buffer to write page.
 *****************************************************************************/
void nandFlashWritePage(uint32_t pageNum, uint8_t *buffer)
{
  /* Write a page */
  int status;
  uint32_t addr;

  addr = PAGENUM_2_ADDR( pageNum );

  if ( !NANDFLASH_AddressValid( addr ) )
  {
    printf( "Invalid page %ld\n", pageNum );
  }
  else
  {
    status = NANDFLASH_WritePage( addr, buffer );

    if ( status != NANDFLASH_STATUS_OK )
    {
      printf( "Page write error %d\n", status );
    }
  }
}
