/**************************************************************************//**
 * @file  spiflash.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "em_cmu.h"
#include "em_gpio.h"
#include "em_usart.h"
#include "spiflash.h"

static volatile bool flashBusy;

static void spiFlashWriteEnable(void);  
static void spiFlashWaitInProgress(void); 
static uint8_t spiFlashSendByte(uint8_t data);
static uint16_t spiFlashSendTwoByte(uint16_t data);
static uint8_t spiFlashReadByte(void);

/**************************************************************************//**
 * @brief Initialize USART for SPI flash.
 *****************************************************************************/
void initSpiFlash(void)
{
  USART_InitSync_TypeDef init = USART_INITSYNC_DEFAULT;

  CMU_ClockEnable(USART_CMUCLK, true);

  /* Configure SPI */  
  init.msbf = true;
  init.baudrate = CMU_ClockFreqGet(cmuClock_HFPER)/4;

  /* Configure USART for SPI */
  USART_InitSync(USART_USED, &init);

  /* Setup USART pins */
  GPIO_PinModeSet(USART_TXPORT, USART_TXPIN, gpioModePushPull, 0); 
  GPIO_PinModeSet(USART_RXPORT, USART_RXPIN, gpioModeInput, 0); 
  GPIO_PinModeSet(USART_CLKPORT, USART_CLKPIN, gpioModePushPull, 0);
  GPIO_PinModeSet(USART_CS_PORT, USART_CS_PIN, gpioModePushPull, 1);	

  /* Enabling pins and setting location, use manual control on CS */
#if defined( USART_ROUTEPEN_RXPEN )
  USART_USED->ROUTEPEN = USART_ROUTEPEN_RXPEN | USART_ROUTEPEN_TXPEN | USART_ROUTEPEN_CLKPEN;
  USART_USED->ROUTELOC0 = ( USART_USED->ROUTELOC0 &
                         ~( _USART_ROUTELOC0_TXLOC_MASK
                          | _USART_ROUTELOC0_CLKLOC_MASK
                          | _USART_ROUTELOC0_RXLOC_MASK ) )
                          | ( USART_TXLOC << _USART_ROUTELOC0_TXLOC_SHIFT )
                          | ( USART_CLKLOC << _USART_ROUTELOC0_CLKLOC_SHIFT )
                          | ( USART_RXLOC << _USART_ROUTELOC0_RXLOC_SHIFT );
#else  
  USART_USED->ROUTE = USART_ROUTE_TXPEN | USART_ROUTE_RXPEN | USART_ROUTE_CLKPEN | USART_LOCATION;
#endif
}

/***************************************************************************//**
 * @brief Reads Flash ID.
 * @return
 *   Flash ID.
 ******************************************************************************/
uint32_t spiFlashReadId(void)
{   	 
  uint32_t flashId, first, second, third;

  /* Check Flash is busy or not */
  spiFlashWaitInProgress();	

  /* Send read ID command, read 3 bytes from the Flash */
  FLASH_CS_CLR();		
  spiFlashSendByte(RDID);
  first = spiFlashReadByte();
  second = spiFlashReadByte();
  third = spiFlashReadByte();
  FLASH_CS_SET();	
         
  flashId = (first << 16) | (second << 8) | third;
  return flashId;
}

/**************************************************************************//**
 * @brief Erases the specified Flash sector or block.
 * @param eraseMode
 *   Sector or Block erase.
 * @param sectorBlockAddr
 *   Sector or block address to erase.
 *   It doesn't check the address is valid or not.
 *****************************************************************************/
void spiFlashSectorBlockErase(uint8_t eraseMode, uint32_t sectorBlockAddr)
{
  /* Check Flash is busy or not */
  spiFlashWaitInProgress();	
  
  /* Send write enable command */
  spiFlashWriteEnable();
  
  /* Send Sector Erase command, follow by 24 bit address (MSB first) */
  /* SE or BE + A23-16 + A15-A8 + A7-A0 */
  FLASH_CS_CLR();		
  spiFlashSendTwoByte(((sectorBlockAddr & 0xFF0000) >> 8) + eraseMode);
  spiFlashSendTwoByte(((sectorBlockAddr & 0xFF) << 8) + ((sectorBlockAddr & 0xFF00) >> 8));
  FLASH_CS_SET();
  flashBusy = true;
}

/**************************************************************************//**
 * @brief Erases chip.
 *****************************************************************************/
void spiFlashChipErase(void)
{ 	
  /* Check Flash is busy or not */
  spiFlashWaitInProgress();	

  /* Send write enable command */
  spiFlashWriteEnable();	
	
  /* Send Chip Erase command  */
  FLASH_CS_CLR();		
  spiFlashSendByte(CERASE);
  FLASH_CS_SET();	
  flashBusy = true;
}

/**************************************************************************//**
 * @brief Writes data from buffer to Flash page (within page boundary).
 * @param writeBuffer
 *   Pointer to the buffer for Flash write.
 * @param startWriteAddr
 *   Start address to write to Flash.
 * @param numOfWrite
 *   Number of bytes to write to Flash, must <= Flash page size.
 *****************************************************************************/
void spiFlashPageWrite(uint8_t* writeBuffer, uint32_t startWriteAddr, uint16_t numOfWrite)
{
  uint16_t* writeDouble;
  
  /* Check Flash is busy or not */
  spiFlashWaitInProgress();	

  /* Send write enable command */
  spiFlashWriteEnable();		
  
  /* Send write command, follow by 24 bit address (MSB first) */
  FLASH_CS_CLR();		
  spiFlashSendTwoByte(((startWriteAddr & 0xFF0000) >> 8) + WRITE);
  spiFlashSendTwoByte(((startWriteAddr & 0xFF) << 8) + ((startWriteAddr & 0xFF00) >> 8));

  /* Send data continuously without read */
  if (numOfWrite & 0x01)
  {
    /* Odd number use byte transfer */
    USART_USED->TXDATA = *writeBuffer++;
    while(--numOfWrite) 
    {
      while (!(USART_USED->STATUS & USART_STATUS_TXBL))
        ;
      USART_USED->TXDATA = *writeBuffer++;
    }
  }
  else
  {
    /* Even number use 16 bit transfer, number of bytes/2 */
    numOfWrite >>= 1;
    writeDouble = (uint16_t *)(writeBuffer);
    
    USART_USED->TXDOUBLE = *writeDouble++;
    while(--numOfWrite) 
    {
      while (!(USART_USED->STATUS & USART_STATUS_TXBL))
        ;
      USART_USED->TXDOUBLE = *writeDouble++;
    } 
  }

  /* Wait last byte to send and flush RX buffer */
  while (!(USART_USED->STATUS & USART_STATUS_TXC))
    ;
  USART_USED->CMD = USART_CMD_CLEARRX;
  FLASH_CS_SET();	
  flashBusy = true;
}

/**************************************************************************//**
 * @brief Writes data from buffer to Flash sector (fix size).
 * @param writeBuffer
 *   Pointer to the buffer for Flash write.
 * @param startWriteAddr
 *   Start address to write to Flash.
 *   No checking on start address is sector aligned or not.
 *****************************************************************************/
void spiFlashSectorWrite(uint8_t* writeBuffer, uint32_t startWriteAddr)
{
  uint32_t writeCount = SPIF_SECTORSIZE/SPIF_PAGESIZE;
  
  while(writeCount--)
  {
    spiFlashPageWrite(writeBuffer, startWriteAddr, SPIF_PAGESIZE);
    writeBuffer += SPIF_PAGESIZE;  
    startWriteAddr +=  SPIF_PAGESIZE;
  }
}

/**************************************************************************//**
 * @brief Writes data from buffer to Flash with across page boundary support. 
 * @param writeBuffer
 *   Pointer to the buffer for Flash write.
 * @param startWriteAddr
 *   Start address to write to Flash.
 * @param numOfWrite
 *   Number of bytes to write to Flash, it is limited by size of buffer.
 *****************************************************************************/
void spiFlashBufferWrite(uint8_t* writeBuffer, uint32_t startWriteAddr, uint16_t numOfWrite)
{
  uint16_t numOfPage, numOfFirst, numOfLast;

  numOfFirst = SPIF_PAGESIZE - (startWriteAddr % SPIF_PAGESIZE);
  numOfPage =  numOfWrite / SPIF_PAGESIZE;
  numOfLast = numOfWrite % SPIF_PAGESIZE;
  
  if(numOfFirst == SPIF_PAGESIZE)
  {
    /* Start address is page aligned */
    if(numOfPage == 0)
    {
      /* Bytes to write < page size and within one page */
      spiFlashPageWrite(writeBuffer, startWriteAddr, numOfWrite);
    }
    else 
    {
      /* Bytes to write > page size and across page boundary */
      while(numOfPage--)
      {
        spiFlashPageWrite(writeBuffer, startWriteAddr, SPIF_PAGESIZE);
        writeBuffer += SPIF_PAGESIZE;  
        startWriteAddr +=  SPIF_PAGESIZE;
      }    
      if(numOfLast)
      {
        spiFlashPageWrite(writeBuffer, startWriteAddr, numOfLast);
      }
    }
  }
  else
  {
    /* Start address is not page aligned */
    if(numOfPage == 0)
    {
      /* Bytes to write < page size */
      if(numOfLast > numOfFirst)
      {
        /* Byes to write + start address > page size, across page boundary */
        spiFlashPageWrite(writeBuffer, startWriteAddr, numOfFirst);
        writeBuffer += numOfFirst; 
        startWriteAddr +=  numOfFirst;
        spiFlashPageWrite(writeBuffer, startWriteAddr, (numOfLast - numOfFirst));
      }
      else
      {
        /* Bytes to write and start address are within one page */
        spiFlashPageWrite(writeBuffer, startWriteAddr, numOfWrite);
      }
    }
    else
    {
      /* Bytes to write > page size */
      numOfWrite -= numOfFirst;
      numOfPage =  numOfWrite / SPIF_PAGESIZE;
      numOfLast = numOfWrite % SPIF_PAGESIZE;	
      
      spiFlashPageWrite(writeBuffer, startWriteAddr, numOfFirst);
      writeBuffer += numOfFirst;  
      startWriteAddr +=  numOfFirst;
     
      while(numOfPage--)
      {
        spiFlashPageWrite(writeBuffer, startWriteAddr, SPIF_PAGESIZE);
        writeBuffer += SPIF_PAGESIZE;  
        startWriteAddr +=  SPIF_PAGESIZE;
      }
      
      if(numOfLast)
      {
        spiFlashPageWrite(writeBuffer, startWriteAddr, numOfLast);
      }
    }
  }      	
}

/**************************************************************************//**
 * @brief Set start address to read data from Flash.
 * @param startReadAddr
 *   Start address to read from flash, CS keeps LOW at return.
 *****************************************************************************/
void spiFlashSetReadAddr(uint32_t startReadAddr)
{
  /* Check Flash is busy or not */
  spiFlashWaitInProgress();	

  /* Send read command, follow by 24 bit address (MSB first) */
  FLASH_CS_CLR();		
  spiFlashSendTwoByte(((startReadAddr & 0xFF0000) >> 8) + READ);
  spiFlashSendTwoByte(((startReadAddr & 0xFF) << 8) + ((startReadAddr & 0xFF00) >> 8));
}

/**************************************************************************//**
 * @brief Reads data from Flash and save it to buffer.
 * @param readBuffer
 *   Pointer to the buffer to save data, buffer size must be multiple of 2.
 * @param startReadAddr
 *   Start address to read from Flash.
 * @param numOfRead
 *   Number of bytes to read from Flash, must be multiple of 2.
 *****************************************************************************/
void spiFlashBufferRead(uint8_t* readBuffer, uint32_t startReadAddr, uint16_t numOfRead)
{
  uint16_t* readDouble;

  /* Check Flash is busy or not */
  spiFlashWaitInProgress();	

  /* Send read command, follow by 24 bit address (MSB first) */
  FLASH_CS_CLR();		
  spiFlashSendTwoByte(((startReadAddr & 0xFF0000) >> 8) + READ);
  spiFlashSendTwoByte(((startReadAddr & 0xFF) << 8) + ((startReadAddr & 0xFF00) >> 8));

  /* Use 16 bit transfer, number of bytes/2 */
  numOfRead >>= 1;
  readDouble = (uint16_t *)(readBuffer);
  USART_USED->CMD = USART_CMD_CLEARTX + USART_CMD_CLEARRX;

#if (USE_AUTOTX == 0)
  USART_USED->TXDOUBLE = 0xffff;
  do
  {
    USART_USED->TXDOUBLE = 0xffff;
    while (!(USART_USED->STATUS & USART_STATUS_RXFULL))
      ;
    *readDouble++ = (uint16_t)(USART_USED->RXDOUBLE);
  } while(--numOfRead);
#else
  /* Check errata when using AUTOTX feature */
  /* Make sure it is fast enough to fetch data to avoid overflow */
  USART_USED->CTRL |= USART_CTRL_AUTOTX;
  do
  {
    while (!(USART_USED->STATUS & USART_STATUS_RXFULL))
      ;
    *readDouble++ = (uint16_t)(USART_USED->RXDOUBLE);
  } while(--numOfRead);
  USART_USED->CTRL &= ~USART_CTRL_AUTOTX;
#endif
  while (!(USART_USED->STATUS & USART_STATUS_TXC))
    ;
  USART_USED->CMD = USART_CMD_CLEARTX + USART_CMD_CLEARRX;
  FLASH_CS_SET();	
}

/**************************************************************************//**
 * @brief Send write enable command to SPI Flash.
 *****************************************************************************/
static void spiFlashWriteEnable(void)  
{
  /* Send write enable command */
  FLASH_CS_CLR();		
  spiFlashSendByte(WREN);
  FLASH_CS_SET();	
}

/**************************************************************************//**
 * @brief Poll WIP bit in read status register without timeout.
 *****************************************************************************/
static void spiFlashWaitInProgress(void) 
{
  uint8_t statusReg;
  
  if (!flashBusy)
  {
    return;
  }
  
  /* Send read status register command, check Flash busy or not */
  FLASH_CS_CLR();		
  spiFlashSendByte(RDSR);
  do
  { 	
    statusReg = spiFlashReadByte();  
  } while((statusReg & WIP));  
  FLASH_CS_SET();
  flashBusy = false;
}

/***************************************************************************//**
 * @brief  Sends one byte to SPI Flash without timeout.
 * @param  data
 *   Data to transmit.
 * @return
 *   Data received.
 ******************************************************************************/
static uint8_t spiFlashSendByte(uint8_t data)
{
  USART_USED->TXDATA = data;
  while (!(USART_USED->STATUS & USART_STATUS_RXDATAV))
    ;
  /* Flush RX buffer even data is not required */
  return (uint8_t)(USART_USED->RXDATA);
}

/***************************************************************************//**
 * @brief  Sends two bytes to SPI Flash without timeout.
 * @param  data
 *   Data to transmit.
 * @return
 *   Data received.
 ******************************************************************************/
static uint16_t spiFlashSendTwoByte(uint16_t data)
{
  USART_USED->TXDOUBLE = data;
  while (!(USART_USED->STATUS & USART_STATUS_RXFULL))
    ;
  /* Flush RX buffer even data is not required */
  return (uint16_t)(USART_USED->RXDOUBLE);
}

/***************************************************************************//**
 * @brief Reads one byte form SPI Flash without timeout.
 * @return
 *   Data received.
 ******************************************************************************/
static uint8_t spiFlashReadByte(void)
{
  USART_USED->TXDATA = 0xff;
  /* Data should be transmitted if RXDATAV = 1 */
  while (!(USART_USED->STATUS & USART_STATUS_RXDATAV))
    ;
  return (uint8_t)(USART_USED->RXDATA);
}
