/**************************************************************************//**
 * @file  usbtomemory.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdio.h>
#include "em_msc.h"
#include "em_usb.h"
#include "ff.h"
#include "msdh.h"
#include "kits.h"
#include "progconfig.h"
#include "usbconfig.h"
#include "usbtomemory.h"
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
#include "nandflash.h"
#include "nandutil.h"
#else
#include "spiflash.h"
#endif

/* File system */
FATFS Fatfs;
FIL binFile;
FRESULT res;

/* USB related data */
STATIC_UBUF( tmpBuf, 1024 );

/* Buffer for external memory data read */
EFM32_ALIGN(4)
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
uint8_t memoryReadBuffer[NAND_FLASH_BLOCK_SIZE];
#else
uint8_t memoryReadBuffer[SPIF_SECTORSIZE];
#endif

char deviceName[DEVICE_NAME_SIZE];
uint8_t mainFileName[FILE_NAME_SIZE];
uint8_t userFileName[FILE_NAME_SIZE];

volatile uint16_t crcTemp;
volatile bool newFamily;
volatile uint32_t chipType;

volatile uint32_t mainFlashSize;
volatile uint32_t mainPageSize;
volatile uint32_t mainPageSizeMask;
volatile uint32_t mainFileSize;
volatile uint32_t mainStartAddr;

volatile uint32_t userPageSize;
volatile uint32_t userFileSize;
volatile uint32_t userStartAddr;

volatile bool useIntMem;
volatile bool intMemValid;
volatile bool progUserPage;
volatile bool progMainFlash;
volatile bool debugLockAfterProg;
volatile bool aapLockAfterProg;

static uint16_t CRC_calc(uint8_t *start, uint8_t *end);
static uint32_t hexToDecimal(uint8_t *hexString);
static void intMemCrc(uint32_t crcVerify, uint32_t binFileSize);
static bool extMemCrc(uint32_t pageOffset, uint32_t crcVerify, uint32_t binFileSize);
static bool getFileInfo(void);
static bool saveBinaryToExtMem(bool binaryType);
static void extToIntMem(void);
static bool usbFileToExtMem(void);

/**************************************************************************//**
 * @brief
 *   This function calculates the CRC-16-CCIT checksum of a memory range.
 *  
 * @note
 *   This implementation uses an initial value of 0, while some implementations
 *   of CRC-16-CCIT uses an initial value of 0xFFFF. If you wish to
 *   precalculate the CRC before uploading the binary to the bootloader you 
 *   can use this function. However, keep in mind that the 'v' and 'c' commands
 *   computes the crc of the entire flash, so any bytes not used by your
 *   application will have the value 0xFF.
 *
 * @param start
 *   Pointer to the start of the memory block
 *
 * @param end
 *   Pointer to the end of the block. This byte is not included in the computed
 *   CRC.
 *
 * @return
 *   The computed CRC value.
 *****************************************************************************/
static uint16_t CRC_calc(uint8_t *start, uint8_t *end)
{
  uint16_t crc = crcTemp;
  uint8_t *data;

  for (data = start; data < end; data++)
  {
    crc  = (crc >> 8) | (crc << 8);
    crc ^= *data;
    crc ^= (crc & 0xff) >> 4;
    crc ^= crc << 12;
    crc ^= (crc & 0xff) << 5;
  }
  return crc;
}

/**************************************************************************//**
 * @brief Function to convert hexadecimal to decimal.
 * @param *hexString
 *   Pointer to hexadecimal digits, ' ' as termination character.
 * @return The computed decimal value. 
 *****************************************************************************/
static uint32_t hexToDecimal(uint8_t *hexString)
{
  uint32_t i, t, decimal=0;

  /* 0-9 = 0x30-0x39, A-F = 0x41-0x46, a-f = 0x61-0x66 */
  for(i=0; hexString[i] != ' '; i++)
  {
    if(hexString[i] <= '9')
    {
      t = hexString[i] - '0';
    }
    else if(hexString[i] < 'a')
    {
      t = hexString[i] - 'A' + 10;
    }
    else 
    {
      t = hexString[i] - 'a' + 10;
    }
    decimal = (decimal << 4) + t;
  }
  return decimal;
}

/**************************************************************************//**
 * @brief Function to calculate and verify CRC in internal memory.
 * @param crcVerify
 *   CRC for verification
 * @param binFileSize
 *   Size of binary file
 *****************************************************************************/
static void intMemCrc(uint32_t crcVerify, uint32_t binFileSize)
{
  uint32_t i;
  uint8_t *intPtr = (uint8_t *)(INT_START_ADDR);

  useIntMem = false;
  if (!intMemValid)
  {
    return;
  }
  
  i = 0;
  crcTemp = 0;

  /* Use minimum external memory page size to calculate CRC */
  while (i != binFileSize)
  {
    crcTemp = CRC_calc(intPtr, (intPtr + MIN_PAGE_SIZE));
    i += MIN_PAGE_SIZE;
    intPtr += MIN_PAGE_SIZE;
  }

  if (crcTemp == crcVerify)
  {
    useIntMem = true;
  }
}

/**************************************************************************//**
 * @brief Function to calculate and verify CRC in external memory.
 * @param pageOffset
 *   Page start address
 * @param crcVerify
 *   CRC for verification
 * @param binFileSize
 *   Size of binary file
 * @return True or False 
 *****************************************************************************/
static bool extMemCrc(uint32_t pageOffset, uint32_t crcVerify, uint32_t binFileSize)
{
  uint32_t pageNumAddr = 0;

  crcTemp = 0;
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
  while ((pageNumAddr*NAND_FLASH_PAGE_SIZE) != binFileSize)
  {
    nandFlashReadPage(pageNumAddr+pageOffset, memoryReadBuffer);
    crcTemp = CRC_calc(memoryReadBuffer, (memoryReadBuffer + NAND_FLASH_PAGE_SIZE));
    pageNumAddr++;
  }
#else
  while (pageNumAddr != binFileSize)
  {
    spiFlashBufferRead(memoryReadBuffer, pageNumAddr+pageOffset, SPIF_PAGESIZE);
    crcTemp = CRC_calc(memoryReadBuffer, (memoryReadBuffer + SPIF_PAGESIZE));
    pageNumAddr +=  SPIF_PAGESIZE;
  }
#endif
  
  if (crcTemp == crcVerify)
  {
    return true;
  }
  else
  {
    return false;
  }
}

/**************************************************************************//**
 * @brief Get binary file information
 * @return True or False 
 *****************************************************************************/
static bool getFileInfo(void)
{
  uint32_t i, j;

  /* Reset status */
  progMainFlash = false;
  progUserPage = false; 

  if ((memoryReadBuffer[0] >= 'a' && memoryReadBuffer[0] <= 'z') 
      || (memoryReadBuffer[0] >= 'A' && memoryReadBuffer[0] <= 'Z'))
  {
    progMainFlash = true;
  }
  if ((memoryReadBuffer[BUFFER_OFFSET] >= 'a' && memoryReadBuffer[BUFFER_OFFSET] <= 'z') 
      || (memoryReadBuffer[BUFFER_OFFSET] >= 'A' && memoryReadBuffer[BUFFER_OFFSET] <= 'Z'))
  {
    progUserPage = true;
  }
  if (!(progMainFlash || progUserPage))
  {
    return false;
  }
  
  /* Get first file name */
  if (progMainFlash)
  {
    for (i=0, j=0; i<NAME_INDEX_MAX; i++, j++)
    {
      if (memoryReadBuffer[i] != ' ')
      {
        mainFileName[j] = memoryReadBuffer[i];
      }
      else
      {
        mainFileName[j++] = '.';
        mainFileName[j++] = 'b';
        mainFileName[j++] = 'i';
        mainFileName[j++] = 'n';
        mainFileName[j++] = '\0';
        break;
      }
    }
    
    if (i == NAME_INDEX_MAX)
    {
      printf("File name too long\n");
      progMainFlash = false;
      progUserPage = false;
      return false;
    }
    
    /* Get first program start address, add space for hex to decimal */
    i++;
    memoryReadBuffer[i + ADDRESS_LENGTH] = ' ';
    mainStartAddr = hexToDecimal(memoryReadBuffer + i);
    
    /* User page only if address is user page, no debug lock info */
    if (mainStartAddr == USER_PAGE_ADDR)
    {
      memoryReadBuffer[i + ADDRESS_LENGTH] = '\n';
      memoryReadBuffer[i + ADDRESS_LENGTH + 1] = '\0';
      progMainFlash = false;
      progUserPage = true;

      /* Copy main information to user */
      memcpy(userFileName, mainFileName, FILE_NAME_SIZE);
      userStartAddr = mainStartAddr;
      return true;
    }
    
    /* Get debug lock information */
    i = i + ADDRESS_LENGTH + 1;
    debugLockAfterProg = false;
    if ((memoryReadBuffer[i] == 'Y') || (memoryReadBuffer[i] == 'y'))
    {
      debugLockAfterProg = true;
    }
    aapLockAfterProg = false;
    if ((memoryReadBuffer[i] == 'A') || (memoryReadBuffer[i] == 'a'))
    {
      aapLockAfterProg = true;
    }
    memoryReadBuffer[++i] = '\n';
    memoryReadBuffer[++i] = '\0';
  }  

  /* Get second file name */
  if (progUserPage)
  {
    for (i=BUFFER_OFFSET, j=0; i<(BUFFER_OFFSET + NAME_INDEX_MAX); i++, j++)
    {
      if (memoryReadBuffer[i] != ' ')
      {
        userFileName[j] = memoryReadBuffer[i];
      }
      else
      {
        userFileName[j++] = '.';
        userFileName[j++] = 'b';
        userFileName[j++] = 'i';
        userFileName[j++] = 'n';
        userFileName[j++] = '\0';
        break;
      }
    }

    if (i == (BUFFER_OFFSET + NAME_INDEX_MAX))
    {
      printf("File name too long\n");
      progMainFlash = false;
      progUserPage = false;
      return false;
    }
    
    /* Get user page start address, add space for hex to decimal */
    i++;
    memoryReadBuffer[i + ADDRESS_LENGTH] = ' ';
    userStartAddr = hexToDecimal(memoryReadBuffer + i);
    memoryReadBuffer[i + ADDRESS_LENGTH] = '\n';
    memoryReadBuffer[i + ADDRESS_LENGTH + 1] = '\0';
  }
  return true;
}

/**************************************************************************//**
 * @brief Get binary file information from NAND flash or SPI NOR flash
 * @return True or False 
 *****************************************************************************/
bool getBinInfoFromExtMem(void)
{
  uint16_t crcMain, crcUser;
  uint32_t* ptrTemp = (uint32_t *)memoryReadBuffer; 
  
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
  nandFlashReadPage(MAIN_INFO_ADDR_PAGE, memoryReadBuffer);
  nandFlashReadPage(USER_INFO_ADDR_PAGE, memoryReadBuffer+BUFFER_OFFSET);
#else
  spiFlashBufferRead(memoryReadBuffer, MAIN_INFO_ADDR, MAIN_INFO_SIZE);
  spiFlashBufferRead(memoryReadBuffer+BUFFER_OFFSET, USER_INFO_ADDR, USER_INFO_SIZE);
#endif
  
  /* Retrieve file name, start address... */
  if (!getFileInfo())
  {
    return false;
  }
  displayClear();
  
  /* Retrieve file size and check CRC */
  if (progMainFlash)
  {
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    nandFlashReadPage(MAIN_INFO_ADDR_PAGE, memoryReadBuffer);
    printf("%s", memoryReadBuffer);
    nandFlashReadPage(MAIN_FILE_ADDR_PAGE, memoryReadBuffer);
    mainFileSize = *ptrTemp;
    crcMain = *(uint16_t *)(memoryReadBuffer+4);
    printf("S:%lu C:%04x\n", mainFileSize, crcMain);
    intMemCrc(crcMain, mainFileSize);
    if (!extMemCrc(MAIN_DATA_ADDR_PAGE, crcMain, mainFileSize))
    {
      return false;
    }
#else
    spiFlashBufferRead(memoryReadBuffer, MAIN_INFO_ADDR, MAIN_INFO_SIZE);
    printf("%s", memoryReadBuffer);
    spiFlashBufferRead(memoryReadBuffer, MAIN_FILE_ADDR, MAIN_FILE_SIZE);
    mainFileSize = *ptrTemp;
    crcMain = *(uint16_t *)(memoryReadBuffer+4);
    printf("S:%lu C:%04x\n", mainFileSize, crcMain);
    intMemCrc(crcMain, mainFileSize);
    if (!extMemCrc(MAIN_DATA_SECTOR, crcMain, mainFileSize))
    {
      return false;
    }
#endif
  }
  
  if (progUserPage)
  {
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    nandFlashReadPage(USER_INFO_ADDR_PAGE, memoryReadBuffer);
    printf("%s", memoryReadBuffer);
    nandFlashReadPage(USER_FILE_ADDR_PAGE, memoryReadBuffer);
    userFileSize = *ptrTemp;
    crcUser = *(uint16_t *)(memoryReadBuffer+4);
    printf("S:%lu C:%04x\n", userFileSize, crcUser);
    if (!extMemCrc(USER_DATA_ADDR_PAGE, crcUser, userFileSize))
    {
      return false;
    }
#else
    spiFlashBufferRead(memoryReadBuffer, USER_INFO_ADDR, USER_INFO_SIZE);
    printf("%s", memoryReadBuffer);
    spiFlashBufferRead(memoryReadBuffer, USER_FILE_ADDR, USER_FILE_SIZE);
    userFileSize = *ptrTemp;
    crcUser = *(uint16_t *)(memoryReadBuffer+4);
    printf("S:%lu C:%04x\n", userFileSize, crcUser);
    if (!extMemCrc(USER_DATA_SECTOR, crcUser, userFileSize))
    {
      return false;
    }
#endif
  }
  return true;  
}

/**************************************************************************//**
 * @brief Save binary file from USB flash drive to NAND flash or SPI NOR flash
 * @param binaryType
 *   True for main flash binary, false for user page binary
 * @return True or False 
 *****************************************************************************/
static bool saveBinaryToExtMem(bool binaryType)
{
  uint16_t byteRead;
  uint32_t i, binFileSize;
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
  uint32_t pageNum, blockNum;
#else
  uint32_t pageAddr, sectorAddr;
#endif

  /* Check binary is main flash or user page then open */
  if (binaryType)
  {
    res = f_open(&binFile, (char const *)mainFileName, FA_OPEN_EXISTING | FA_READ);
  }
  else
  {
    res = f_open(&binFile, (char const *)userFileName, FA_OPEN_EXISTING | FA_READ);
  }
  
  if (res)
  {
    printf("No binary file:%d\n", res);
    return false;
  }
  
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
  /* Check binary is main flash or user page */
  if (binaryType)
  {
    pageNum = MAIN_DATA_ADDR_PAGE;
    blockNum = MAIN_DATA_ADDR_BLOCK;
  }
  else
  {
    pageNum = USER_DATA_ADDR_PAGE;
    blockNum = USER_DATA_ADDR_BLOCK;
  }
  crcTemp = 0;
  binFileSize = 0;
  
  while (USBH_DeviceConnected())
  {
    /* Read one block of data */
    res = f_read(&binFile, memoryReadBuffer, NAND_FLASH_BLOCK_SIZE, &byteRead);
    if (res || byteRead == 0)
    {
      /* error or eof */
      break;
    }
      
    /* Erase 1 flash block for storage, reset i to point to buffer start */
    i = 0;
    nandFlashEraseBlock(blockNum);

    /* Fill hole with 0xff if not flash page align */
    for (;;)
    {
      if (byteRead & NAND_FLASH_PAGE_SIZE_MASK)
      {
        memoryReadBuffer[byteRead] = 0xff;
        byteRead++;
      }
      else
      {
        break;
      }
    }
    
    /* Calculate binary file CRC */
    binFileSize += byteRead;
    crcTemp = CRC_calc(memoryReadBuffer, (memoryReadBuffer + byteRead));
      
    while (byteRead)
    {
      /* Write data to pages, maximum size equal to one block */
      nandFlashWritePage(pageNum, memoryReadBuffer + i);
      i += NAND_FLASH_PAGE_SIZE;
      pageNum++;
      byteRead -= NAND_FLASH_PAGE_SIZE;
    }
        
    blockNum++;
  }
#else
  if (binaryType)
  {
    pageAddr = MAIN_DATA_PAGE;
    sectorAddr = MAIN_DATA_SECTOR;
  }
  else
  {
    pageAddr = USER_DATA_PAGE;
    sectorAddr = USER_DATA_SECTOR;
  }
  crcTemp = 0;
  binFileSize = 0;
  
  while (USBH_DeviceConnected())
  {
    /* Read one sector of data */
    res = f_read(&binFile, memoryReadBuffer, SPIF_SECTORSIZE, &byteRead);
    if (res || byteRead == 0)
    {
      /* error or eof */
      break;
    }
      
    /* Erase 1 flash sector for storage, reset i to point to buffer start */
    i = 0;
    spiFlashSectorBlockErase(SERASE, sectorAddr);

    /* Fill hole with 0xff if not flash page align */
    for (;;)
    {
      if (byteRead & SPIF_PAGESIZE_MASK)
      {
        memoryReadBuffer[byteRead] = 0xff;
        byteRead++;
      }
      else
      {
        break;
      }
    }
    
    /* Calculate binary file CRC */
    binFileSize += byteRead;
    crcTemp = CRC_calc(memoryReadBuffer, (memoryReadBuffer + byteRead));
      
    while (byteRead)
    {
      /* Write data to pages, maximum size equal to one sector */
      spiFlashPageWrite((memoryReadBuffer + i), pageAddr, SPIF_PAGESIZE);
      i += SPIF_PAGESIZE;  
      pageAddr +=  SPIF_PAGESIZE;
      byteRead -= SPIF_PAGESIZE;
    }
        
    sectorAddr += SPIF_SECTORSIZE;
  }
#endif

  /* Close binary file */
  f_close(&binFile);

  if (!binFileSize || res || byteRead)
  {
    printf("Save error!\n");
    return false;
  }

  /* Calculate external memory CRC */
  if (binaryType)
  {
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    i = MAIN_DATA_ADDR_PAGE;
#else
    i = MAIN_DATA_SECTOR;
#endif
  }
  else
  {
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    i = USER_DATA_ADDR_PAGE;
#else
    i = USER_DATA_SECTOR;
#endif
  }    
    
  if (extMemCrc(i, crcTemp, binFileSize))
  {
    /* Save file information */
    memoryReadBuffer[0] = binFileSize;
    memoryReadBuffer[1] = binFileSize >> 8;
    memoryReadBuffer[2] = binFileSize >> 16;
    memoryReadBuffer[3] = binFileSize >> 24;
    memoryReadBuffer[4] = crcTemp;
    memoryReadBuffer[5] = crcTemp >>8;
    memoryReadBuffer[6] = 0;
    memoryReadBuffer[7] = 0;
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    if (binaryType)
    {
      mainFileSize = binFileSize;
      nandFlashEraseBlock(MAIN_FILE_ADDR_BLOCK);
      nandFlashWritePage(MAIN_FILE_ADDR_PAGE, memoryReadBuffer);
    }
    else
    {
      userFileSize = binFileSize;
      nandFlashEraseBlock(USER_FILE_ADDR_BLOCK);
      nandFlashWritePage(USER_FILE_ADDR_PAGE, memoryReadBuffer);
    }
#else
    /* Sector had already erased since same sector for all main and user info */ 
    if (binaryType)
    {
      mainFileSize = binFileSize;
      spiFlashPageWrite(memoryReadBuffer, MAIN_FILE_ADDR, MAIN_FILE_SIZE);
    }
    else
    {
      userFileSize = userFileSize;
      spiFlashPageWrite(memoryReadBuffer, USER_FILE_ADDR, USER_FILE_SIZE);
    }
#endif
    return true;   
  }
  else
  {
    printf("CRC verify error!\n");
    return false;   
  }
}

/**************************************************************************//**
 * @brief Get data from external memory to internal memory in GG
          if file size <= 512 KB
 * @return True or False 
 *****************************************************************************/
static void extToIntMem(void)
{
  uint32_t timeOut;
  uint32_t wordIndex;
  uint32_t intMemAddr;
  uint32_t pageNumAddr;
  uint32_t *wordPtr = (uint32_t *)memoryReadBuffer;

  useIntMem = false;
  if (!intMemValid)
  {
    return;
  }

  if (mainFileSize <= INT_START_ADDR)
  {
    /* Enable memory controller with read-while-write */
    MSC->LOCK = MSC_UNLOCK_CODE;
    MSC->WRITECTRL = MSC_RWDOUBLE;

    /* Unlock mass erase and send mass erase command */
    MSC->MASSLOCK = MSC_MASSLOCK_LOCKKEY_UNLOCK;
    MSC->WRITECMD = MSC_ERASEUPPER;
    
    /* Waiting for erase to complete then lock mass erase */
    timeOut = MSC_PROGRAM_TIMEOUT;
    while ((MSC->STATUS & MSC_STATUS_BUSY) && (timeOut != 0))
    {
      timeOut--;
    }
    if (timeOut == 0)
    {
      return;
    }
    MSC->MASSLOCK = MSC_MASSLOCK_LOCKKEY_LOCK;

    /* Set write start address */
    intMemAddr = INT_START_ADDR;
    
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    pageNumAddr = MAIN_DATA_ADDR_PAGE;  
    while ((pageNumAddr*NAND_FLASH_PAGE_SIZE) != mainFileSize)
    {
      wordIndex = 0;
      nandFlashReadPage(pageNumAddr, memoryReadBuffer);
      
      /* Next page if reach boundary */
      if ((intMemAddr & INT_PAGE_MASK) == 0)
      {
        MSC->ADDRB = intMemAddr;
        MSC->WRITECMD = MSC_WRITECMD_LADDRIM;
      }
      
      while(wordIndex < NAND_FLASH_PAGE_SIZE/4)
      {
        MSC->WDATA = wordPtr[wordIndex++];
        while (!(MSC->STATUS & MSC_STATUS_WDATAREADY))
          ;
        MSC->WDATA = wordPtr[wordIndex++];
        MSC->WRITECMD = MSC_WRITECMD_WRITEONCE;
        
        /* Wait for the transaction to finish. */
        timeOut = MSC_PROGRAM_TIMEOUT;
        while ((MSC->STATUS & MSC_STATUS_BUSY) && (timeOut != 0))
        {
          timeOut--;
        }
        if (timeOut == 0)
        {
          return;
        }
      }
      pageNumAddr++;
      intMemAddr += NAND_FLASH_PAGE_SIZE;
    }
#else
    pageNumAddr = MAIN_DATA_PAGE;  
    while (pageNumAddr != mainFileSize)
    {
      wordIndex = 0;
      spiFlashBufferRead(memoryReadBuffer, pageNumAddr, SPIF_PAGESIZE);

      /* Next page if reach boundary */
      if ((intMemAddr & INT_PAGE_MASK) == 0)
      {
        MSC->ADDRB = intMemAddr;
        MSC->WRITECMD = MSC_WRITECMD_LADDRIM;
      }

      while(wordIndex < SPIF_PAGESIZE/4)
      {
        MSC->WDATA = wordPtr[wordIndex++];
        while (!(MSC->STATUS & MSC_STATUS_WDATAREADY))
          ;
        MSC->WDATA = wordPtr[wordIndex++];
        MSC->WRITECMD = MSC_WRITECMD_WRITEONCE;
        
        /* Wait for the transaction to finish. */
        timeOut = MSC_PROGRAM_TIMEOUT;
        while ((MSC->STATUS & MSC_STATUS_BUSY) && (timeOut != 0))
        {
          timeOut--;
        }
        /* Check for timeout */
        if (timeOut == 0)
        {
          return;
        }
      }
      pageNumAddr +=  SPIF_PAGESIZE;
      intMemAddr += SPIF_PAGESIZE;
    }
#endif
    MSC->WRITECTRL = 0;
    MSC->LOCK = 0;
  }
}

/**************************************************************************//**
 * @brief Get data from USB flash drive to NAND flash or SPI NOR flash
 * @return True or False 
 *****************************************************************************/
static bool usbFileToExtMem(void)
{
  /* Initialize filesystem */
  res = f_mount(0, &Fatfs);

  if (res != FR_OK)
  {
    printf("FAT-error!\n");
    return false;
  }

  /* Open text file */
  res = f_open(&binFile, TEXT_FILENAME, FA_OPEN_EXISTING | FA_READ);
  if (res)
  {
    printf("No text file!\n");
    f_mount(0, NULL);
    return false;
  }

  /* Get first file information */
  if (f_gets((char *)memoryReadBuffer, INFO_SIZE, &binFile) == NULL)
  {
    printf("File read error!\n");
    f_close(&binFile);
    f_mount(0, NULL);
    return false;
  }
  
  /* Get second file information, check valid or not later */
  f_gets((char *)(memoryReadBuffer + BUFFER_OFFSET), INFO_SIZE, &binFile);

  /* Close text file, get file information, save if valid */
  f_close(&binFile);
  if (!getFileInfo())
  {
    f_mount(0, NULL);
    return false;
  }

#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
  /* Save program main flash information */
  nandFlashEraseBlock(MAIN_INFO_ADDR_BLOCK);
  if (progMainFlash)
  {
    nandFlashWritePage(MAIN_INFO_ADDR_PAGE, memoryReadBuffer);
  }

  /* Save program user page information */
  nandFlashEraseBlock(USER_INFO_ADDR_BLOCK);
  if (progUserPage)
  {
    if (progMainFlash)
    {
      nandFlashWritePage(USER_INFO_ADDR_PAGE, memoryReadBuffer+BUFFER_OFFSET);
    }
    else
    {
      nandFlashWritePage(USER_INFO_ADDR_PAGE, memoryReadBuffer);
    }
  }
#else
  /* Save program main flash information, erase one sector for main and user */
  spiFlashSectorBlockErase(SERASE, MAIN_INFO_ADDR);
  if (progMainFlash)
  {
    spiFlashPageWrite(memoryReadBuffer, MAIN_INFO_ADDR, MAIN_INFO_SIZE);
  }

  /* Save program user page information, same sector so no need to erase */
  if (progUserPage)
  {
    if (progMainFlash)
    {
      spiFlashPageWrite(memoryReadBuffer+BUFFER_OFFSET, USER_INFO_ADDR, USER_INFO_SIZE);
    }
    else
    {
      spiFlashPageWrite(memoryReadBuffer, USER_INFO_ADDR, USER_INFO_SIZE);
    }
  }
#endif
  
  /* Save binary to external memory */
  if (progMainFlash)
  {
    if (!saveBinaryToExtMem(true))
    {
      f_mount(0, NULL);
      return false;
    }
  }

  if (progUserPage)
  {
    if (!saveBinaryToExtMem(false))
    {
      f_mount(0, NULL);
      return false;
    }
  }
  
  /* UNMOUNT drive */
  f_mount(0, NULL);
  return true;
}

/**************************************************************************//**
 * @brief Upload image from USB to memory
 * @return True or False 
 *****************************************************************************/
bool uploadImageFromUsb(void)
{
  int connectionResult;

  /* Wait for device connection foreover */
  printf("Plug in USB MSD\n");
  printf("Wait...\n");
  
  connectionResult = USBH_WaitForDeviceConnectionB(tmpBuf, USB_WAIT_TIMEOUT);

  if (connectionResult == USB_STATUS_OK)
  {
    if (MSDH_Init(tmpBuf, sizeof(tmpBuf)))
    {
      /* Save binary file to external and internal memory if available */
      if (usbFileToExtMem())
      {
        extToIntMem();
        printf("Upload image done!\n");
        return true;
      }
      else
      {
        printf("Upload image fail!\n");
      }
    }
    else
    {
      printf("MSD init error!\n");
    }
  }
  else if (connectionResult == USB_STATUS_TIMEOUT)
  {
    printf("No device plug-in\n");
    setErrorLed(true);
    return false;
  }
  else if (connectionResult == USB_STATUS_DEVICE_MALFUNCTION)
  {
    printf("Device can't detect!\n");
  }
  
  setErrorLed(true);
  printf("Remove device!\n");
  while (USBH_DeviceConnected())
  {
    ;
  }
  printf("Device remove!\n");
  return false;
}

/**************************************************************************//**
 * @brief Initialize USB HOST stack
 *****************************************************************************/
void initUsbHost(void)
{
  USBH_Init_TypeDef is = USBH_INIT_DEFAULT;
  USBH_Init(&is);
}

/***************************************************************************//**
 * @brief
 *   This function is required by the FAT file system in order to provide
 *   timestamps for created files. Since we do not have a reliable clock we
 *   hardcode a value here.
 *
 *   Refer to reptile/fatfs/doc/en/fattime.html for the format of this DWORD.
 * @return
 *    A DWORD containing the current time and date as a packed datastructure.
 ******************************************************************************/
DWORD get_fattime( void )
{
  return (28 << 25) | (2 << 21) | (1 << 16);
}
