/**************************************************************************//**
 * @file  menuutil.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "em_emu.h"
#include "em_gpio.h"
#include "kits.h"
#include "delay.h"
#include "menuutil.h"
#include "progconfig.h"
#include "usbtomemory.h"
#include "state_machine.h"

uint32_t progDevCnt;
volatile bool pressPb0 = true;
volatile bool pressPb1;
volatile uint8_t menuLevel = ERASE_USER_MENU;
volatile bool gangMode4;
volatile bool gangMode8;

static void progSequence(void);
static void eraseMainSequence(void);
static void eraseUserSequence(void);

/******************************************************************************
 * @brief  Menu utility
 *****************************************************************************/
void menuSelect(void)
{
  bool subMenuExit = false;
  
  while (1)
  {
    if (!subMenuExit)
    {
      EMU_EnterEM1();
    }
    subMenuExit = false;
    
    if (pressPb0)
    {
      __disable_irq();
      pressPb0 = false;
      __enable_irq();

      switch (menuLevel)
      {
      case PROG_MENU:
        menuLevel++;
        menuRow0();
        menuRow1Sel();
        menuRow2();
        menuRow3();
        break;
        
      case UPLOAD_MENU:
        menuLevel++;
        menuRow0();
        menuRow1();
        menuRow2Sel();
        menuRow3();
        break;
        
      case ERASE_MAIN_MENU:
        menuLevel++;
        menuRow0();
        menuRow1();
        menuRow2();
        menuRow3Sel();
        break;
        
      case ERASE_USER_MENU:
        menuLevel = PROG_MENU;
        menuRow0Sel();
        menuRow1();
        menuRow2();
        menuRow3();
        break;

      case PROG_SUB_MENU:
      case UPLOAD_SUB_MENU:
      case ERASE_MAIN_SUB_MENU:
      case ERASE_USER_SUB_MENU:
        /* Back to main menu */
        setErrorLed(false);             /* For DISP_INTERFACE == DISP_EXT_LCD */
        menuLevel = PROG_MENU;
        displayClear();
        menuRow0Sel();
        menuRow1();
        menuRow2();
        menuRow3();
        break;
        
      default:
        break;
      }
    }

    if (pressPb1)
    {
      __disable_irq();
      pressPb1 = false;
      __enable_irq();

      setErrorLed(false);
      setBusyLed(false);

      switch (menuLevel)
      {
      case PROG_MENU:
      case PROG_SUB_MENU:
        if (!(menuLevel & SUB_MENU_MASK))
        {
          /* First entry, check valid binary in memory? */
          if (getBinInfoFromExtMem())
          {
            menuLevel |= SUB_MENU_MASK;
            /* Wait key press, PB0 to exit, PB1 to execute */
            while (1)
            {
              if (pressPb0)
              {
                subMenuExit = true;
                break;
              }
              if (pressPb1)
              {
                subMenuExit = false;
                break;
              }
            }
          }
          else
          {
            printf("No binary in memory\n");
            setErrorLed(true);
            /* Wait key press, PB0 to exit, PB1 to check again */
            while (1)
            {
              if (pressPb0)
              {
                menuLevel |= SUB_MENU_MASK;
                subMenuExit = true;
                break;
              }
              if (pressPb1)
              {
                subMenuExit = true;
                break;
              }
            }
          }
        }
        if (!subMenuExit)
        {
          progSequence();
        }
        break;
        
      case UPLOAD_MENU:
      case UPLOAD_SUB_MENU:
        menuLevel |= SUB_MENU_MASK;
        displayClear();
        /* Load image to memory and readback to verify */
        if (uploadImageFromUsb())
        {
          /* Reset program count if load image success */
          if (getBinInfoFromExtMem())
          {
            progDevCnt = 0;
          }
          else
          {
            setErrorLed(true);
          }
        }
        break;
        
      case ERASE_MAIN_MENU:
      case ERASE_MAIN_SUB_MENU:
        menuLevel |= SUB_MENU_MASK;
        eraseMainSequence();
        break;
        
      case ERASE_USER_MENU:
      case ERASE_USER_SUB_MENU:
        menuLevel |= SUB_MENU_MASK;
        eraseUserSequence();
        break;

      default:
        break;
      }
    }
  }
}

/******************************************************************************
 * @brief  Program sequence x1 or x4 or x8
 *****************************************************************************/
static void progSequence(void)
{
  if (!gangMode4 && !gangMode8)
  {
    progDevice();
  }
  else
  {
    GPIO_PinOutClear(MUX_OE_PORT, MUX_OE_PIN);
    GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    GPIO_PinOutClear(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
    GPIO_PinOutClear(MUX_CTRLC_PORT, MUX_CTRLC_PIN);
    delayMs(MUX_DELAY);
    if (progDevice())
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    delayMs(MUX_DELAY);
    if (progDevice())
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    GPIO_PinOutSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
    delayMs(MUX_DELAY);
    if (progDevice())
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    delayMs(MUX_DELAY);
    if (progDevice())
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    if (gangMode8)
    {
      GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      GPIO_PinOutClear(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
      GPIO_PinOutSet(MUX_CTRLC_PORT, MUX_CTRLC_PIN);
      delayMs(MUX_DELAY);
      if (progDevice())
      {
        GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
        return;
      }
      
      GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      delayMs(MUX_DELAY);
      if (progDevice())
      {
        GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
        return;
      }
      
      GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      GPIO_PinOutSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
      delayMs(MUX_DELAY);
      if (progDevice())
      {
        GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
        return;
      }
      
      GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      GPIO_PinOutSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
      GPIO_PinOutSet(MUX_CTRLC_PORT, MUX_CTRLC_PIN);
      delayMs(MUX_DELAY);
      progDevice();
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
    }
  }
}

/******************************************************************************
 * @brief  Erase main flash sequence x1 or x4 or x8
 *****************************************************************************/
static void eraseMainSequence(void)
{
  if (!gangMode4 && !gangMode8)
  {
    eraseFlash(true);
  }
  else
  {
    GPIO_PinOutClear(MUX_OE_PORT, MUX_OE_PIN);
    GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    GPIO_PinOutClear(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
    GPIO_PinOutClear(MUX_CTRLC_PORT, MUX_CTRLC_PIN);
    delayMs(MUX_DELAY);
    if (eraseFlash(true))
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    delayMs(MUX_DELAY);
    if (eraseFlash(true))
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    GPIO_PinOutSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
    delayMs(MUX_DELAY);
    if (eraseFlash(true))
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    delayMs(MUX_DELAY);
    if (eraseFlash(true))
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    if (gangMode8)
    {
      GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      GPIO_PinOutClear(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
      GPIO_PinOutSet(MUX_CTRLC_PORT, MUX_CTRLC_PIN);
      delayMs(MUX_DELAY);
      if (eraseFlash(true))
      {
        GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
        return;
      }
      
      GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      delayMs(MUX_DELAY);
      if (eraseFlash(true))
      {
        GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
        return;
      }
      
      GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      GPIO_PinOutSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
      delayMs(MUX_DELAY);
      if (eraseFlash(true))
      {
        GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
        return;
      }
      
      GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      GPIO_PinOutSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
      GPIO_PinOutSet(MUX_CTRLC_PORT, MUX_CTRLC_PIN);
      delayMs(MUX_DELAY);
      eraseFlash(true);
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
    }
  }
}

/******************************************************************************
 * @brief  Erase user page sequence x1 or x4 or x8
 *****************************************************************************/
static void eraseUserSequence(void)
{
  if (!gangMode4 && !gangMode8)
  {
    eraseFlash(false);
  }
  else
  {
    GPIO_PinOutClear(MUX_OE_PORT, MUX_OE_PIN);
    GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    GPIO_PinOutClear(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
    GPIO_PinOutClear(MUX_CTRLC_PORT, MUX_CTRLC_PIN);
    delayMs(MUX_DELAY);
    if (eraseFlash(false))
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    delayMs(MUX_DELAY);
    if (eraseFlash(false))
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    GPIO_PinOutSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
    delayMs(MUX_DELAY);
    if (eraseFlash(false))
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
    delayMs(MUX_DELAY);
    if (eraseFlash(false))
    {
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
      return;
    }
    
    if (gangMode8)
    {
      GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      GPIO_PinOutClear(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
      GPIO_PinOutSet(MUX_CTRLC_PORT, MUX_CTRLC_PIN);
      delayMs(MUX_DELAY);
      if (eraseFlash(false))
      {
        GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
        return;
      }
      
      GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      delayMs(MUX_DELAY);
      if (eraseFlash(false))
      {
        GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
        return;
      }
      
      GPIO_PinOutClear(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      GPIO_PinOutSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
      delayMs(MUX_DELAY);
      if (eraseFlash(false))
      {
        GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
        return;
      }
      
      GPIO_PinOutSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN);
      GPIO_PinOutSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN);
      GPIO_PinOutSet(MUX_CTRLC_PORT, MUX_CTRLC_PIN);
      delayMs(MUX_DELAY);
      eraseFlash(false);
      GPIO_PinOutSet(MUX_OE_PORT, MUX_OE_PIN);
    }
  }
}
