/**************************************************************************//**
 * @file  dap.h
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#ifndef _DAP_H_
#define _DAP_H_

#include "kits.h"
#include "progconfig.h"

/* ACK responses */
#define ACK_OK     1
#define ACK_WAIT   2
#define ACK_FAULT  4

/* Address of DP read registers */
#define DP_IDCODE  0
#define DP_CTRL    1
#define DP_RESEND  2
#define DP_RDBUFF  3

/* Adressses of DP write registers */
#define DP_ABORT   0
#define DP_STAT    1
#define DP_SELECT  2

/* AHB-AP registers */
#define AP_CSW 0
#define AP_TAR 1
#define AP_DRW 3
#define AP_IDR 3  /* In bank 0xf */

/* AAP registers */
#define AAP_CMD    0 
#define AAP_CMDKEY 1
#define AAP_STATUS 2
#define AAP_IDR    3  /* In bank 0xf */

#define AAP_CMD_ZERO      0xF0E00000
#define AAP_CMDKEY_ZERO   0xF0E00004
#define AAP_STATUS_ZERO   0xF0E00008
#define AAP_IDR_ZERO      0xF0E000FC

/* Valid values for the DP IDCODE register */
#define EFM32_DPID_1      0x2BA01477  // EFM32/EZR32/EFR32 M3 or M4
#define EFM32_DPID_2      0x0BC11477  // EFM32/EZR32 M0+
#define EFM32_DPID_3      0x0bc12477  // SWMD

/* Valid value for the AAP IDR register */
#define EFM32_AAP_ID_P1 0x16e60001
#define EFM32_AAP_ID_P2 0x26e60011

/* Valid values for the AHB-AP IDR register */
#define EFM32_AHBAP_ID_1  0x24770011 // EFM32/EZR32/EFR32 M3 or M4
#define EFM32_AHBAP_ID_2  0x04770031 // EFM32/EZR32 M0+

/* Key which must be written to AAP_CMDKEY before
 * writing to AAP_CMD */
#define AAP_UNLOCK_KEY 0xcfacc118
   
/* Value to write to AIRCR in order to do a soft
 * reset of the target */
#define AIRCR_RESET_CMD 0x05FA0006

/* Bit fields for the CSW register */
#define AP_CSW_32BIT_TRANSFER   0x02
#define AP_CSW_AUTO_INCREMENT   0x10
#define AP_CSW_MASTERTYPE_DEBUG (1 << 29)
#define AP_CSW_HPROT            (1 << 25)
#define AP_CSW_DEFAULT (AP_CSW_32BIT_TRANSFER | AP_CSW_MASTERTYPE_DEBUG | AP_CSW_HPROT)

/* Powerup request and acknowledge bits in CTRL/STAT */
#define DP_CTRL_CDBGPWRUPREQ  (1U << 28)
#define DP_CTRL_CDBGPWRUPACK  (1U << 29)
#define DP_CTRL_CSYSPWRUPREQ  (1U << 30)
#define DP_CTRL_CSYSPWRUPACK  (1U << 31)

/* Commands to run/step and let CPU run. 
 * Write these to DHCSR */
#define RUN_CMD  0xA05F0001
#define STOP_CMD 0xA05F0003
#define STEP_CMD 0xA05F0005

/* Flash addresses to retrieve the device unique id */
#define UNIQUE_ID_HIGH_ADDR 0x0FE081F4
#define UNIQUE_ID_LOW_ADDR  0x0FE081F0

/* AAP bit fields */
#define AAP_CMD_DEVICEERASE 1
#define AAP_CMD_SYSRESETREQ 2
#define AAP_STATUS_ERASEBUSY 1

/* Number of times to retry an SWD operation when receiving 
 * a WAIT response */
#define SWD_RETRY_COUNT 200

/* Number of times to retry the connection sequence */
#define CONNECT_RETRY_COUNT 3
   
/* Number of times to retry reading the AHB-IDR register when connecting */
#define AHB_IDR_RETRY_COUNT 20

/* Number of times to retry reading the CTLR/STAT
 * register while waiting for powerup acknowledge */
#define PWRUP_TIMEOUT 100

/* Number of times to retry reading status registers while
 * waiting for a debug event (such as a halt of soft reset) */
#define DEBUG_EVENT_TIMEOUT 200

/* Number of times to wait for MSC operation to complete */
#define MSC_TIMEOUT 100

/* JTAG to SWD bit sequence, transmitted LSB first */
#define JTAG2SWD 0xE79E

#define DRIVE_MODE              0x05
#define SWDIO_SET_CLR_MASK      (1 << SWDIO_PIN)          
#define SWCLK_SET_CLR_MASK      (1 << SWCLK_PIN)          
#define SWD_SWC_SET_CLR_MASK    SWDIO_SET_CLR_MASK | SWCLK_SET_CLR_MASK

#if (EXTERNAL_MEMORY == EBI_NAND_FLASH) && (USE_EFM32_STK == TRUE)
#define SWDIO_INPUT_MASK_LOW    (0x1 << (4 * SWDIO_PIN)) | (DRIVE_MODE << (4 * (SWCLK_PIN))) | 0x00000440
#else
#define SWDIO_INPUT_MASK_LOW    (0x1 << (4 * SWDIO_PIN)) | (DRIVE_MODE << (4 * (SWCLK_PIN)))
#endif
#define SWDIO_INPUT_MASK_HIGH   (0x1 << (4 * (SWDIO_PIN-8)))| (DRIVE_MODE << (4 * (SWCLK_PIN-8)))

#if (EXTERNAL_MEMORY == EBI_NAND_FLASH) && (USE_EFM32_STK == TRUE)
#define SWDIO_OUTPUT_MASK_LOW   (DRIVE_MODE << (4 * SWDIO_PIN)) | (DRIVE_MODE << (4 * (SWCLK_PIN))) | 0x00000440
#else
#define SWDIO_OUTPUT_MASK_LOW   (DRIVE_MODE << (4 * SWDIO_PIN)) | (DRIVE_MODE << (4 * (SWCLK_PIN)))
#endif
#define SWDIO_OUTPUT_MASK_HIGH  (DRIVE_MODE << (4 * (SWDIO_PIN-8))) | (DRIVE_MODE << (4 * (SWCLK_PIN-8)))

#if (SWDIO_PIN < 8)
#define SWDIO_SET_INPUT() { \
    GPIO->P[SWDIO_PORT].MODEL = SWDIO_INPUT_MASK_LOW; \
    GPIO->P[SWDIO_PORT].DOUTCLR = SWDIO_SET_CLR_MASK; }
#else 
#define SWDIO_SET_INPUT() { \
    GPIO->P[SWDIO_PORT].MODEH = SWDIO_INPUT_MASK_HIGH; \
    GPIO->P[SWDIO_PORT].DOUTCLR = SWDIO_SET_CLR_MASK; }
#endif

#if (SWDIO_PIN < 8)
#define SWDIO_SET_OUTPUT() { \
    GPIO->P[SWDIO_PORT].MODEL = SWDIO_OUTPUT_MASK_LOW; \
    GPIO->P[SWDIO_PORT].DOUTCLR = SWDIO_SET_CLR_MASK; }
#else
#define SWDIO_SET_OUTPUT() { \
    GPIO->P[SWDIO_PORT].MODEH = SWDIO_OUTPUT_MASK_HIGH; \
    GPIO->P[SWDIO_PORT].DOUTCLR = SWDIO_SET_CLR_MASK; }
#endif

#define SWDIO_SET() GPIO->P[SWDIO_PORT].DOUTSET = SWDIO_SET_CLR_MASK
#define SWDIO_CLR() GPIO->P[SWDIO_PORT].DOUTCLR = SWDIO_SET_CLR_MASK
#define SWCLK_SET() GPIO->P[SWCLK_PORT].DOUTSET = SWCLK_SET_CLR_MASK
#define SWCLK_CLR() GPIO->P[SWCLK_PORT].DOUTCLR = SWCLK_SET_CLR_MASK

#define SWCLK_CYCLE()   \
  SWCLK_CLR();          \
  SWCLK_SET()

#define SWDIO_CYCLE()   \
  SWDIO_SET();          \
  SWDIO_CLR()

/* Note that the macros write to the entire port at once. */
#define WRITE_BIT(bit) \
if ( bit ) { \
  GPIO->P[SWDIO_PORT].DOUT = SWDIO_SET_CLR_MASK; \
  GPIO->P[SWDIO_PORT].DOUT = SWD_SWC_SET_CLR_MASK; \
} else { \
  GPIO->P[SWDIO_PORT].DOUT = 0; \
  GPIO->P[SWDIO_PORT].DOUT = SWCLK_SET_CLR_MASK; \
}

#define READ_BIT(bit) \
  GPIO->P[SWCLK_PORT].DOUT = 0; \
  bit = ((GPIO->P[SWDIO_PORT].DIN >> SWDIO_PIN) & 0x1); \
  GPIO->P[SWCLK_PORT].DOUT = SWCLK_SET_CLR_MASK

void JTAG_to_SWD_Sequence(void);
void exitDormantState(void);
void writeAP(int reg, uint32_t data);
void writeDP(int reg, uint32_t data);
void writeDpIgnoreAck(int reg, uint32_t data);
void readAP(int reg, uint32_t *data);
void readDP(int reg, uint32_t *data);
uint32_t initDp(void);
uint32_t readApId(void);
void aapExtensionSequence(void);
void initAhbAp(void);

#endif
