/**************************************************************************//**
 * @file  debug_lock.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include "em_device.h"
#include "dap.h"
#include "utils.h"
#include "debug_lock.h"
#include "errors.h"
#include "em_gpio.h"
#include "em_ebi.h"
#include "em_cmu.h"
#include "flash_write.h"
#include "delay.h"
#include "progconfig.h"
#include "state_machine.h"
#include "usbtomemory.h"

/* The address of the Debug Lock Word. This word must be 
 * cleared in order to lock debug acces */
#define DEBUG_LOCK_WORD (0x0FE04000 + 127 * 4)
#define AAP_LOCK_WORD   (0x0FE04000 + 124 * 4)

extern volatile uint32_t userPageSize;
extern volatile uint32_t mainPageSize;
extern volatile uint32_t mainFlashSize;


/**********************************************************
 * Verify that the flash is cleared. To save time
 * we only check the first word on every page. 
 * this could be modified to check every word.
 **********************************************************/
void verifyFlashIsErased(void)
{
  uint32_t addr = 0;
  uint32_t value;
  
  while ( addr < mainFlashSize ) 
  {
    value = readMem(addr);
    if ( value != 0xFFFFFFFF ) 
    {
      printf("Error at 0x%.8lx\n", addr);
      printf("Value is 0x%.8lx\n", value);
      RAISE(SWD_ERROR_DEVICE_ERASE_FAILED);
    }
    addr += mainPageSize;
  }
}


/**********************************************************
 * Verify that the user page is cleared.
 **********************************************************/
void verifyUserIsErased(void)
{
  uint32_t addr = USER_PAGE_ADDR;
  uint32_t addrEnd = addr + userPageSize - 1;
  uint32_t value;
  
  while ( addr < addrEnd ) 
  {
    value = readMem(addr);
    if ( value != 0xFFFFFFFF ) 
    {
      printf("Error at 0x%.8lx\n", addr);
      printf("Value is 0x%.8lx\n", value);
      RAISE(SWD_ERROR_DEVICE_ERASE_FAILED);
    }
    addr += 4;
  }
}


/**********************************************************
 * Locks the target by clearing the Debug Lock Word and 
 * then performing a pin reset. 
 **********************************************************/
void lockTarget(void)
{
  uint32_t apId;
  
  /* Clear Debug Lock Word */
  writeWordToFlash(DEBUG_LOCK_WORD, 0);
  
  /* Verify that DLW is cleared */
  uint32_t dlw = readMem(DEBUG_LOCK_WORD);
  if ( dlw != 0 ) {
    RAISE(SWD_ERROR_CLR_DLW_FAILED);
  }
  
  /* Perform a pin reset. After this the target should be locked. */
  hardResetTarget();
  delayMs(LOCK_RESET_DELAY);

  /* Verify that target is locked by reading the AAP ID */
  initDp();        
  apId = readApId();
  
  if ( (apId == EFM32_AAP_ID_P1) || (apId == EFM32_AAP_ID_P2) )
  {
    return;
  }
  else if ( apId == EFM32_AHBAP_ID_2 )
  {
    apId = readMem(AAP_IDR_ZERO);
    if ( apId == EFM32_AAP_ID_P1 ) 
    {
      return;
    }
  }
  else
  {
    RAISE(SWD_ERROR_LOCK_FAILED);
  }
}


/**********************************************************
 * AAP locks the target by clearing the AAP Lock Word and 
 * then performing a pin reset. 
 **********************************************************/
void aapLockTarget(void)
{
  /* Clear Debug Lock Word */
  writeWordToFlash(AAP_LOCK_WORD, 0);
  
  /* Perform a pin reset. After this the target should be locked. */
  hardResetTarget();
  delayMs(LOCK_RESET_DELAY);
}
