/**************************************************************************//**
 * @file  errors.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

/*******************************************************************
 * 
 * This file (and errors.h) implements a simple exception-like
 * system for handling errors in C. By using this we can avoid
 * checking for error return values on every function call.
 * The exception system is based on the library functions setjmp() 
 * and longjmp(). Four macros are defined, TRY, CATCH, ENDTRY and RAISE(x). 
 *  
 * The macros are used like this: 
 * 
 * TRY
 *   someFunction();
 *   someOtherFunction();
 * CATCH
 *   printf("Error occured: %s\n", getErrorString(errorCode));
 * ENDTRY
 * 
 * It is possible to raise an exception like this: 
 * 
 * void someFunction(void)
 * {
 *   .....
 *   if ( someErrorCondition )
 *   {
 *     RAISE(SWD_ERROR_SOME_ERROR);
 *   }
 *   ....
 * }
 * 
 * The error codes are defined in errors.h. After a call to RAISE()
 * The execution will jump to the CATCH block which will receive 
 * the error in a variable called errorCode. The getErrorString() 
 * function can be used to conveniently print an error message. 
 * 
 * TRY/CATCH blocks can be nested. I.e. someFunction() can 
 * include its own TRY/CATCH block. Any RAISE()'ed error will
 * return to the closest CATCH block. The maximum nesting level
 * is specified with EXCEPTION_MAX_LEVEL. 
 * 
 * IMPORTANT
 * 
 *   Do not RAISE() an error within a CATCH block. 
 * 
 *   Exiting early from a TRY block in any other way than
 *   RAISE() WILL break the system and lead to undefined 
 *   behaviour. Therefore make sure to never have a return 
 *   statement within the TRY or CATCH blocks. 
 * 
 *****************************************************************/

#include "errors.h"


jmp_buf swdErrors[EXCEPTION_MAX_LEVEL];
int swdErrorIndex = -1;

char *getErrorString(int errorCode)
{
  switch (errorCode)
  {
  case SWD_ERROR_OK:
    return "No error.";
  case SWD_ERROR_WAIT:
    return "Timed out while waiting for WAIT response.";
  case SWD_ERROR_FAULT:
    return "Target returned FAULT response.";
  case SWD_ERROR_PROTOCOL:
    return "Protocol error. Target does not respond.";
  case SWD_ERROR_PARITY:
    return "Parity error.";
  case SWD_ERROR_MCU_LOCKED:
    return "MCU locked.";
  case SWD_ERROR_INVALID_IDR:
    return "Invalid IDR value.";
  case SWD_ERROR_FLASH_WRITE_FAILED:
    return "Write to flash failed.";
  case SWD_ERROR_UNLOCK_FAILED:
    return "Failed to unlock target.";
  case SWD_ERROR_AAP_EXTENSION_FAILED:
    return "Unable to access AAP registers. Is the reset pin connected?";
  case SWD_ERROR_LOCK_FAILED:
    return "Failed to lock target.";
  case SWD_ERROR_CLR_DLW_FAILED:
    return "Failed to clear Debug Lock Word.";
  case SWD_ERROR_MASS_ERASE_TIMEOUT:
    return "Timed out while waiting for Mass Erase to complete.";
  case SWD_ERROR_VERIFY_FW_FAILED:
    return "Application verification failed.";
  case SWD_ERROR_TIMEOUT_WAITING_RESET:
    return "Timeout while waiting for target to reset.";
  case SWD_ERROR_TARGET_NOT_HALTED:
    return "Target is halted.";
  case SWD_ERROR_FLASHLOADER_ERROR:
    return "Error in flashloader.";
  case SWD_ERROR_DEVICE_ERASE_FAILED:
    return "Device Erase failed."; 
  case SWD_ERROR_TIMEOUT_HALT:
    return "Timeout while waiting to halt target.";
  case SWD_ERROR_DEBUG_POWER:
    return "Failed to power up debug interface.";
  case SWD_ERROR_INVALID_IDCODE:
    return "Invalid IDCODE.";
  default:
    return "Unknown error.";
  }
}
