/**************************************************************************//**
 * @file  flash_write.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdio.h>
#include <stdbool.h>
#include "em_device.h"
#include "dap.h"
#include "errors.h"
#include "flash_write.h"
#include "kits.h"
#include "utils.h"
#include "delay.h"
#include "state_machine.h"
#include "progconfig.h"
#include "usbtomemory.h"
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
#include "nandutil.h"
#else
#include "spiflash.h"
#endif


extern uint32_t progDevCnt;
extern volatile bool useIntMem;
extern volatile bool newFamily;
extern volatile uint32_t chipType;
extern volatile uint32_t mainPageSize;
extern volatile uint32_t mainPageSizeMask;
extern volatile uint32_t mainFlashSize;
extern volatile bool debugLockAfterProg;
extern volatile bool aapLockAfterProg;
extern uint8_t memoryReadBuffer[];
extern volatile uint32_t mscWriteCtrl;
extern volatile uint32_t mscWriteCmd;
extern volatile uint32_t mscAddrb;
extern volatile uint32_t mscWdata;
extern volatile uint32_t mscStatusReg;
extern volatile uint32_t mscMassLock;

static void uploadImage(uint32_t startAddr, uint32_t size);
static void uploadImageInt(uint32_t startAddr, uint32_t size);
static void uploadImageGecko(uint32_t startAddr, uint32_t size);
static void uploadImageGeckoInt(uint32_t startAddr, uint32_t size);
static void uploadImageGiant(uint32_t startAddr, uint32_t size);
static void uploadImageGiantInt(uint32_t startAddr, uint32_t size);
static void uploadImageUser(uint32_t size);
static void uploadImageUserGecko(uint32_t size);


/**********************************************************
 * User page is erased by writing directly to the MSC 
 * registers 
 **********************************************************/
void eraseUserPage(void)
{
  uint32_t mscStatus;
  uint32_t timeOut;
  
  /* Enable write in MSC */
  writeMem((uint32_t)(mscWriteCtrl), MSC_WRITECTRL_WREN);

  /* Enter the start address of the page */
  writeMem((uint32_t)(mscAddrb), USER_PAGE_ADDR);
  
  /* Load address into internal register */
  writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
  
  /* Start page erase operation */
  writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_ERASEPAGE);
  
  /* Wait until erase is complete */
  delayMs(ERASE_DELAY);
  timeOut = ERASE_LOOPCNT;
  do {
    mscStatus = readMem((uint32_t)(mscStatusReg));
    timeOut--;
  } while ( (mscStatus & MSC_STATUS_BUSY) && timeOut > 0 );
  
  if ( mscStatus & MSC_STATUS_BUSY ) {
    RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
  }
}


/**********************************************************
 * Erases enough pages of flash to fit a image of the 
 * given size. The size of flash pages is retrieved
 * from the memory. The pages are erased by 
 * writing directly to the MSC registers 
 * 
 * @param startAddr
 *    Start address to erase (must in multiple of page size)
 * @param size
 *    Size in bytes of firmware image
 **********************************************************/
void erasePages(uint32_t startAddr, uint32_t size)
{
  uint32_t mscStatus;
  uint32_t timeOut;
  
  if (chipType == _DEVINFO_PART_DEVICE_FAMILY_GG)
  {
    /* Enable double word write if device is GG */
    writeMem((uint32_t)(mscWriteCtrl), (MSC_WRITECTRL_WREN + 0x04));
  }
  else
  {
    /* Enable write in MSC */
    writeMem((uint32_t)(mscWriteCtrl), MSC_WRITECTRL_WREN);
  }
  
  /* Adjust size with start address */
  size += startAddr;

  /* Device is GG */
  if (chipType == _DEVINFO_PART_DEVICE_FAMILY_GG)
  {
    /* Erase upper 512K if 1M GG and binary >512K */
    if ((size > 512 * 1024) && (mainFlashSize == 1024 * 1024))
    {
      /* Unlock mass erase */
      writeMem((uint32_t)(mscMassLock), MSC_MASSLOCK_LOCKKEY_UNLOCK);
    
      /* Start upper half mass erase operation */
      writeMem((uint32_t)(mscWriteCmd), (0x1UL << 9));
    
      /* Wait until erase is complete */
      delayMs(ERASE_DELAY);
      timeOut = ERASE_LOOPCNT;
      do {
        mscStatus = readMem((uint32_t)(mscStatusReg));
        timeOut--;
      } while ( (mscStatus & MSC_STATUS_BUSY) && timeOut > 0 );
    
      if ( mscStatus & MSC_STATUS_BUSY ) {
        RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
      }
      
      /* Adjust size to 512K */
      size = 512 * 1024;
    }
  }

  /* Not start from address 0 or device is G or TG */
  if ((startAddr != 0) || (mainPageSize == 512))
  {
    /* Loop until enough pages have been erased */
    while ( startAddr < size ) 
    {
      /* Enter the start address of the page */
      writeMem((uint32_t)(mscAddrb), startAddr);
    
      /* Load address into internal register */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
    
      /* Start page erase operation */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_ERASEPAGE);
    
      /* Wait until erase is complete */
      delayMs(ERASE_DELAY);
      timeOut = ERASE_LOOPCNT;
      do {
        mscStatus = readMem((uint32_t)(mscStatusReg));
        timeOut--;
      } while ( (mscStatus & MSC_STATUS_BUSY) && timeOut > 0 );
    
      if ( mscStatus & MSC_STATUS_BUSY ) {
        RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
      }
    
      /* Move to next page */
      startAddr += mainPageSize;
    }
  }
  /* Start from address 0 and device is not G or TG */
  else
  {
    /* Unlock mass erase */
    writeMem((uint32_t)(mscMassLock), MSC_MASSLOCK_LOCKKEY_UNLOCK);
    
    /* Start mass erase operation */
    writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_ERASEMAIN0);
    
    /* Wait until erase is complete */
    delayMs(ERASE_DELAY);
    timeOut = ERASE_LOOPCNT;
    do {
      mscStatus = readMem((uint32_t)(mscStatusReg));
      timeOut--;
    } while ( (mscStatus & MSC_STATUS_BUSY) && timeOut > 0 );
    
    if ( mscStatus & MSC_STATUS_BUSY ) {
      RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
    }
  }
}


/***********************************************************
 * Writes a firmware image to internal flash of the
 * target using direct writes to the MSC registers.
 * Writing starts at start address at the target
 * and data is copied from the external memory.
 * 
 * @param startAddr
 *    Start address to write (must in multiple of page size)
 *    
 * @param size
 *    Size in bytes of firmware image
 **********************************************************/
static void uploadImage(uint32_t startAddr, uint32_t size)
{
  uint32_t w;
  uint32_t pageNum = 0;
  uint32_t ptrIndex = 0;
  uint32_t *wordPtr = (uint32_t *)memoryReadBuffer;
  
  /* Adjust size with start address */
  size += startAddr;
   
  /* Start writing all words from the given buffer. 
   * Writing starts at start address on the target */
  for (w=startAddr; w<size; w+=4, ptrIndex++) 
  {
    if ((w & mainPageSizeMask) == 0)
    {
      /* Enter address */
      writeMem((uint32_t)(mscAddrb), w);
    
      /* Load address into internal register */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
    }
    
    /* Last page finish, read another page */
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    if ((w & NAND_FLASH_PAGE_SIZE_MASK) == 0)
    {
      nandFlashReadPage(pageNum, memoryReadBuffer);
      pageNum++;
      ptrIndex = 0;
    }
#else
    if ((w & SPIF_PAGESIZE_MASK) == 0)
    {
      spiFlashBufferRead(memoryReadBuffer, pageNum, SPIF_PAGESIZE);
      pageNum +=  SPIF_PAGESIZE;
      ptrIndex = 0;
    }
#endif
    
    /* Write value to WDATA */
    writeMem((uint32_t)(mscWdata), wordPtr[ptrIndex]);
    
    /* Start flash write */
    writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_WRITEONCE);
    
#if (SKIP_POLLING == FALSE)
    uint32_t timeOut;
    uint32_t mscStatus;
    
    /* Wait for write to complete */
    timeOut = MSC_TIMEOUT;
    do {
      mscStatus = readMem((uint32_t)(mscStatusReg));
      timeOut--;
    } while ( ((mscStatus & MSC_STATUS_WDATAREADY) == 0) && timeOut );
    
    if ( timeOut == 0 )
    {
      RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
    }
#else
    if (newFamily)
    {
      delayUs(WRITE_DELAY);
    }
#endif
  }
}

/***********************************************************
 * Writes a firmware image to internal flash of the
 * target using direct writes to the MSC registers.
 * Writing starts at start address at the target
 * and data is copied from the internal memory.
 * 
 * @param startAddr
 *    Start address to write (must in multiple of page size)
 *    
 * @param size
 *    Size in bytes of firmware image
 **********************************************************/
static void uploadImageInt(uint32_t startAddr, uint32_t size)
{
  uint32_t w;
  uint32_t *intPtr = (uint32_t *)(INT_START_ADDR);
  
  /* Adjust size with start address */
  size += startAddr;
   
  /* Start writing all words from the given buffer. 
   * Writing starts at start address on the target */
  for (w=startAddr; w<size; w+=4) 
  {
    if ((w & mainPageSizeMask) == 0)
    {
      /* Enter address */
      writeMem((uint32_t)(mscAddrb), w);
    
      /* Load address into internal register */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
    }
    
    /* Write value to WDATA */
    writeMem((uint32_t)(mscWdata), *intPtr++);
    
    /* Start flash write */
    writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_WRITEONCE);
    
#if (SKIP_POLLING == FALSE)
    uint32_t timeOut;
    uint32_t mscStatus;
    
    /* Wait for write to complete */
    timeOut = MSC_TIMEOUT;
    do {
      mscStatus = readMem((uint32_t)(mscStatusReg));
      timeOut--;
    } while ( ((mscStatus & MSC_STATUS_WDATAREADY) == 0) && timeOut );
    
    if ( timeOut == 0 )
    {
      RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
    }
#else
    if (newFamily)
    {
      delayUs(WRITE_DELAY);
    }
#endif
  }
}


/**********************************************************
 * Writes a firmware image to internal flash of the
 * Gecko using direct writes to the MSC registers.
 * Writing starts at start address at the target
 * and data is copied from the external memory
 * 
 * @param startAddr
 *    Start address to write (must in multiple of page size)
 *    
 * @param size
 *    Size in bytes of firmware image
 **********************************************************/
static void uploadImageGecko(uint32_t startAddr, uint32_t size)
{
  uint32_t w;
  uint32_t pageNum = 0;
  uint32_t ptrIndex = 0;
  uint32_t *wordPtr = (uint32_t *)memoryReadBuffer;
  
  /* Adjust size with start address */
  size += startAddr;

  /* Start writing all words from the given buffer. 
   * Writing starts at start address on the target */
  for (w=startAddr; w<size; w+=4, ptrIndex++) 
  {
    /* Last page finish, read another page */
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    if ((w & NAND_FLASH_PAGE_SIZE_MASK) == 0)
    {
      nandFlashReadPage(pageNum, memoryReadBuffer);
      pageNum++;
      ptrIndex = 0;
    }
#else
    if ((w & SPIF_PAGESIZE_MASK) == 0)
    {
      spiFlashBufferRead(memoryReadBuffer, pageNum, SPIF_PAGESIZE);
      pageNum +=  SPIF_PAGESIZE;
      ptrIndex = 0;
    }
#endif
    
    /* No flash write if data = erased word */
    if (wordPtr[ptrIndex] != 0xffffffff)
    {
      /* Enter address */
      writeMem((uint32_t)(mscAddrb), w);
    
      /* Load address into internal register */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
    
      /* Write value to WDATA */
      writeMem((uint32_t)(mscWdata), wordPtr[ptrIndex]);
    
      /* Start flash write */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_WRITEONCE);
    
#if (SKIP_POLLING == FALSE)
      uint32_t timeOut;
      uint32_t mscStatus;
    
      /* Wait for write to complete */
      timeOut = MSC_TIMEOUT;
      do {
        mscStatus = readMem((uint32_t)(mscStatusReg));
        timeOut--;
      } while ( ((mscStatus & MSC_STATUS_WDATAREADY) == 0) && timeOut );
    
      if ( timeOut == 0 )
      {
        RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
      }
#else
#if defined( MSC_READCTRL_PREFETCH )
      /* Need dealy if using EFM32GG */
      delayUs(12);
#endif
#endif
    }
  }
}

/**********************************************************
 * Writes a firmware image to internal flash of the
 * Gecko using direct writes to the MSC registers.
 * Writing starts at start address at the target
 * and data is copied from the internal memory
 * 
 * @param startAddr
 *    Start address to write (must in multiple of page size)
 *    
 * @param size
 *    Size in bytes of firmware image
 **********************************************************/
static void uploadImageGeckoInt(uint32_t startAddr, uint32_t size)
{
  uint32_t w;
  uint32_t *intPtr = (uint32_t *)(INT_START_ADDR);
  
  /* Adjust size with start address */
  size += startAddr;

  /* Start writing all words from the given buffer. 
   * Writing starts at start address on the target */
  for (w=startAddr; w<size; w+=4, intPtr++) 
  {
    /* No flash write if data = erased word */
    if (*intPtr != 0xffffffff)
    {
      /* Enter address */
      writeMem((uint32_t)(mscAddrb), w);
    
      /* Load address into internal register */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
    
      /* Write value to WDATA */
      writeMem((uint32_t)(mscWdata), *intPtr);
    
      /* Start flash write */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_WRITEONCE);
    
#if (SKIP_POLLING == FALSE)
      uint32_t timeOut;
      uint32_t mscStatus;
    
      /* Wait for write to complete */
      timeOut = MSC_TIMEOUT;
      do {
        mscStatus = readMem((uint32_t)(mscStatusReg));
        timeOut--;
      } while ( ((mscStatus & MSC_STATUS_WDATAREADY) == 0) && timeOut );
    
      if ( timeOut == 0 )
      {
        RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
      }
#else
#if defined( MSC_READCTRL_PREFETCH )
      /* Need dealy if using EFM32GG */
      delayUs(12);
#endif
#endif
    }
  }
}


/***********************************************************
 * Writes a firmware image to internal flash of the
 * Giant Gecko using direct writes to the MSC registers.
 * Writing starts at start address at the target
 * and data is copied from the external memory.
 * 
 * @param startAddr
 *    Start address to write (must in multiple of page size)
 *    
 * @param size
 *    Size in bytes of firmware image
 **********************************************************/
static void uploadImageGiant(uint32_t startAddr, uint32_t size)
{
  uint32_t w;
  uint32_t pageNum = 0;
  uint32_t ptrIndex = 0;
  uint32_t *wordPtr = (uint32_t *)memoryReadBuffer;

  /* Adjust size with start address */
  size += startAddr;

  /* Start writing all words from the given buffer. 
   * Writing starts at start address on the target */
  for (w=startAddr; w<size; w+=8, ptrIndex++) 
  {
    if ((w & mainPageSizeMask) == 0)
    {
      /* Enter address */
      writeMem((uint32_t)(mscAddrb), w);
    
      /* Load address into internal register */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
    }
    
    /* Last page finish, read another page */
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    if ((w & NAND_FLASH_PAGE_SIZE_MASK) == 0)
    {
      nandFlashReadPage(pageNum, memoryReadBuffer);
      pageNum++;
      ptrIndex = 0;
    }
#else
    if ((w & SPIF_PAGESIZE_MASK) == 0)
    {
      spiFlashBufferRead(memoryReadBuffer, pageNum, SPIF_PAGESIZE);
      pageNum +=  SPIF_PAGESIZE;
      ptrIndex = 0;
    }
#endif
    
    /* Write frist word to WDATA */
    writeMem((uint32_t)(mscWdata), wordPtr[ptrIndex++]);
    
    /* Write second word to WDATA */
    writeMem((uint32_t)(mscWdata), wordPtr[ptrIndex]);

    /* Start flash write */
    writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_WRITEONCE);
    
    /* Wait for write to complete */
#if (SKIP_POLLING == FALSE)
    uint32_t timeOut;
    uint32_t mscStatus;
    
    /* Wait for write to complete */
    timeOut = MSC_TIMEOUT;
    do {
      mscStatus = readMem((uint32_t)(mscStatusReg));
      timeOut--;
    } while ( ((mscStatus & MSC_STATUS_WDATAREADY) == 0) && timeOut );
    
    if ( timeOut == 0 )
    {
      RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
    }
#endif
  }
}


/***********************************************************
 * Writes a firmware image to internal flash of the
 * Giant Gecko using direct writes to the MSC registers.
 * Writing starts at start address at the target
 * and data is copied from the internal memory.
 * 
 * @param startAddr
 *    Start address to write (must in multiple of page size)
 *    
 * @param size
 *    Size in bytes of firmware image
 **********************************************************/
static void uploadImageGiantInt(uint32_t startAddr, uint32_t size)
{
  uint32_t w;
  uint32_t *intPtr = (uint32_t *)(INT_START_ADDR);

  /* Adjust size with start address */
  size += startAddr;

  /* Start writing all words from the given buffer. 
   * Writing starts at start address on the target */
  for (w=startAddr; w<size; w+=8) 
  {
    if ((w & mainPageSizeMask) == 0)
    {
      /* Enter address */
      writeMem((uint32_t)(mscAddrb), w);
    
      /* Load address into internal register */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
    }
    
    /* Write frist word to WDATA */
    writeMem((uint32_t)(mscWdata), *intPtr++);
    
    /* Write second word to WDATA */
    writeMem((uint32_t)(mscWdata), *intPtr++);

    /* Start flash write */
    writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_WRITEONCE);
    
    /* Wait for write to complete */
#if (SKIP_POLLING == FALSE)
    uint32_t timeOut;
    uint32_t mscStatus;
    
    /* Wait for write to complete */
    timeOut = MSC_TIMEOUT;
    do {
      mscStatus = readMem((uint32_t)(mscStatusReg));
      timeOut--;
    } while ( ((mscStatus & MSC_STATUS_WDATAREADY) == 0) && timeOut );
    
    if ( timeOut == 0 )
    {
      RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
    }
#endif
  }
}


/**********************************************************
 * Writes a firmware image to internal user page
 * using direct writes to the MSC registers.
 * Writing starts at user page address at the target
 * and data is copied from the external memory
 * 
 * @param size
 *    Size in bytes of firmware image
 **********************************************************/
static void uploadImageUser(uint32_t size)
{
  uint32_t w;
  uint32_t pageNum;
  uint32_t ptrIndex = 0;
  uint32_t *wordPtr = (uint32_t *)memoryReadBuffer;
  
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
  pageNum = USER_DATA_ADDR_PAGE;
#else
  pageNum = USER_DATA_SECTOR;
#endif
  
  /* Adjust size with start address */
  size += USER_PAGE_ADDR;

  /* Start writing all words from the given buffer. 
   * Writing starts at start address on the target */
  for ( w=USER_PAGE_ADDR; w<size; w+=4, ptrIndex++) 
  {
    if ((w & mainPageSizeMask) == 0)
    {
      /* Enter address */
      writeMem((uint32_t)(mscAddrb), w);
    
      /* Load address into internal register */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
    }

    /* Last page finish, read another page */
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    if ((w & NAND_FLASH_PAGE_SIZE_MASK) == 0)
    {
      nandFlashReadPage(pageNum, memoryReadBuffer);
      pageNum++;
      ptrIndex = 0;
    }
#else
    if ((w & SPIF_PAGESIZE_MASK) == 0)
    {
      spiFlashBufferRead(memoryReadBuffer, pageNum, SPIF_PAGESIZE);
      pageNum +=  SPIF_PAGESIZE;
      ptrIndex = 0;
    }
#endif
    
    /* Write value to WDATA */
    writeMem((uint32_t)(mscWdata), wordPtr[ptrIndex]);
    
    /* Start flash write */
    writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_WRITEONCE);
    
#if (SKIP_POLLING == FALSE)
    uint32_t timeOut;
    uint32_t mscStatus;
    
    /* Wait for write to complete */
    timeOut = MSC_TIMEOUT;
    do {
      mscStatus = readMem((uint32_t)(mscStatusReg));
      timeOut--;
    } while ( ((mscStatus & MSC_STATUS_WDATAREADY) == 0) && timeOut );
    
    if ( timeOut == 0 )
    {
      RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
    }
#else
    if (newFamily)
    {
      delayUs(WRITE_DELAY);
    }
#endif
  }
}


/**********************************************************
 * Writes a firmware image to internal user page of
 * the Gecko using direct writes to the MSC registers.
 * Writing starts at user page address at the target
 * and data is copied from the external memory
 * 
 * @param size
 *    Size in bytes of firmware image
 **********************************************************/
static void uploadImageUserGecko(uint32_t size)
{
  uint32_t w;
  uint32_t pageNum;
  uint32_t ptrIndex = 0;
  uint32_t *wordPtr = (uint32_t *)memoryReadBuffer;
  
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
  pageNum = USER_DATA_ADDR_PAGE;
#else
  pageNum = USER_DATA_SECTOR;
#endif
  
  /* Adjust size with start address */
  size += USER_PAGE_ADDR;

  /* Start writing all words from the given buffer. 
   * Writing starts at start address on the target */
  for ( w=USER_PAGE_ADDR; w<size; w+=4, ptrIndex++) 
  {
    /* Last page finish, read another page */
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    if ((w & NAND_FLASH_PAGE_SIZE_MASK) == 0)
    {
      nandFlashReadPage(pageNum, memoryReadBuffer);
      pageNum++;
      ptrIndex = 0;
    }
#else
    if ((w & SPIF_PAGESIZE_MASK) == 0)
    {
      spiFlashBufferRead(memoryReadBuffer, pageNum, SPIF_PAGESIZE);
      pageNum +=  SPIF_PAGESIZE;
      ptrIndex = 0;
    }
#endif
    
    /* No flash write if data = erased word */
    if (wordPtr[ptrIndex] != 0xffffffff)
    {
      /* Enter address */
      writeMem((uint32_t)(mscAddrb), w);
    
      /* Load address into internal register */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
    
      /* Write value to WDATA */
      writeMem((uint32_t)(mscWdata), wordPtr[ptrIndex]);
    
      /* Start flash write */
      writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_WRITEONCE);
    
#if (SKIP_POLLING == FALSE)
      uint32_t timeOut;
      uint32_t mscStatus;
    
      /* Wait for write to complete */
      timeOut = MSC_TIMEOUT;
      do {
        mscStatus = readMem((uint32_t)(mscStatusReg));
        timeOut--;
      } while ( ((mscStatus & MSC_STATUS_WDATAREADY) == 0) && timeOut );
    
      if ( timeOut == 0 )
      {
        RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
      }
#endif
    }
  }
}


/**********************************************************
 * Writes one word to flash 
 * @param addr The address to write to
 * @param data The value to write
 **********************************************************/
void writeWordToFlash(uint32_t addr, uint32_t data)
{
  uint32_t timeOut = MSC_TIMEOUT;
  uint32_t mscStatus;
  
  /* Load address */
  writeMem((uint32_t)(mscAddrb), addr);
  
  /* Load address into internal register */
  writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_LADDRIM);
  
  /* Write word to WDATA */
  writeMem((uint32_t)(mscWdata), data);
  
  /* Start flash write */
  writeMem((uint32_t)(mscWriteCmd), MSC_WRITECMD_WRITEONCE);
  
  /* Wait for write to complete */
  do {
    mscStatus = readMem((uint32_t)(mscStatusReg));
    timeOut--;
  } while ( ((mscStatus & MSC_STATUS_WDATAREADY) == 0) && timeOut );
    
  if ( timeOut == 0 )
  {
    RAISE(SWD_ERROR_FLASH_WRITE_FAILED);
  }
}


/*************************************************************
 * Uses direct writes to MSC registers in order to 
 * program the main flash.
 * @param startAddr
 *   Start address to program (must in multiple of page size)
 * @param binFileSize
 *   Size of binary file
 ************************************************************/
void flashMainWithMsc(uint32_t startAddr, uint32_t binFileSize)
{  
  printf("ERASE/PROG/VER MAIN\n");
#if (DISP_INTERFACE != DISP_NONE)
  DWT->CYCCNT = 0;
#endif
  haltTarget();
  
  /* Erase all pages that we are going to write to */
  erasePages(startAddr, binFileSize);
  
  /* Write firmware image to flash */
  if (chipType == _DEVINFO_PART_DEVICE_FAMILY_GG)
  {
    if (useIntMem)
    {
      uploadImageGiantInt(startAddr, binFileSize);
    }
    else
    {
      uploadImageGiant(startAddr, binFileSize);
    }
  }
  else if (chipType == _DEVINFO_PART_DEVICE_FAMILY_G)
  {
    if (useIntMem)
    {
      uploadImageGeckoInt(startAddr, binFileSize);
    }
    else
    {
      uploadImageGecko(startAddr, binFileSize);
    }
  }
  else
  {
    if (useIntMem)
    {
      uploadImageInt(startAddr, binFileSize);
    }
    else
    {
      uploadImage(startAddr, binFileSize);
    }
  }
    
  /* Verify application */
  if (useIntMem)
  {
    if (!verifyFirmwareInt(startAddr, binFileSize))
    {
      setErrorLed(true);
      return;
    }
  }
  else
  {
    if (!verifyFirmware(startAddr, binFileSize))
    {
      setErrorLed(true);
      return;
    }
  }
  
  /* Lock device */
  if (debugLockAfterProg)
  {
    stateDebugLock();
  }

  if (aapLockAfterProg && newFamily)
  {
    stateAapLock();
  }
  
#if (DISP_INTERFACE != DISP_NONE)
  setLcdRow3();
  float progTime = (float)(DWT->CYCCNT)/(float)48000000;
  printf("Time:%2d.%02ds #%lu\n", (int16_t)progTime, (int16_t)(progTime * 100)%100, ++progDevCnt);
#endif
}


/*************************************************************
 * Uses direct writes to MSC registers in order to 
 * program the user page.
 * @param binFileSize
 *   Size of binary file
 ************************************************************/
void flashUserWithMsc(uint32_t binFileSize)
{  
  printf("ERASE/PROG/VER USER\n");
#if (DISP_INTERFACE != DISP_NONE)
  DWT->CYCCNT = 0;
#endif
  haltTarget();
  
  /* Erase user page that we are going to write to */
  eraseUserPage();
  
  /* User page doesn't support double word write */
  if (chipType == _DEVINFO_PART_DEVICE_FAMILY_G)
  {
    uploadImageUserGecko(binFileSize);
  }
  else
  {
    uploadImageUser(binFileSize);
  }
    
  /* Verify flash */
  if (!verifyFirmware(USER_PAGE_ADDR, binFileSize))
  {
    setErrorLed(true);
    return;
  }
 
#if (DISP_INTERFACE != DISP_NONE)
  setLcdRow3();
  float progTime = (float)(DWT->CYCCNT)/(float)48000000;
  printf("Time:%2d.%02ds #%lu\n", (int16_t)progTime, (int16_t)(progTime * 100)%100, ++progDevCnt);
#endif
}

/*************************************************************
 * Uses direct writes to MSC registers in order to 
 * program the main flash and user page.
 * @param startAddr
 *   Start address to program (must in multiple of page size)
 * @param mainBinSize
 *   Size of main flash binary file
 * @param userBinSize
 *   Size of userpage binary file
 ************************************************************/
void flashMainUserWithMsc(uint32_t startAddr, uint32_t mainBinSize, uint32_t userBinSize)
{  
  printf("ERASE/PROG/VER M+U\n");

#if (DISP_INTERFACE != DISP_NONE)
  DWT->CYCCNT = 0;
#endif
  haltTarget();

  /* Program user page first since program main flash may lock the chip */ 
  eraseUserPage();
  
  /* User page doesn't support double word write */
  if (chipType == _DEVINFO_PART_DEVICE_FAMILY_G)
  {
    uploadImageUserGecko(userBinSize);
  }
  else
  {
    uploadImageUser(userBinSize);
  }
    
  /* Verify flash */
  if (!verifyFirmware(USER_PAGE_ADDR, userBinSize))
  {
    setErrorLed(true);
    return;
  }
  
  /* Erase all pages that we are going to write to */
  erasePages(startAddr, mainBinSize);
  
  /* Write firmware image to flash */
  if (chipType == _DEVINFO_PART_DEVICE_FAMILY_GG)
  {
    if (useIntMem)
    {
      uploadImageGiantInt(startAddr, mainBinSize);
    }
    else
    {
      uploadImageGiant(startAddr, mainBinSize);
    }
  }
  else if (chipType == _DEVINFO_PART_DEVICE_FAMILY_G)
  {
    if (useIntMem)
    {
      uploadImageGeckoInt(startAddr, mainBinSize);
    }
    else
    {
      uploadImageGecko(startAddr, mainBinSize);
    }
  }
  else
  {
    if (useIntMem)
    {
      uploadImageInt(startAddr, mainBinSize);
    }
    else
    {
      uploadImage(startAddr, mainBinSize);
    }
  }
    
  /* Verify application */
  if (useIntMem)
  {
    if (!verifyFirmwareInt(startAddr, mainBinSize))
    {
      setErrorLed(true);
      return;
    }
  }
  else
  {
    if (!verifyFirmware(startAddr, mainBinSize))
    {
      setErrorLed(true);
      return;
    }
  }

  /* Lock device */
  if (debugLockAfterProg)
  {
    stateDebugLock();
  }

  if (aapLockAfterProg && newFamily)
  {
    stateAapLock();
  }
  
#if (DISP_INTERFACE != DISP_NONE)
  setLcdRow3();
  float progTime = (float)(DWT->CYCCNT)/(float)48000000;
  printf("Time:%2d.%02ds #%lu\n", (int16_t)progTime, (int16_t)(progTime * 100)%100, ++progDevCnt);
#endif
}
