/**************************************************************************//**
 * @file  kits.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include "em_device.h"
#include "em_cmu.h"
#include "em_gpio.h"
#include "em_system.h"
#include "dap.h"
#include "kits.h"
#include "progconfig.h"

extern volatile bool intMemValid;
extern volatile bool pressPb0;
extern volatile bool pressPb1;
extern volatile bool gangMode4;
extern volatile bool gangMode8;

/**********************************************************
 * Toggles a LED on on the kit to indicate whether  
 * programmer is ready or not. 
 **********************************************************/
void setBusyLed(bool on)
{
  if ( on )
  {
    GPIO_PinOutSet(GPIO_LED1_PORT, GPIO_LED1_PIN);
  }
  else
  {
    GPIO_PinOutClear(GPIO_LED1_PORT, GPIO_LED1_PIN);
  }
}

/**********************************************************
 * Toggles a LED on on the kit to indicate whether an 
 * error occurs.
 **********************************************************/
void setErrorLed(bool on)
{
  if ( on )
  {
    GPIO_PinOutSet(GPIO_LED0_PORT, GPIO_LED0_PIN);
  }
  else
  {
    GPIO_PinOutClear(GPIO_LED0_PORT, GPIO_LED0_PIN);
  }
}

/**********************************************************
 * Interrupt handler for push button PB0. 
 **********************************************************/
void GPIO_ODD_IRQHandler(void)
{
  GPIO->IFC = GPIO->IF;
  pressPb0 = true;
}

/**********************************************************
 * Interrupt handler for push button PB1. 
 **********************************************************/
void GPIO_EVEN_IRQHandler(void)
{
  GPIO->IFC = GPIO->IF;
  pressPb1 = true;
}

/**************************************************************************//**
 * @brief Initialize programmer input and output
 *****************************************************************************/
void initKit(void)
{
  /* Select 48 MHz external crystal oscillator */
  CMU_ClockSelectSet( cmuClock_HF, cmuSelect_HFXO );
  CMU_OscillatorEnable(cmuOsc_HFRCO, false, false);
  CMU_ClockEnable(cmuClock_GPIO, true);

#if defined( MSC_READCTRL_PREFETCH )
  /* Enable prefetch if available */
  MSC->READCTRL |= MSC_READCTRL_PREFETCH;
#endif

#if (DISP_INTERFACE != DISP_NONE)
  /* Power up trace and debug clocks. Needed for DWT. */
  CoreDebug->DEMCR |= CoreDebug_DEMCR_TRCENA_Msk;
  /* Enable DWT cycle counter. Used to measure program speed. */
  DWT->CTRL |= DWT_CTRL_CYCCNTENA_Msk;
#endif

  /* High drive for SWCLK and SWD, both on the same GPIO port */
  GPIO_DriveModeSet((GPIO_Port_TypeDef)SWCLK_PORT, gpioDriveModeHigh);
  
  /* Enable SWCLK and SWDIO pin */
  GPIO_PinModeSet((GPIO_Port_TypeDef)SWCLK_PORT, SWCLK_PIN, gpioModePushPullDrive, 0);
  GPIO_PinModeSet((GPIO_Port_TypeDef)SWDIO_PORT, SWDIO_PIN, gpioModePushPullDrive, 1);
  
  /* High drive for RESET */
  GPIO_DriveModeSet((GPIO_Port_TypeDef)RESET_PORT, gpioDriveModeHigh);
  /* Enable RESET pin */
  GPIO_PinModeSet((GPIO_Port_TypeDef)RESET_PORT, RESET_PIN, gpioModePushPullDrive, 1);
  
  /* Setup keys GPIO */
  GPIO_PinModeSet(GPIO_KEY0_PORT, GPIO_KEY0_PIN, gpioModeInputPullFilter, 1);
  GPIO_IntConfig(GPIO_KEY0_PORT, GPIO_KEY0_PIN, false, true, true);
  NVIC_EnableIRQ(GPIO_ODD_IRQn);

  GPIO_PinModeSet(GPIO_KEY1_PORT, GPIO_KEY1_PIN, gpioModeInputPullFilter, 1);
  GPIO_IntConfig(GPIO_KEY1_PORT, GPIO_KEY1_PIN, false, true, true);
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);

  /* Setup LEDs GPIO */
  GPIO_PinModeSet(GPIO_LED0_PORT, GPIO_LED0_PIN, gpioModePushPull, 0);
  GPIO_PinModeSet(GPIO_LED1_PORT, GPIO_LED1_PIN, gpioModePushPull, 0);

  /* Setup MUX output enable GPIO */
  GPIO_PinModeSet(MUX_OE_PORT, MUX_OE_PIN, gpioModePushPull, 1);

  /* Setup programmer mode x1, x4 or x8 */
  if (GPIO_PinInGet(GANG4_PORT, GANG4_PIN) == 0)
  {
    gangMode4 = true;
  }
  
  if (GPIO_PinInGet(GANG8_PORT, GANG8_PIN) == 0)
  {
    gangMode8 = true;
  }

  /* Setup MUX control GPIO */
  if (gangMode4 || gangMode8)
  {
    GPIO_PinModeSet(MUX_CTRLA_PORT, MUX_CTRLA_PIN, gpioModePushPull, 0);
    GPIO_PinModeSet(MUX_CTRLB_PORT, MUX_CTRLB_PIN, gpioModePushPull, 0);
    GPIO_PinModeSet(MUX_CTRLC_PORT, MUX_CTRLC_PIN, gpioModePushPull, 0);
  }

  /* Use internal memory if it is GG 1M byte part */
  if ((SYSTEM_GetFamily() == systemPartFamilyGiant)
      && (SYSTEM_GetFlashSize() == 1024))
  {
    intMemValid = true;
  }
}
