/**************************************************************************//**
 * @file  state_machine.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdint.h>
#include <stdbool.h>
#include <setjmp.h>
#include <stdio.h>
#include "em_gpio.h"
#include "dap.h"
#include "utils.h"
#include "state_machine.h"
#include "errors.h"
#include "debug_lock.h"
#include "flash_write.h"
#include "kits.h"
#include "delay.h"
#include "progconfig.h"


extern char deviceName[20];
extern volatile uint32_t chipType;

extern volatile uint32_t mainFlashSize;
extern volatile uint32_t mainPageSizeMask;
extern volatile uint32_t mainFileSize;
extern volatile uint32_t mainStartAddr;

extern volatile uint32_t userPageSize;
extern volatile uint32_t userFileSize;
extern volatile uint32_t userStartAddr;

extern volatile bool progUserPage;
extern volatile bool progMainFlash;
extern volatile bool debugLockAfterProg;
extern volatile bool aapLockAfterProg;

extern volatile bool pressPb0;
extern volatile bool pressPb1;

/**********************************************************
 * Handles the Connect state
 * @returns
 *   Connect status
 **********************************************************/  
bool stateConnect(void)
{
  uint32_t retry = CONNECT_RETRY_COUNT;
  bool chipConnected = false;    
  bool abort = false;
 
  /* Reset target */
  hardResetTarget();
    
  delayMs(RESET_DELAY);
    
  /* Try connecting several times */
  while ( retry-- > 0 ) 
  {
    TRY
    /* Try to connect */
    connectToTarget();
        
    /* If we get here the connection was successful */
    chipConnected = true;
        
    /* Stop retrying */
    abort = true;
    CATCH
        
    if ( errorCode == SWD_ERROR_MCU_LOCKED ) 
    {
      /* Abort if the target is locked */
      printf("Target is locked\n");
      chipConnected = false;
          
      /* Stop retrying */
      abort = true;
    }
    else 
    {
      /* Print error message if connection failed */
      printf("SWD Error: %d\n", errorCode);
      delayUs(200);
    }
    ENDTRY
        
    /* We cannot return in the middle of a TRY block. 
     * Check flag and return from this point */
    if ( abort )
    {
      return chipConnected;
    }
  }
    
  /* Connection failed. */
  chipConnected = false;
  printf("Fail - %d retries\n", CONNECT_RETRY_COUNT);
  return chipConnected;
}


/**********************************************************
 * Handles the Lock state. This state enables Debug Lock.
 **********************************************************/  
void stateDebugLock(void)
{
  setLcdRow2();
  TRY
    /* Execute command */
    lockTarget();
    printf("Target debug locked\n");
  CATCH
    /* Print error message */
    setErrorLed(true);
    printf("Debug lock fail: %d\n", errorCode);
  ENDTRY;
}


/**********************************************************
 * Handles the Lock state. This state enables AAP Lock.
 **********************************************************/  
void stateAapLock(void)
{
  setLcdRow2();
  TRY
    /* Execute command */
    aapLockTarget();
    printf("Target AAP locked\n");
  CATCH
    /* Print error message */
    setErrorLed(true);
    printf("AAP lock fail: %d\n", errorCode);
  ENDTRY;
}


/**********************************************************
 * Program the device 
 * @returns
 *   Error LED status
 **********************************************************/
bool progDevice(void)
{
  uint32_t i;
  uint32_t j;
  
  /* Connect to programm device */
  if (stateConnect())
  {
    if (progMainFlash && !progUserPage)
    {
      /* Program main flash only */
      i = mainStartAddr;
      if ((i & mainPageSizeMask) != 0)
      {
        printf("Wrong start address\n");
        goto connectError;
      }
      
      j = mainFlashSize;
      if ((i + mainFileSize) > j)
      {
        printf("Binary is too big\n");
        goto connectError;
      }
      
      /* Start programming if connect and valid program start address */
      setBusyLed(true);
      setErrorLed(false);
      setLcdRow2();
      printf("%s\n", deviceName);
      
      TRY
        /* Execute command */
        flashMainWithMsc(i, mainFileSize);
      CATCH
        /* Print error message */
        setErrorLed(true);
        printf("SWD error: %d\n", errorCode);
      ENDTRY
    }
    
    else if (!progMainFlash && progUserPage)
    {
      /* Program user page only */
      j = userPageSize;
      if (userFileSize > j)
      {
        printf("Binary is too big\n");
        goto connectError;
      }
      
      /* Start programming if connect and valid size */
      setBusyLed(true);
      setErrorLed(false);
      setLcdRow2();
      printf("%s\n", deviceName);

      TRY
        /* Execute command */
        flashUserWithMsc(userFileSize);
      CATCH
        /* Print error message */
        setErrorLed(true);
        printf("SWD error: %d\n", errorCode);
      ENDTRY
    }
    else
    {
      /* Program main flash and user page */
      j = userPageSize;
      if (userFileSize > j)
      {
        printf("Binary is too big\n");
        goto connectError;
      }

      i = mainStartAddr;
      if ((i & mainPageSizeMask) != 0)
      {
        printf("Wrong start address\n");
        goto connectError;
      }

      j = mainFlashSize;
      if ((i + mainFileSize) > j)
      {
        printf("Binary is too big\n");
        goto connectError;
      }
      j = mainFileSize;

      /* Start programming if connect and valid program start address */
      setBusyLed(true);
      setErrorLed(false);
      setLcdRow2();
      printf("%s\n", deviceName);

      TRY
        /* Execute command */
        flashMainUserWithMsc(i, j, userFileSize);
      CATCH
        /* Print error message */
        setErrorLed(true);
        printf("SWD error: %d\n", errorCode);
      ENDTRY
    }
    setBusyLed(false);
  }
  else
  {
    /* Connect fail or illeagl address */
  connectError:
    setErrorLed(true);
  }

  /* Reset key status */
  __disable_irq();
  pressPb0 = false;
  pressPb1 = false; 
  __enable_irq();
  
  return(bool)(GPIO_PinOutGet(GPIO_LED0_PORT, GPIO_LED0_PIN));
}


/**********************************************************
 * Erase the main or user page flash 
 * @param mainOrUser
 *   True for main flash, false for user page
 * @returns
 *   Error LED status
 **********************************************************/
bool eraseFlash(bool mainOrUser)
{
  if (stateConnect())
  {
    /* Start erase if connect */
    setBusyLed(true);
    setErrorLed(false);
    displayClear();
    printf("%s\n", deviceName);
    
    TRY
      /* Execute command */
      if (mainOrUser)
      {
        erasePages(0, mainFlashSize);
        verifyFlashIsErased();
        printf("Erase Main done\n");
      }
      else
      {
        eraseUserPage();
        verifyUserIsErased();
        printf("Erase User done\n");
      }      
    CATCH
      /* Print error message */
      setErrorLed(true);
      if (mainOrUser)
      {
        printf("Erase Main error\n");
      }
      else
      {
        printf("Erase User error\n");
      }      
    ENDTRY
      setBusyLed(false);
  }
  else
  {
    /* Connect fail */
    setErrorLed(true);
  }
  return(bool)(GPIO_PinOutGet(GPIO_LED0_PORT, GPIO_LED0_PIN));
}
