/**************************************************************************//**
 * @file  utils.c
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdio.h>
#include "em_gpio.h"
#include "dap.h"
#include "utils.h"
#include "errors.h"
#include "delay.h"
#include "progconfig.h"
#include "usbtomemory.h"
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
#include "nandutil.h"
#else
#include "spiflash.h"
#endif

volatile uint32_t mscWriteCtrl;
volatile uint32_t mscWriteCmd;
volatile uint32_t mscAddrb;
volatile uint32_t mscWdata;
volatile uint32_t mscStatusReg;
volatile uint32_t mscMassLock;

extern volatile bool newFamily;
extern volatile uint32_t chipType;
extern volatile uint32_t mainPageSize;
extern volatile uint32_t mainPageSizeMask;
extern volatile uint32_t mainFlashSize;
extern volatile uint32_t userPageSize;
extern char deviceName[20];
extern uint8_t memoryReadBuffer[];

static void getDeviceName(char deviceName[]);


/**********************************************************
 * Reads the unique ID of the target from the DI page.
 * 
 * @returns
 *    The unique ID of target
 **********************************************************/
uint64_t readUniqueId(void)
{
  uint64_t uniqueId;
  
  /* Retrive high part of unique ID */
  uniqueId = readMem(UNIQUE_ID_HIGH_ADDR);
  uniqueId = uniqueId << 32;
  
  /* Retrive low part of unique ID */
  uniqueId |= readMem(UNIQUE_ID_LOW_ADDR);  

  return uniqueId;
}


/**********************************************************
 * Halts the target CPU
 **********************************************************/
void haltTarget(void)
{
  int timeout = DEBUG_EVENT_TIMEOUT;
  writeAP(AP_TAR, (uint32_t)&(CoreDebug->DHCSR));
  writeAP(AP_DRW, STOP_CMD);
  
  uint32_t dhcrState;
  do {
    writeAP(AP_TAR, (uint32_t)&(CoreDebug->DHCSR));
    readAP(AP_DRW, &dhcrState);
    readDP(DP_RDBUFF, &dhcrState);
    timeout--;
  } while ( !(dhcrState & CoreDebug_DHCSR_S_HALT_Msk) && timeout > 0 ); 
  
  if ( !(dhcrState & CoreDebug_DHCSR_S_HALT_Msk) ) {
    RAISE(SWD_ERROR_TIMEOUT_HALT);
  }
}


/**********************************************************
 * Lets the target CPU run freely (stops halting)
 **********************************************************/
void runTarget(void)
{
  writeAP(AP_TAR, (uint32_t)&(CoreDebug->DHCSR));
  writeAP(AP_DRW, RUN_CMD);
}


/**********************************************************
 * Single steps the target
 **********************************************************/
void stepTarget(void)
{
  writeAP(AP_TAR, (uint32_t)&(CoreDebug->DHCSR));
  writeAP(AP_DRW, STEP_CMD);
}


/**********************************************************
 * Retrieve the device name from the DI page of the target
 * 
 * @param deviceName[out]
 *    Device name is stored in this buffer when 
 *    the function returns. The calling function is
 *    responsible for allocating memory for the string
 **********************************************************/
static void getDeviceName(char deviceName[])
{
  char familyCode[10];
  
  uint32_t part = readMem((uint32_t)&(DEVINFO->PART));
  uint32_t msize = readMem((uint32_t)&(DEVINFO->MSIZE));
  chipType = (part &  _DEVINFO_PART_DEVICE_FAMILY_MASK) >> _DEVINFO_PART_DEVICE_FAMILY_SHIFT;  
  uint32_t partNum = (part &  _DEVINFO_PART_DEVICE_NUMBER_MASK) >> _DEVINFO_PART_DEVICE_NUMBER_SHIFT;  

  switch (chipType)
  {
  case _DEVINFO_PART_DEVICE_FAMILY_EFM32GG:     /* Giant Gecko   */
    sprintf(familyCode, "%s", "EFM32GG");
    mainPageSize = 4096;
    userPageSize = 2048;
    mainPageSizeMask = 0xfff;
    setMscBaseAddrP1();
    newFamily = false;
    break;

  case _DEVINFO_PART_DEVICE_FAMILY_EFM32LG:     /* Leopard Gecko */
    sprintf(familyCode, "%s", "EFM32LG");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP1();
    newFamily = false;
    break;

  case _DEVINFO_PART_DEVICE_FAMILY_EFM32WG:     /* Wonder Gecko  */
    sprintf(familyCode, "%s", "EFM32WG");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP1();
    newFamily = false;
    break;

  case _DEVINFO_PART_DEVICE_FAMILY_EFM32G:      /* Gecko */
    sprintf(familyCode, "%s", "EFM32G");
    mainPageSize = 512;
    userPageSize = 512;
    mainPageSizeMask = 0x1ff;
    setMscBaseAddrP1();
    newFamily = false;
    break;

  case _DEVINFO_PART_DEVICE_FAMILY_EFM32TG:     /* Tiny Gecko */
    sprintf(familyCode, "%s", "EFM32TG");
    mainPageSize = 512;
    userPageSize = 512;
    mainPageSizeMask = 0x1ff;
    setMscBaseAddrP1();
    newFamily = false;
    break;

  case _DEVINFO_PART_DEVICE_FAMILY_EFM32ZG:     /* Zero Gecko */
    sprintf(familyCode, "%s", "EFM32ZG");
    mainPageSize = 1024;
    userPageSize = 1024;
    mainPageSizeMask = 0x3ff;
    setMscBaseAddrP1();
    newFamily = false;
    break;
    
  case _DEVINFO_PART_DEVICE_FAMILY_EFM32HG:     /* Happy Gecko */
    sprintf(familyCode, "%s", "EFM32HG");
    mainPageSize = 1024;
    userPageSize = 1024;
    mainPageSizeMask = 0x3ff;
    setMscBaseAddrP1();
    newFamily = false;
    break;
    
  case _DEVINFO_PART_DEVICE_FAMILY_EZR32LG:     /* EZR32 Leopard Gecko */
    sprintf(familyCode, "%s", "EZR32LG");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP1();
    newFamily = false;
    break;
    
  case _DEVINFO_PART_DEVICE_FAMILY_EZR32WG:     /* EZR32 Wonder Gecko  */
    sprintf(familyCode, "%s", "EZR32WG");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP1();
    newFamily = false;
    break;

  case _DEVINFO_PART_DEVICE_FAMILY_EZR32HG:     /* EZR32 Happy Gecko */
    sprintf(familyCode, "%s", "EZR32HG");
    mainPageSize = 1024;
    userPageSize = 1024;
    mainPageSizeMask = 0x3ff;
    setMscBaseAddrP1();
    newFamily = false;
    break;
    
  case _DEVICE_FAMILY_EFM32PG1B:                /* Pearl Gecko */
    sprintf(familyCode, "%s", "EFM32PG1B");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFM32JG1B:                /* Jade Gecko */
    sprintf(familyCode, "%s", "EFM32JG1B");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32MG1P:                /* EFR32MG */
    sprintf(familyCode, "%s", "EFR32MG1P");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32MG1B:
    sprintf(familyCode, "%s", "EFR32MG1B");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32MG1V:
    sprintf(familyCode, "%s", "EFR32MG1V");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32BG1P:                /* EFR32BG */
    sprintf(familyCode, "%s", "EFR32BG1P");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32BG1B:
    sprintf(familyCode, "%s", "EFR32BG1B");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32BG1V:
    sprintf(familyCode, "%s", "EFR32BG1V");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32SG1P:                /* EFR32SG */
    sprintf(familyCode, "%s", "EFR32SG1P");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32SG1B:
    sprintf(familyCode, "%s", "EFR32SG1B");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32SG1V:
    sprintf(familyCode, "%s", "EFR32SG1V");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32FG1P:                /* EFR32FG */
    sprintf(familyCode, "%s", "EFR32FG1P");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32FG1B:
    sprintf(familyCode, "%s", "EFR32FG1B");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  case _DEVICE_FAMILY_EFR32FG1V:
    sprintf(familyCode, "%s", "EFR32FG1V");
    mainPageSize = 2048;
    userPageSize = 2048;
    mainPageSizeMask = 0x7ff;
    setMscBaseAddrP2();
    newFamily = true;
    break;
    
  default:
    chipType = 0;                               /* Unknown family */
    break;
  }
  
  if (chipType)
  {
    mainFlashSize = (msize & _DEVINFO_MSIZE_FLASH_MASK) >> _DEVINFO_MSIZE_FLASH_SHIFT;
    sprintf(deviceName, "%s%luF%lu", familyCode, partNum, mainFlashSize);
    mainFlashSize *= 1024;
  }
}


/**********************************************************
 * Get the wrap-around period for the TAR register. This
 * is device dependent and affects the burst write
 * algorithms. This function hard-codes the values
 * based on information from the DI-page. 
 * 
 * @returns
 *   The wrap-around period of the TAR register
 **********************************************************/
uint32_t getTarWrap(void)
{
  /* Hard-code result based on device family, ZG/HG has 1kB and others have 4kB wrap */
  switch (chipType)
  {
  case _DEVINFO_PART_DEVICE_FAMILY_EFM32GG:     /* Giant Gecko   */
  case _DEVINFO_PART_DEVICE_FAMILY_EFM32LG:     /* Leopard Gecko */
  case _DEVINFO_PART_DEVICE_FAMILY_EFM32WG:     /* Wonder Gecko  */
  case _DEVINFO_PART_DEVICE_FAMILY_EFM32G:      /* Gecko */
  case _DEVINFO_PART_DEVICE_FAMILY_EFM32TG:     /* Tiny Gecko */
  case _DEVINFO_PART_DEVICE_FAMILY_EZR32LG:     /* EZR32 Leopard Gecko */
  case _DEVINFO_PART_DEVICE_FAMILY_EZR32WG:     /* EZR32 Wonder Gecko  */
  case _DEVICE_FAMILY_EFM32PG1B:                /* Pearl Gecko */
  case _DEVICE_FAMILY_EFM32JG1B:                /* Jade Gecko */
  case _DEVICE_FAMILY_EFR32MG1P:                /* EFR32MG */
  case _DEVICE_FAMILY_EFR32MG1B:
  case _DEVICE_FAMILY_EFR32MG1V:
  case _DEVICE_FAMILY_EFR32BG1P:                /* EFR32BG */
  case _DEVICE_FAMILY_EFR32BG1B:
  case _DEVICE_FAMILY_EFR32BG1V:
  case _DEVICE_FAMILY_EFR32SG1P:                /* EFR32SG */
  case _DEVICE_FAMILY_EFR32SG1B:
  case _DEVICE_FAMILY_EFR32SG1V:
  case _DEVICE_FAMILY_EFR32FG1P:                /* EFR32FG */
  case _DEVICE_FAMILY_EFR32FG1B:
  case _DEVICE_FAMILY_EFR32FG1V:
    return TAR_WRAP_4K;

  case _DEVINFO_PART_DEVICE_FAMILY_EFM32ZG:     /* Zero Gecko */
  case _DEVINFO_PART_DEVICE_FAMILY_EFM32HG:     /* Happy Gecko */
  case _DEVINFO_PART_DEVICE_FAMILY_EZR32HG:     /* EZR32 Happy Gecko */
    return TAR_WRAP_1K;

  default:                                      /* Unknown family */
    return 0;
  }
}


/**********************************************************
 * Resets the target CPU by using the AIRCR register. 
 * The target will be halted immediately when coming
 * out of reset. Does not reset the debug interface.
 **********************************************************/
void resetAndHaltTarget(void)
{
  uint32_t dhcsr;
  int timeout = DEBUG_EVENT_TIMEOUT;
  
  /* Halt target first. This is necessary before setting
   * the VECTRESET bit */
  haltTarget();
  
  /* Set halt-on-reset bit */
  writeMem((uint32_t)&(CoreDebug->DEMCR), CoreDebug_DEMCR_VC_CORERESET_Msk);
  
  /* Clear exception state and reset target */
  writeAP(AP_TAR, (uint32_t)&(SCB->AIRCR));
  writeAP(AP_DRW, (0x05FA << SCB_AIRCR_VECTKEY_Pos) |
                  SCB_AIRCR_VECTCLRACTIVE_Msk |
                  SCB_AIRCR_VECTRESET_Msk);
    
  /* Wait for target to reset */
  do { 
    delayUs(10);
    timeout--;
    dhcsr = readMem((uint32_t)&(CoreDebug->DHCSR));
  } while ( dhcsr & CoreDebug_DHCSR_S_RESET_ST_Msk );
  
  
  /* Check if we timed out */
  dhcsr = readMem((uint32_t)&(CoreDebug->DHCSR));
  if ( dhcsr & CoreDebug_DHCSR_S_RESET_ST_Msk ) 
  {
    RAISE(SWD_ERROR_TIMEOUT_WAITING_RESET);
  }
  
  /* Verify that target is halted */
  if ( !(dhcsr & CoreDebug_DHCSR_S_HALT_Msk) ) 
  {
    RAISE(SWD_ERROR_TARGET_NOT_HALTED);
  }
}


/**********************************************************
 * Resets the target CPU by using the AIRCR register. 
 * Does not reset the debug interface
 **********************************************************/
void resetTarget(void)
{  
  uint32_t dhcsr;
  int timeout = DEBUG_EVENT_TIMEOUT;
  
  /* Clear the VC_CORERESET bit */
  writeMem((uint32_t)&(CoreDebug->DEMCR), 0);
  
  /* Do a dummy read of sticky bit to make sure it is cleared */
  readMem((uint32_t)&(CoreDebug->DHCSR));
  dhcsr = readMem((uint32_t)&(CoreDebug->DHCSR));
  
  /* Reset CPU */
  writeMem((uint32_t)&(SCB->AIRCR), AIRCR_RESET_CMD);
  
  /* Wait for reset to complete */
  /* First wait until sticky bit is set. This means we are
   * or have been in reset */
  delayUs(100);
  do { 
    delayUs(10);
    dhcsr = readMem((uint32_t)&(CoreDebug->DHCSR));
    timeout--;
  } while ( !(dhcsr & CoreDebug_DHCSR_S_RESET_ST_Msk) && timeout > 0 );
    
  /* Throw error if sticky bit is never set */
  if ( !(dhcsr & CoreDebug_DHCSR_S_RESET_ST_Msk) ) {
    RAISE(SWD_ERROR_TIMEOUT_WAITING_RESET);
  }
    
  /* Wait for sticky bit to be cleared. When bit is cleared are we out of reset */
  timeout = DEBUG_EVENT_TIMEOUT;
  do { 
    delayUs(10);
    dhcsr = readMem((uint32_t)&(CoreDebug->DHCSR));
    timeout--;
  } while ( (dhcsr & CoreDebug_DHCSR_S_RESET_ST_Msk) && timeout > 0 );
  
  /* Throw error if bit is never cleared */
  if ( dhcsr & CoreDebug_DHCSR_S_RESET_ST_Msk ) {
    RAISE(SWD_ERROR_TIMEOUT_WAITING_RESET);
  }
}


/**********************************************************
 * Performs a pin reset on the target
 **********************************************************/
void hardResetTarget(void)
{
  GPIO_PinOutClear((GPIO_Port_TypeDef)RESET_PORT, RESET_PIN);
  
  delayMs(RESET_PULSE_WIDTH);
  
  GPIO_PinOutSet((GPIO_Port_TypeDef)RESET_PORT, RESET_PIN);
}


/**********************************************************
 * Waits for the REGRDY bit in DCRSR. This bit indicates
 * that the DCRDR/DCRSR registers are ready to accept
 * new data. 
 **********************************************************/
void waitForRegReady(void)
{
  uint32_t dhcsr;
  do {
    dhcsr = readMem((uint32_t)&CoreDebug->DHCSR);
  } while ( !(dhcsr & CoreDebug_DHCSR_S_REGRDY_Msk) );
}


/**********************************************************
 * Verifies the current firmware against the external
 * memory. This function assumes the firmware has
 * been written to target at startAddr. 
 *
 * @param startAddr
 *    Start address to verify (must in multiple of page size)
 * 
 * @param size
 *    Size (in bytes) of firmware image
 * 
 * @returns 
 *    True if target firmware matches external memory. 
 *    False otherwise. 
 **********************************************************/
bool verifyFirmware(uint32_t startAddr, uint32_t size)
{
  uint32_t w;
  uint32_t pageNum;
  uint32_t ptrIndex = 0;
  uint32_t *wordPtr = (uint32_t *)memoryReadBuffer;

  bool ret = true;
  uint32_t value;
  uint32_t tarWrap =  getTarWrap();

  if (startAddr == USER_PAGE_ADDR)
  {
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    pageNum = USER_DATA_ADDR_PAGE;
#else
    pageNum = USER_DATA_SECTOR;
#endif
  }
  else
  {
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    pageNum = MAIN_DATA_ADDR_PAGE;
#else
    pageNum = MAIN_DATA_PAGE;
#endif
  }
  
  /* Adjust size with start address */
  size += startAddr;

  /* Set autoincrement on TAR */
  writeAP(AP_CSW, AP_CSW_DEFAULT | AP_CSW_AUTO_INCREMENT);
  
  for ( w=startAddr; w<size; w+=4, ptrIndex++) 
  {   
    /* Last page finish, read another page */
#if (EXTERNAL_MEMORY == EBI_NAND_FLASH)
    if ((w & NAND_FLASH_PAGE_SIZE_MASK) == 0)
    {
      nandFlashReadPage(pageNum, memoryReadBuffer);
      pageNum++;
      ptrIndex = 0;
    }
#else
    if ((w & SPIF_PAGESIZE_MASK) == 0)
    {
      spiFlashBufferRead(memoryReadBuffer, pageNum, SPIF_PAGESIZE);
      pageNum += SPIF_PAGESIZE;
      ptrIndex = 0;
    }
#endif
        
    /* TAR must be initialized at every TAR wrap boundary
     * because the autoincrement wraps around at these */
    if ( (w & tarWrap) == 0 )
    {
      writeAP(AP_TAR, w);
      
      /* Do one dummy read. Subsequent reads will return the 
       * correct result. */
      readAP(AP_DRW, &value);
    }
    
    /* Read the value from addr */
    readAP(AP_DRW, &value);
    
    /* Verify that the read value matches what is expected */
    if ( value != wordPtr[ptrIndex] ) 
    {
      printf("Fail at 0x%.8lx\n", w);
      printf("Flash 0x%.8lx\n", value);
      printf("Memory 0x%.8lx\n", wordPtr[ptrIndex]);
      ret = false;
      break;
    }
  }
  
  /* Disable autoincrement on TAR */
  writeAP(AP_CSW, AP_CSW_DEFAULT);
     
  return ret;
}


/**********************************************************
 * Verifies the current firmware against the internal
 * memory. This function assumes the firmware has
 * been written to target at startAddr. 
 *
 * @param startAddr
 *    Start address to verify (must in multiple of page size)
 * 
 * @param size
 *    Size (in bytes) of firmware image
 * 
 * @returns 
 *    True if target firmware matches internal memory. 
 *    False otherwise. 
 **********************************************************/
bool verifyFirmwareInt(uint32_t startAddr, uint32_t size)
{
  uint32_t w;
  uint32_t *wordPtr = (uint32_t *)(INT_START_ADDR);

  bool ret = true;
  uint32_t value;
  uint32_t tarWrap =  getTarWrap();

  /* Adjust size with start address */
  size += startAddr;

  /* Set autoincrement on TAR */
  writeAP(AP_CSW, AP_CSW_DEFAULT | AP_CSW_AUTO_INCREMENT);
  
  for ( w=startAddr; w<size; w+=4, wordPtr++) 
  {   
    /* TAR must be initialized at every TAR wrap boundary
     * because the autoincrement wraps around at these */
    if ( (w & tarWrap) == 0 )
    {
      writeAP(AP_TAR, w);
      
      /* Do one dummy read. Subsequent reads will return the 
       * correct result. */
      readAP(AP_DRW, &value);
    }
    
    /* Read the value from addr */
    readAP(AP_DRW, &value);
    
    /* Verify that the read value matches what is expected */
    if ( value != *wordPtr ) 
    {
      printf("Fail at 0x%.8lx\n", w);
      printf("Flash 0x%.8lx\n", value);
      printf("Memory 0x%.8lx\n", *wordPtr);
      ret = false;
      break;
    }
  }
  
  /* Disable autoincrement on TAR */
  writeAP(AP_CSW, AP_CSW_DEFAULT);
     
  return ret;
}


/**********************************************************
 * Returns true if the @param dpId is a valid
 * IDCODE value. 
 **********************************************************/
bool verifyDpId(uint32_t dpId)
{
  if ( dpId == EFM32_DPID_1 ) 
  {
    return true; // Valid for G, LG, GG, TG, WG
  }
  else if ( dpId == EFM32_DPID_2 )
  {
    return true; // Valid for ZG
  } 
  else if ( dpId == EFM32_DPID_3 )
  {
    return true; // Valid for SWMD
  }   
  else 
  {
    return false;
  }
}

/**********************************************************
 * Returns true if the @param apId is a valid
 * IDR value for the AHB-AP. 
 **********************************************************/
bool verifyAhbApId(uint32_t apId)
{
  if ( apId == EFM32_AHBAP_ID_1 ) 
  {
    return true; // Valid for G, LG, GG, TG, WG
  }
  else if ( apId == EFM32_AHBAP_ID_2 )
  {
    return true; // Valid for ZG
  } 
  else 
  {
    return false;
  }
}


/**********************************************************
 * This function will check if a M0+ device is
 * locked. The method is different from other MCUs,
 * we have to read the AAP registers from the internal
 * memory space.
 * 
 * This process can fail (we receive a FAULT response) on 
 * other devices so wee need to check for failure on the AP 
 * transaction and clear the STICKYERR flag in CTRL/STAT 
 * before continuing. 
 **********************************************************/
void checkIfMzeroIsLocked(void)
{
  int readError = SWD_ERROR_OK;
  uint32_t readVal;
  uint32_t apId = 0;
  
  /* Try reading the AAP_IDR register on Zero. 
   * Do in a separate TRY/CATCH block in case to allow
   * failure in this transaction.
   */
  TRY 
    apId = readMem(AAP_IDR_ZERO);
  CATCH
    /* If transaction failed. Store error code */
    readError = errorCode;
  ENDTRY
  
  /* If the transaction was OK we check if we got
   * access to the AAP registers. If we do, the device
   * is locked. 
   */
  if ( readError == SWD_ERROR_OK )
  {
    if ( apId == EFM32_AAP_ID_P1 ) 
    {
      RAISE(SWD_ERROR_MCU_LOCKED);
    }
  } 
  /* We received a FAULT or WAIT error. This is normal on non-ZG devices. 
   * If this happens we have to clear the STICKYERR flag before continuing. 
   * If we do not do this all subsequent AP transactions will fail. 
   */
  else if ( readError == SWD_ERROR_FAULT || readError == SWD_ERROR_WAIT )
  {
    /* Read CTRL/STAT register */
    readDP(DP_CTRL, &readVal);
    
    /* See if STICKYERR is set */
    if ( readVal & (1 << 5) )
    {
      /* Clear sticky error */
      writeDP(DP_ABORT, (1 << 2));
    } 
  } 
  /* We received another error, e.g. protocol error. 
   * Report the error back to the calling function */
  else  
  {
    RAISE(readError);
  }
}


/**********************************************************
 * Writes a value to a CPU register in the target.
 * 
 * @param reg
 *   The register number to write to
 * 
 * @param value
 *   The value to write to the register
 **********************************************************/
void writeCpuReg(int reg, uint32_t value)
{
  /* Wait until debug register is ready to accept new data */
  waitForRegReady();
  
  /* Write value to Data Register */
  writeAP(AP_TAR, (uint32_t)&(CoreDebug->DCRDR));
  writeAP(AP_DRW, value);
  
  /* Write register number ot Selector Register. 
   * This will update the CPU register */
  writeAP(AP_TAR, (uint32_t)&(CoreDebug->DCRSR));
  writeAP(AP_DRW, 0x10000 | reg); 
}


/**********************************************************
 * Performs the initialization sequence on the SW-DP. 
 * After this completes the debug interface can be used. 
 * Raises an exception on any error during connection. 
 **********************************************************/
void connectToTarget(void)
{
  uint32_t dpId,apId;
  
  /* Delay for retry */
  delayUs(500);
  dpId = initDp();
  
  /* Verify that the DP returns the correct ID */
  if ( !verifyDpId(dpId) ) 
  {
    printf("IDCODE=0x%.8lx\n", dpId);
    RAISE(SWD_ERROR_INVALID_IDCODE);
  }
  
  /* Verify that the AP returns the correct ID */
  int retry = AHB_IDR_RETRY_COUNT;
  while ( retry > 0 )
  {
    apId = readApId();
    if ( verifyAhbApId(apId) )
    {
      /* Success. AHB-AP registers found */
      break;
    }
    
    retry--;
  }
  
  /* In case we did NOT find the AHB-AP registers check 
   * if the chip is locked or if some other error ocurred. */  
  if ( !verifyAhbApId(apId) ) 
  {
    /* If the ID is from AAP, the MCU is locked. 
     * Debug access is not available. */
    if ( (apId == EFM32_AAP_ID_P1) || (apId == EFM32_AAP_ID_P2) ) 
    {
      RAISE(SWD_ERROR_MCU_LOCKED);
    } 
    else
    {
      /* The AP ID was not from AAP nor AHB-AP. This is an error. */ 
      printf("IDR=0x%.8lx\n", apId);
      RAISE(SWD_ERROR_INVALID_IDR);  
    }
  } 
    
  /* Set up parameters for AHB-AP. This must be done before accessing
   * internal memory. */
  initAhbAp();
  
  /*
   * On M0+ it is possible that the device is locked
   * even though we reach this point. We have to see if 
   * we can access the AAP registers which are memory mapped 
   * on these devices. The function will raise an exception
   * if the device is a M0+ and is currently locked.
   */
  if ( apId == EFM32_AHBAP_ID_2 )
  {
    checkIfMzeroIsLocked();
  }
  
  haltTarget();
   
  /* Get device name */
  getDeviceName(deviceName);
  if (chipType == 0)
  {
    printf("Unknown device\n");
    RAISE(SWD_ERROR_INVALID_IDR);  
  }    
}
