/**************************************************************************//**
 * @file  utils.h
 * @brief EFM32 standalone programmer.
 * @version 0.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#ifndef _UTILS_H_
#define _UTILS_H_

/* MSC register base address and offset */
#define MSCBASE_ADDR_P1         0x400c0000
#define MSCBASE_ADDR_P2         0x400e0000
#define WRITECTRL_OFFSET        0x08
#define WRITECMD_OFFSET         0x0c
#define ADDRB_OFFSET            0x10
#define WDATA_OFFSET            0x18
#define STATUS_OFFSET           0x1c
#define MASSLOCK_OFFSET         0x54

/* Family code */
#define _DEVICE_FAMILY_EFM32PG1B        81 /* EFM32PG */
#define _DEVICE_FAMILY_EFM32JG1B        83 /* EFM32JG */
#define _DEVICE_FAMILY_EFR32MG1P        16 /* EFR32MG */
#define _DEVICE_FAMILY_EFR32MG1B        17
#define _DEVICE_FAMILY_EFR32MG1V        18
#define _DEVICE_FAMILY_EFR32BG1P        19 /* EFR32BG */
#define _DEVICE_FAMILY_EFR32BG1B        20
#define _DEVICE_FAMILY_EFR32BG1V        21
#define _DEVICE_FAMILY_EFR32SG1P        22 /* EFR32SG */
#define _DEVICE_FAMILY_EFR32SG1B        23
#define _DEVICE_FAMILY_EFR32SG1V        24
#define _DEVICE_FAMILY_EFR32FG1P        25 /* EFR32FG */
#define _DEVICE_FAMILY_EFR32FG1B        26
#define _DEVICE_FAMILY_EFR32FG1V        27

/* TAR wrap mask */
#define TAR_WRAP_4K                     0xFFF
#define TAR_WRAP_1K                     0x3FF

uint64_t readUniqueId(void);

void resetTarget(void);
void resetAndHaltTarget(void);
void stepTarget(void);
void runTarget(void);
void haltTarget(void);
void hardResetTarget(void);

void initAhbAp(void);
void waitForRegReady(void);

bool verifyFirmware(uint32_t startAddr, uint32_t size);
bool verifyFirmwareInt(uint32_t startAddr, uint32_t size);
void connectToTarget(void);

void writeCpuReg(int reg, uint32_t value);

bool verifyDpId(uint32_t dpId);
bool verifyAhbApId(uint32_t dpId);

uint32_t getTarWrap(void);
void checkIfMzeroIsLocked(void);

extern volatile uint32_t mscWriteCtrl;
extern volatile uint32_t mscWriteCmd;
extern volatile uint32_t mscAddrb;
extern volatile uint32_t mscWdata;
extern volatile uint32_t mscStatusReg;
extern volatile uint32_t mscMassLock;

/**********************************************************
 * Setup MSC registers' address for platform 1
 **********************************************************/
__STATIC_INLINE void setMscBaseAddrP1(void)
{
  mscWriteCtrl = (MSCBASE_ADDR_P1 + WRITECTRL_OFFSET);
  mscWriteCmd = (MSCBASE_ADDR_P1 + WRITECMD_OFFSET);
  mscAddrb = (MSCBASE_ADDR_P1 + ADDRB_OFFSET);
  mscWdata = (MSCBASE_ADDR_P1 + WDATA_OFFSET);
  mscStatusReg = (MSCBASE_ADDR_P1 + STATUS_OFFSET);
  mscMassLock = (MSCBASE_ADDR_P1 + MASSLOCK_OFFSET);
}

/**********************************************************
 * Setup MSC registers' address for platform 2
 **********************************************************/
__STATIC_INLINE void setMscBaseAddrP2(void)
{
  mscWriteCtrl = (MSCBASE_ADDR_P2 + WRITECTRL_OFFSET);
  mscWriteCmd = (MSCBASE_ADDR_P2 + WRITECMD_OFFSET);
  mscAddrb = (MSCBASE_ADDR_P2 + ADDRB_OFFSET);
  mscWdata = (MSCBASE_ADDR_P2 + WDATA_OFFSET);
  mscStatusReg = (MSCBASE_ADDR_P2 + STATUS_OFFSET);
  mscMassLock = (MSCBASE_ADDR_P2 + MASSLOCK_OFFSET);
}

/**********************************************************
 * Reads one word from internal memory
 * 
 * @param addr 
 *    The address to read from
 * 
 * @returns 
 *    The value at @param addr
 **********************************************************/
__STATIC_INLINE uint32_t readMem(uint32_t addr)
{
  uint32_t ret;

  writeAP(AP_TAR, addr);
  readAP(AP_DRW, &ret);
  readDP(DP_RDBUFF, &ret);
  return ret;
}

/**********************************************************
 * Writes one word to internal memory
 * 
 * @param addr 
 *    The address to write to 
 *
 * @param data
 *    The value to write
 * 
 * @returns 
 *    The value at @param addr
 **********************************************************/
__STATIC_INLINE void writeMem(uint32_t addr, uint32_t data)
{
  writeAP(AP_TAR, addr);
  writeAP(AP_DRW, data);
}

#endif
