/**************************************************************************//**
 * @file
 * @brief The DMADRV example
 * @version 1.0.0
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdio.h>
#include "em_chip.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "dmadrv.h"
#include "rtcdriver.h"
#include "retargetserial.h"

#define LOOP_COUNT              4
#define BUFFER_SIZE             8
#define DMA_TX_INTERVAL         3000

#if defined(_EFM32_GIANT_FAMILY)
/* For GG STK retarget USART */
#define DMA_USART_SIGNAL        dmadrvPeripheralSignal_USART1_TXBL
#else
/* For PG STK retarget USART */
#define DMA_USART_SIGNAL        dmadrvPeripheralSignal_USART0_TXBL
#endif

RTCDRV_TimerID_t id;
unsigned int channel;

uint8_t pingBuffer[BUFFER_SIZE];
uint8_t pongBuffer[BUFFER_SIZE];

/**************************************************************************//**
 * @brief  Call-back called when DMA transfer is complete
 *****************************************************************************/
bool transferComplete(unsigned int channel, unsigned int primary, void *user)
{
  uint32_t i;
  (void) user;

  /* Fill alternate buffer */
  if (primary & 0x01) 
  {
    for (i=0; i<BUFFER_SIZE-1; i++)
    {
      pingBuffer[i] += 2;
    }
  }
  else
  {
    for (i=0; i<BUFFER_SIZE-1; i++)
    {
      pongBuffer[i] += 2;
    }
  }
  
  /* End of Ping-Pong? */
  if (primary == LOOP_COUNT)
    return false;
  else
    return true;
}

/**************************************************************************//**
 * @brief RTC Callback to trigger DMA
 *****************************************************************************/
void rtcCallback( RTCDRV_TimerID_t id, void *user )
{
  uint32_t i;

  /* Initialize Ping-Pong buffers */
  for (i=0; i<BUFFER_SIZE-1; i++)
  {
    pingBuffer[i] = '1';
    pongBuffer[i] = '2';
  }
  pingBuffer[BUFFER_SIZE-1] = '\n';
  pongBuffer[BUFFER_SIZE-1] = '\n';

  /* Start the Ping-Pong DMA transfer. */
  DMADRV_MemoryPeripheralPingPong( channel,
                           DMA_USART_SIGNAL,
                           (void*)&(RETARGET_UART->TXDATA),
                           pingBuffer,
                           pongBuffer,
                           true,
                           BUFFER_SIZE,
                           dmadrvDataSize1,
                           transferComplete,
                           NULL );
}

/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main( void )
{
  /* Chip errata */
  CHIP_Init();

  /* Initialize DCDC regulator if available */
#if defined( _EMU_DCDCCTRL_MASK )  
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif
  
  /* Initialize HFXO */
  CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_DEFAULT;
  CMU_HFXOInit(&hfxoInit);

  /* Switch HFCLK to HFXO and disable HFRCO */
  CMU_ClockSelectSet(cmuClock_HF, cmuSelect_HFXO);
  CMU_OscillatorEnable(cmuOsc_HFRCO, false, false);
  
  /* Initialize USART. */
  RETARGET_SerialInit();
  RETARGET_SerialCrLf(1);
  
  /* Initialize RTC driver. */
  RTCDRV_Init();
  RTCDRV_AllocateTimer(&id);
  RTCDRV_StartTimer(id, rtcdrvTimerTypePeriodic, DMA_TX_INTERVAL, rtcCallback, NULL);
  
  /* Initialize DMA driver and request a DMA channel. */
  DMADRV_Init();
  DMADRV_AllocateChannel(&channel, NULL);
  
  while (1)
  {
    EMU_EnterEM1();
  }
}
