/**************************************************************************//**
 * @file
 * @brief The LDMA Examples
 * @version 1.0.0
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2016 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * This file is licensed under the Silabs License Agreement. See the file
 * "Silabs_License_Agreement.txt" for details. Before using this software for
 * any purpose, you must agree to the terms of that agreement.
 *
 ******************************************************************************/

#include <stdio.h>
#include "em_device.h"
#include "em_chip.h"
#include "em_adc.h"
#include "em_cmu.h"
#include "em_emu.h"
#include "em_prs.h"
#include "em_ldma.h"
#include "em_cryotimer.h"
#include "retargetserial.h"

/* DMA channel used for the examples */
#define USE_DMA_CHANNEL         0
#define USE_DMA_CH_MASK         (1 << USE_DMA_CHANNEL)

/* Memory to memory transfer buffer size and constant for GPIO PRS */
#define BUFFER_SIZE             127
#define TRANSFER_SIZE           (BUFFER_SIZE - 1)
#define GPIO_PRS_CHANNEL        1

/* ADC buffer size and constants for ADC setup */
#define ADC_BUFFER_SIZE         16
#define ADC_CLOCK               10000000
#define ADC_INPUT_PORT          adcPosSelAPORT1XCH22
#define ADC_PRS_CHANNEL         0
#define ADC_FIFO_DEPTH          4
#define ADC_DST_ADDR_OFFSET     0
#define ADC_DUMMY_VALUE         256

/* Constant for loop transfer */
#define LOOP_REPEAT             2

/* Constants for inter-channel sync transfer */
#define RX_SIZE_MAX             8
#define RX_TRIG_LENGTH          1
#define SYNC_SET                0x80
#define SYNC_CLEAR              0x80
#define MATCH_VALUE             0x80
#define WRI_ADDR                &GPIO->P[5].DOUTTGL
#define WRI_DATA                0x10

/* 2D buffer size and constants for 2D copy */
#define BUFFER_2D_WIDTH         64
#define BUFFER_2D_HEIGHT        64
#define TRANSFER_WIDTH          16
#define TRANSFER_HEIGHT         16
#define SRC_ROW_INDEX           16
#define SRC_COL_INDEX           16
#define DST_ROW_INDEX           48
#define DST_COL_INDEX           32

/* Ping-Pong buffer size and constant for Ping-Pong transfer */
#define PP_BUFFER_SIZE          8
#define PP_LOOP_COUNT           4

/* Descriptor linked list size */
#define LIST0_SIZE              4
#define LIST1_SIZE              2

/* Buffer for linked and looped transfer */
uint8_t str1[] = "String_1111\n";
uint8_t str2[] = "String_2222\n";
uint8_t str3[] = "String_3333\n";
uint8_t str4[] = "String_4444\n";

/* Buffer for single descriptor looped transfer */
uint32_t adcBuffer[ADC_BUFFER_SIZE];

/* Buffer for inter-channel sync transfer */
uint8_t rxSyncBuffer[RX_SIZE_MAX];
uint8_t ledon[] = "Press any key to toggle LED0\n";

/* Buffer for memory to memory transfer */
uint16_t srcBuffer[BUFFER_SIZE];
uint16_t dstBuffer[BUFFER_SIZE];

/* Buffer for 2D copy transfer */
uint8_t src2d[BUFFER_2D_WIDTH][BUFFER_2D_HEIGHT];
uint8_t dst2d[BUFFER_2D_WIDTH][BUFFER_2D_HEIGHT];

/* Buffer for Ping-Pong transfer */
uint8_t pingBuffer[PP_BUFFER_SIZE];
uint8_t pongBuffer[PP_BUFFER_SIZE];

/* Ping-Pong transfer count */
uint32_t ppCount;

/* Descriptor linked list for LDMA transfer */
LDMA_Descriptor_t descLink0[LIST0_SIZE];
LDMA_Descriptor_t descLink1[LIST1_SIZE];

/**************************************************************************//**
 * @brief Setup CRYOTIMER as PRS source to trigger ADC
 *****************************************************************************/
void cryoTimerPrsSetup(void)
{
  CRYOTIMER_Init_TypeDef cryotimerInit = CRYOTIMER_INIT_DEFAULT;

  /* Enable LFRCO for CRYOTIMER */
  CMU_OscillatorEnable(cmuOsc_LFRCO, true, true);
  
  /* Trigger period = 32 x 1/32768 = 1024 Hz */
  CMU_ClockEnable(cmuClock_CRYOTIMER, true);
  cryotimerInit.enable = false;
  cryotimerInit.period = cryotimerPeriod_32;
  CRYOTIMER_Init(&cryotimerInit);
  
  /* Output to PRS channel */
  CMU_ClockEnable(cmuClock_PRS, true);
  PRS_SourceSignalSet(ADC_PRS_CHANNEL, PRS_CH_CTRL_SOURCESEL_CRYOTIMER, PRS_CH_CTRL_SIGSEL_CRYOTIMERPERIOD, prsEdgePos);
}

/**************************************************************************//**
 * @brief Initialize ADC for single conversion
 *****************************************************************************/
void adcSingleSetup(void)
{
  ADC_Init_TypeDef       init       = ADC_INIT_DEFAULT;
  ADC_InitSingle_TypeDef singleInit = ADC_INITSINGLE_DEFAULT;

  CMU_ClockEnable(cmuClock_ADC0, true);

  /* Initialize common settings for single conversion */
  init.timebase = ADC_TimebaseCalc(0);
  init.prescale = ADC_PrescaleCalc(ADC_CLOCK, 0);
  ADC_Init(ADC0, &init);

  /* Initialize for single conversion */
  singleInit.reference = adcRefVDD;
  singleInit.prsEnable = true;
  singleInit.posSel = ADC_INPUT_PORT;
  singleInit.negSel = adcNegSelVSS;
  ADC_InitSingle(ADC0, &singleInit);
  
  /* Set data valid level to trigger DMA */
  ADC0->SINGLECTRLX |= (ADC_FIFO_DEPTH - 1) << _ADC_SINGLECTRLX_DVL_SHIFT;
}

/**************************************************************************//**
 * @brief Setup push button BTN1 as PRS source for DMAREQ0.
 *****************************************************************************/
static void gpioPrsSetup(void)
{
  /* Enable GPIO clock */
  CMU_ClockEnable(cmuClock_GPIO, true);

  /* Configure push button BTN1 as input */
  GPIO_PinModeSet(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, gpioModeInputPullFilter, 1);
  /* Configure interrupt on push button PB1 for rising edge but not enabled - PRS sensing instead */
  GPIO_IntConfig(BSP_GPIO_PB1_PORT, BSP_GPIO_PB1_PIN, true, false, false);
 
  /* Select GPIO as PRS source and push button BTN1 as signal for PRS channel */
  CMU_ClockEnable(cmuClock_PRS, true);
  PRS_SourceSignalSet(GPIO_PRS_CHANNEL, PRS_CH_CTRL_SOURCESEL_GPIOL, PRS_CH_CTRL_SIGSEL_GPIOPIN7, prsEdgePos);
  /* Select PRS channel for DMA request 0 */
  PRS->DMAREQ0 = PRS_DMAREQ0_PRSSEL_PRSCH1;
}

/***************************************************************************//**
 * @brief
 *   LDMA IRQ handler.
 ******************************************************************************/
void LDMA_IRQHandler( void )
{
  uint32_t pending;

  /* Read and clear interrupt source */
  pending = LDMA->IFC;

  /* Check for LDMA error */
  if ( pending & LDMA_IF_ERROR )
  {
    /* Loop here to enable the debugger to see what has happened */
    while (1)
      ;
  }
  
  /* For Ping-Pong transfer */
  ppCount++;
}

/***************************************************************************//**
 * @brief
 *   Initialize the LDMA controller.
 ******************************************************************************/
void ldmaSetup(void)
{
  LDMA_Init_t init = LDMA_INIT_DEFAULT;
  LDMA_Init( &init );
}

/***************************************************************************//**
 * @brief
 *   Single direct register DMA transfer example.
 ******************************************************************************/
void ldmaSingleDirectReg(void)
{
  uint32_t i;
  
  /* Initialize buffers for memory transfer */
  for (i=0; i<BUFFER_SIZE; i++)
  {
    srcBuffer[i] = i;
    dstBuffer[i] = 0;
  }
  
  /* Print source and destination buffers before LDMA transfer */
  printf("Source buffer"); 
  for (i=0; i<BUFFER_SIZE; i++)
  {
    if ((i%16) == 0)
    {
      printf("\n%4lu:", i);
    }
    printf("%4d", srcBuffer[i]);
  }
  
  printf("\n\nDestination buffer before LDMA transfer"); 
  for (i=0; i<BUFFER_SIZE; i++)
  {
    if ((i%16) == 0)
    {
      printf("\n%4lu:", i);
    }
    printf("%4d", dstBuffer[i]);
  }
  printf("\n\nPress and release pushbutton BTN1 to start memory transfer"); 

  /* Writes directly to the LDMA channel registers */
  LDMA->CH[USE_DMA_CHANNEL].CTRL = LDMA_CH_CTRL_SIZE_HALFWORD
                                    + LDMA_CH_CTRL_REQMODE_ALL
                                    + LDMA_CH_CTRL_BLOCKSIZE_UNIT4
                                    + (TRANSFER_SIZE << _LDMA_CH_CTRL_XFERCNT_SHIFT);
  LDMA->CH[USE_DMA_CHANNEL].SRC = (uint32_t)&srcBuffer;
  LDMA->CH[USE_DMA_CHANNEL].DST = (uint32_t)&dstBuffer;
  
  /* Enable interrupt and wait PRS on DMAREQ0 to start transfer */
  LDMA->CH[USE_DMA_CHANNEL].REQSEL = ldmaPeripheralSignal_PRS_REQ0;
  LDMA->IFC = USE_DMA_CH_MASK;
  LDMA->IEN = USE_DMA_CH_MASK;
  LDMA->CHEN = USE_DMA_CH_MASK;

  /* Wait LDMA finish */
  while (!LDMA_TransferDone(USE_DMA_CHANNEL))
  {
    EMU_EnterEM1();
  }
  
  /* Print dstination buffer after LDMA transfer */
  printf("\nDestination buffer after LDMA transfer"); 
  for (i=0; i<BUFFER_SIZE; i++)
  {
    if ((i%16) == 0)
    {
      printf("\n%4lu:", i);
    }
    printf("%4d", dstBuffer[i]);
  }
  printf("\n"); 
}

/***************************************************************************//**
 * @brief
 *   Descriptor linked list example.
 ******************************************************************************/
void ldmaDescLinkList(void)
{
  /* Use peripheral transfer configuration macro */
  LDMA_TransferCfg_t periTransferTx =
        LDMA_TRANSFER_CFG_PERIPHERAL(ldmaPeripheralSignal_USART0_TXBL);
  
  /* LINK descriptor macros to form linked list, SINGLE descriptor macro for the last one */
  LDMA_Descriptor_t xfer[] =
  {
    LDMA_DESCRIPTOR_LINKREL_M2P_BYTE( &str1, &USART0->TXDATA, sizeof(str1)-1, 1 ),
    /* To remove the 3rd descriptor, change the last parameter of 2nd descriptor from 1 to 2 */
    LDMA_DESCRIPTOR_LINKREL_M2P_BYTE( &str2, &USART0->TXDATA, sizeof(str2)-1, 1 ),
    LDMA_DESCRIPTOR_LINKREL_M2P_BYTE( &str3, &USART0->TXDATA, sizeof(str3)-1, 1 ),
    LDMA_DESCRIPTOR_SINGLE_M2P_BYTE( &str4, &USART0->TXDATA, sizeof(str4)-1 )
  };
  
  descLink0[0] = xfer[0];
  descLink0[1] = xfer[1];
  descLink0[2] = xfer[2];
  descLink0[3] = xfer[3];

  /* No interrupt during transferring decsriptor 1 and 3 */
  descLink0[0].xfer.doneIfs = 0;
  descLink0[2].xfer.doneIfs = 0;
  LDMA_StartTransfer(USE_DMA_CHANNEL, (void*)&periTransferTx, (void*)&descLink0);

  /* Wait LDMA finish to return */
  while (!LDMA_TransferDone(USE_DMA_CHANNEL))
  {
    EMU_EnterEM1();
  }
}
  
/***************************************************************************//**
 * @brief
 *   Single Descriptor Looped Transfer.
 ******************************************************************************/
void ldmaSingleDescLoop(void)
{
  uint32_t i;

  /* Fill buffer with dummy value */
  for (i=0; i<ADC_BUFFER_SIZE; i++)
  {
    adcBuffer[i] = ADC_DUMMY_VALUE;
  }

  /* Use looped peripheral transfer configuration macro */
  LDMA_TransferCfg_t periTransferTx =
        LDMA_TRANSFER_CFG_PERIPHERAL_LOOP(ldmaPeripheralSignal_ADC0_SINGLE, LOOP_REPEAT);

  /* Use LINK descriptor macro for initialization and looping */
  LDMA_Descriptor_t xfer[] =
  {
    LDMA_DESCRIPTOR_LINKREL_P2M_BYTE(&ADC0->SINGLEDATA, &adcBuffer, ADC_FIFO_DEPTH, 1),
    LDMA_DESCRIPTOR_LINKREL_P2M_BYTE(&ADC0->SINGLEDATA, ADC_DST_ADDR_OFFSET, ADC_FIFO_DEPTH, 0)
  };

  descLink0[0] = xfer[0];
  descLink0[1] = xfer[1];

  /* First descriptor to initialize transfer */
  descLink0[0].xfer.doneIfs = 0;
  descLink0[0].xfer.blockSize = ldmaCtrlBlockSizeUnit4;
  descLink0[0].xfer.ignoreSrec = 1;
  descLink0[0].xfer.reqMode = ldmaCtrlReqModeAll;
  descLink0[0].xfer.size = ldmaCtrlSizeWord; 
  
  /* Second descriptor for single descriptor looping */
  descLink0[1].xfer.decLoopCnt = 1;
  descLink0[1].xfer.doneIfs = 0;
  descLink0[1].xfer.blockSize = ldmaCtrlBlockSizeUnit4;
  descLink0[1].xfer.ignoreSrec = 1;
  descLink0[1].xfer.reqMode = ldmaCtrlReqModeAll;
  descLink0[1].xfer.size = ldmaCtrlSizeWord; 

  /* Use relative addressing to keep destination address, stop after looping */
  descLink0[1].xfer.dstAddrMode = ldmaCtrlSrcAddrModeRel;
  descLink0[1].xfer.link = 0;
  LDMA_StartTransfer(USE_DMA_CHANNEL, (void*)&periTransferTx, (void*)&descLink0);

  /* Start timer to trigger ADC, wait LDMA finish */
  CRYOTIMER_Enable(true);
  while (!LDMA_TransferDone(USE_DMA_CHANNEL))
  {
    EMU_EnterEM1();
  }

  /* Stop timer and print ADC buffer */   
  CRYOTIMER_Enable(false);
  for (i=0; i<ADC_BUFFER_SIZE; i++)
  {
    printf ("ADC[%lu]: %4fV\n", i, ((float)adcBuffer[i] * (float)3.3)/4096);
  }
}

/***************************************************************************//**
 * @brief
 *   Descriptor List with Looping example.
 ******************************************************************************/
void ldmaDescListLoop(void)
{
  /* Use looped peripheral transfer configuration macro */
  LDMA_TransferCfg_t periTransferTx =
        LDMA_TRANSFER_CFG_PERIPHERAL_LOOP(ldmaPeripheralSignal_USART0_TXBL, LOOP_REPEAT);

  /* LINK descriptor macros for looping, SINGLE descriptor macro for single transfer */
  LDMA_Descriptor_t xfer[] =
  {
    LDMA_DESCRIPTOR_LINKREL_M2P_BYTE(&str1, &USART0->TXDATA, sizeof(str1)-1, 1),
    LDMA_DESCRIPTOR_LINKREL_M2P_BYTE(&str2, &USART0->TXDATA, sizeof(str2)-1, -1),
    LDMA_DESCRIPTOR_SINGLE_M2P_BYTE(&str3, &USART0->TXDATA, sizeof(str3)-1)
  };

  descLink0[0] = xfer[0];
  descLink0[1] = xfer[1];
  descLink0[2] = xfer[2];

  /* Enable looping and no interrupt during looping */
  descLink0[0].xfer.doneIfs = 0;
  descLink0[1].xfer.doneIfs = 0;
  descLink0[1].xfer.decLoopCnt = 1;
  LDMA_StartTransfer(USE_DMA_CHANNEL, (void*)&periTransferTx, (void*)&descLink0);

  /* Wait LDMA finish to return */
  while (!LDMA_TransferDone(USE_DMA_CHANNEL))
  {
    EMU_EnterEM1();
  }
}

/***************************************************************************//**
 * @brief
 *   Simple Inter-Channel Synchronization example.
 ******************************************************************************/
void ldmaSimpleInterSync(void)
{
  /* Use peripheral transfer configuration macro for first DMA channel */
  LDMA_TransferCfg_t periTransferTx =
        LDMA_TRANSFER_CFG_PERIPHERAL(ldmaPeripheralSignal_USART0_TXBL);

  /* Use memory to peripheral descriptor macro, wait trigger after sending data */
  LDMA_Descriptor_t xferTx[] =
  {
    LDMA_DESCRIPTOR_LINKREL_M2P_BYTE(&ledon, &USART0->TXDATA, sizeof(ledon)-1, 1),
    /* Wait to match the SYNCTRIG[n] to continue */
    LDMA_DESCRIPTOR_LINKREL_SYNC(0, SYNC_CLEAR, MATCH_VALUE, MATCH_VALUE, 1),
    /* Use WRI descriptor to toggle LED0 */
    LDMA_DESCRIPTOR_SINGLE_WRITE(WRI_DATA, WRI_ADDR), 
  };

  descLink0[0] = xferTx[0];
  descLink0[1] = xferTx[1];
  descLink0[2] = xferTx[2];
  
  /* No interrupt during transfer and wait trigger */
  descLink0[0].xfer.doneIfs = 0;
  
  /* Use peripheral transfer configuration macro for second DMA channel */
  LDMA_TransferCfg_t periTransferRx =
        LDMA_TRANSFER_CFG_PERIPHERAL(ldmaPeripheralSignal_USART0_RXDATAV);

  /* Use peripheral to memory descriptor macro, send trigger after receiving data */
  LDMA_Descriptor_t xferRx[] =
  {
    LDMA_DESCRIPTOR_LINKREL_P2M_BYTE(&USART0->RXDATA, &rxSyncBuffer, RX_TRIG_LENGTH, 1),
    /* Set SYNCTRIG[n] to trigger */
    LDMA_DESCRIPTOR_SINGLE_SYNC(SYNC_SET, 0, 0, 0)
  };

  descLink1[0] = xferRx[0];
  descLink1[1] = xferRx[1];
  
  /* Start both channels */
  RETARGET_UART->CMD = USART_CMD_CLEARRX;
  LDMA_StartTransfer(USE_DMA_CHANNEL, (void*)&periTransferTx, (void*)&descLink0);
  LDMA_StartTransfer(USE_DMA_CHANNEL+1, (void*)&periTransferRx, (void*)&descLink1);
  
  /* No interrupt for DMA channel used for trigger */
  LDMA_IntDisable((USE_DMA_CHANNEL+1) << 1);
  
  /* Wait LDMA finish to return */
  while (!LDMA_TransferDone(USE_DMA_CHANNEL))
  {
    EMU_EnterEM1();
  }
  printf("Trigger received\n"); 
  RETARGET_UART->CMD = USART_CMD_CLEARRX;
}

/***************************************************************************//**
 * @brief
 *    The 2D Copy example.
 ******************************************************************************/
void ldma2Dcopy(void)
{
  uint32_t x, y;
  
  /* Initialize buffers for 2D copy */
  for (x=0; x<BUFFER_2D_WIDTH; x++)
  {
    for(y=0; y<BUFFER_2D_WIDTH; y++)
    {
      src2d[x][y] = y;
      dst2d[x][y] = 0;
    }
  }

  /* Print source and destination buffers before 2D copy */
  printf("2D Rectangle Source buffer"); 
  for (x=SRC_ROW_INDEX; x<(SRC_ROW_INDEX + TRANSFER_WIDTH); x++)
  {
    for(y=SRC_COL_INDEX; y<(SRC_COL_INDEX + TRANSFER_HEIGHT); y++)
    {
      if ((y%16) == 0)
      {
        printf("\n[%2lu][%2lu]:", x, y);
      }
      printf("%4d", src2d[x][y]);
    }
  }
  
  printf("\n\n2D Rectangle Destination buffer before LDMA transfer"); 
  for (x=DST_ROW_INDEX; x<(DST_ROW_INDEX + TRANSFER_WIDTH); x++)
  {
    for(y=DST_COL_INDEX; y<(DST_COL_INDEX + TRANSFER_HEIGHT); y++)
    {
      if ((y%16) == 0)
      {
        printf("\n[%2lu][%2lu]:", x, y);
      }
      printf("%4d", dst2d[x][y]);
    }
  }
  
  /* Use looped memory transfer configuration macro */
  LDMA_TransferCfg_t memTransfer = LDMA_TRANSFER_CFG_MEMORY_LOOP(TRANSFER_HEIGHT-2);

  /* First descriptor to initialize transfer, second descriptor macro for looping */
  LDMA_Descriptor_t xfer[] =
  {
    LDMA_DESCRIPTOR_LINKREL_M2M_BYTE(&src2d[SRC_ROW_INDEX][SRC_COL_INDEX], &dst2d[DST_ROW_INDEX][DST_COL_INDEX], TRANSFER_WIDTH, 1),
    LDMA_DESCRIPTOR_LINKREL_M2M_BYTE(BUFFER_2D_WIDTH-TRANSFER_WIDTH, BUFFER_2D_WIDTH-TRANSFER_WIDTH, TRANSFER_WIDTH, 0)
  };
  
  descLink0[0] = xfer[0];
  descLink0[1] = xfer[1];

  /* Use relative addressing for source & destination, enable looping */
  descLink0[1].xfer.srcAddrMode = ldmaCtrlSrcAddrModeRel;
  descLink0[1].xfer.dstAddrMode = ldmaCtrlSrcAddrModeRel;
  descLink0[1].xfer.decLoopCnt = 1;

  /* Stop after looping */
  descLink0[1].xfer.link = 0;
  LDMA_StartTransfer(USE_DMA_CHANNEL, (void*)&memTransfer, (void*)&descLink0);

  /* Wait LDMA finish */
  while (!LDMA_TransferDone(USE_DMA_CHANNEL))
  {
    EMU_EnterEM1();
  }

  /* Print destination buffers after 2D copy */
  printf("\n\n2D Rectangle Destination buffer after LDMA transfer"); 
  for (x=DST_ROW_INDEX; x<(DST_ROW_INDEX + TRANSFER_WIDTH); x++)
  {
    for(y=DST_COL_INDEX; y<(DST_COL_INDEX + TRANSFER_HEIGHT); y++)
    {
      if ((y%16) == 0)
      {
        printf("\n[%2lu][%2lu]:", x, y);
      }
      printf("%4d", dst2d[x][y]);
    }
  }
  printf("\n"); 
}

/***************************************************************************//**
 * @brief
 *   Ping-Pong example.
 ******************************************************************************/
void ldmaPingPong(void)
{
  uint32_t i;

  /* Initialize Ping-Pong buffers */
  for (i=0; i<PP_BUFFER_SIZE-1; i++)
  {
    pingBuffer[i] = '1';
    pongBuffer[i] = '2';
  }
  pingBuffer[PP_BUFFER_SIZE-1] = '\n';
  pongBuffer[PP_BUFFER_SIZE-1] = '\n';

  /* Use peripheral transfer configuration macro */
  LDMA_TransferCfg_t periTransferTx =
        LDMA_TRANSFER_CFG_PERIPHERAL(ldmaPeripheralSignal_USART0_TXBL);

  /* LINK descriptor macros for Ping-Pong transfer */
  LDMA_Descriptor_t xfer[] =
  {
    LDMA_DESCRIPTOR_LINKREL_M2P_BYTE(&pingBuffer, &USART0->TXDATA, PP_BUFFER_SIZE, 1),
    LDMA_DESCRIPTOR_LINKREL_M2P_BYTE(&pongBuffer, &USART0->TXDATA, PP_BUFFER_SIZE, -1),
  };

  descLink0[0] = xfer[0];
  descLink0[1] = xfer[1];

  /* Start Ping-Pong transfer */
  ppCount = 0;
  LDMA_StartTransfer(USE_DMA_CHANNEL, (void*)&periTransferTx, (void*)&descLink0);

  /* Fill alternate buffer until end of Ping-Pong transfer */
  while (!LDMA_TransferDone(USE_DMA_CHANNEL))
  {
    EMU_EnterEM1();
    if (ppCount & 0x01) 
    {
      for (i=0; i<PP_BUFFER_SIZE-1; i++)
      {
        pingBuffer[i] += 2;
      }
    }
    else
    {
      for (i=0; i<PP_BUFFER_SIZE-1; i++)
      {
        pongBuffer[i] += 2;
      }
    }
    
    /* End of Ping-Pong? */
    if (ppCount == PP_LOOP_COUNT)
    {
      /* Stop Ping-Pong transfer */
      descLink0[0].xfer.link = 0;
      descLink0[1].xfer.link = 0;
    }    
  }
}

/***************************************************************************//**
 * @brief
 *   Print LDMA Demo Menu.
 ******************************************************************************/
void printLdmaMenu(void)
{
  printf("\nLDMA Examples\n");
  printf("Press 1 for Single Direct Register DMA Transfer Example\n"); 
  printf("Press 2 for Descriptor Linked List Example\n"); 
  printf("Press 3 for Single Descriptor Looped Transfer Example\n"); 
  printf("Press 4 for Descriptor List with Looping Example\n"); 
  printf("Press 5 for Simple Inter-Channel Synchronization Example\n"); 
  printf("Press 6 for 2D Copy Example\n"); 
  printf("Press 7 for Ping-Pong Example\n"); 
  printf("Press ? to print this menu\n");
}

/**************************************************************************//**
 * @brief  Main function
 *****************************************************************************/
int main(void)
{
  int8_t inputChar;

  /* Chip errata */
  CHIP_Init();

  /* Init DCDC regulator if available */
#if defined( _EMU_DCDCCTRL_MASK )  
  EMU_DCDCInit_TypeDef dcdcInit = EMU_DCDCINIT_DEFAULT;
  EMU_DCDCInit(&dcdcInit);
#endif
  
  /* Initialize HFXO */
  CMU_HFXOInit_TypeDef hfxoInit = CMU_HFXOINIT_DEFAULT;
  CMU_HFXOInit(&hfxoInit);

  /* Switch HFCLK to HFXO and disable HFRCO */
  CMU_ClockSelectSet(cmuClock_HF, cmuSelect_HFXO);
  CMU_OscillatorEnable(cmuOsc_HFRCO, false, false);

  /* Enable atomic read-clear operation on reading IFC register */
  MSC->CTRL |= MSC_CTRL_IFCREADCLEAR;
  
  /* Initialize and retarget USART. */
  RETARGET_SerialInit();
  RETARGET_SerialCrLf(1);

  /* Disable retarget RX interrupt */
  RETARGET_UART->IEN = 0;
  
  /* Initialize GPIO for PRS */
  gpioPrsSetup();

  /* Initialize CRYOTIMER for PRS */
  cryoTimerPrsSetup();

  /* Initialize ADC */
  adcSingleSetup();
  
  /* Initialize LDMA */
  ldmaSetup();
  
  /* Initialize LED driver and print menu */
  BSP_LedsInit();
  printLdmaMenu();

  while (1)
  {
    /* Wait key press */
    inputChar = RETARGET_RX(RETARGET_UART);

    switch (inputChar)
    {
    case '1':
      printf("\nSingle Direct Register DMA Transfer Example\n"); 
      ldmaSingleDirectReg();
      break;

    case '2':
      printf("\nDescriptor Linked List Example\n"); 
      ldmaDescLinkList();  
      break;
      
    case '3':
      printf("\nSingle Descriptor Looped Transfer Example (Press pushbutton BTN0 to ground the ADC input)\n"); 
      ldmaSingleDescLoop();
      break;
      
    case '4':
      printf("\nDescriptor List with Looping Example\n"); 
      ldmaDescListLoop();
      break;

    case '5':
      printf("\nSimple Inter-Channel Synchronization Example\n"); 
      ldmaSimpleInterSync();
      break;

    case '6':
      printf("\n2D Copy Example\n"); 
      ldma2Dcopy();
      break;

    case '7':
      printf("\nPing-Pong Example\n"); 
      ldmaPingPong();
      break;

    case '?':
      printLdmaMenu();
      break;

    default:
      break;
    }
  }
}
