//************************************************************************************
// JTAG_Chain_F02x.c
//------------------------------------------------------------------------------------
// This program contains some primitive routines which gather information through the 
// JTAG port on multiple JTAG compatible devices under test (DUT) connected in a 
// chain. The TCK & TMS JTAG pins on the DUT are connected in parallel to port pins on
// the C8051F02x master device and the TDI & TDO pins are connected in 
// series.
//
// **NOTE: The first device in the chain (device 0) is the one whose TDO pin is
//         connected to the TDO pin of the master device.
//
// Target device: C8051F02x
//
// Tool chain: KEIL Eval 'c'
//************************************************************************************

//------------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------------

#include <c8051f020.h>                    // SFR declarations

//------------------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F02x
//------------------------------------------------------------------------------------

sfr16 DP       = 0x82;                    // data pointer
sfr16 TMR3RL   = 0x92;                    // Timer3 reload value
sfr16 TMR3     = 0x94;                    // Timer3 counter
sfr16 ADC0     = 0xbe;                    // ADC0 data
sfr16 ADC0GT   = 0xc4;                    // ADC0 greater than window
sfr16 ADC0LT   = 0xc6;                    // ADC0 less than window
sfr16 RCAP2    = 0xca;                    // Timer2 capture/reload
sfr16 T2       = 0xcc;                    // Timer2
sfr16 RCAP4    = 0xe4;                    // Timer4 capture/reload
sfr16 T4       = 0xf4;                    // Timer4
sfr16 DAC0     = 0xd2;                    // DAC0 data
sfr16 DAC1     = 0xd5;                    // DAC1 data

//------------------------------------------------------------------------------------
// Global CONSTANTS
//------------------------------------------------------------------------------------

#define MAX_NUM_DEVICES_IN_CHAIN 10

sbit  LED = P1^6;                         // green LED: '1' = ON; '0' = OFF
sbit  SW2 = P3^7;                         // SW2='0' means switch pressed

#define SYSCLK       22118400             // SYSCLK frequency in Hz

sbit   TCK = P3^7;      // JTAG Test Clock -- Connected to TCK pin on all devices.
sbit   TMS = P3^6;      // JTAG Mode Select -- Connected to TMS pin on all devices.
sbit   TDI = P3^5;      // JTAG Data Input(output of master) -- Connected to the 
                        //      TDI pin of device n.
sbit   TDO = P3^4;      // JTAG Data Output (input to master)-- Connected to the 
                        //      TDO pin of device 0.

#define   TRUE 1
#define   FALSE 0

// JTAG Instruction Register Addresses
#define   INST_LENGTH 16                   // number of bits in the C8051Fxxx 
#define   BYPASS      0xffff               // Instruction Register
#define   EXTEST      0x0000
#define   SAMPLE      0x0002

#define   RESET       0x2fff               // System RESET Instruction

#define   IDCODE      0x1004               // IDCODE Instruction address/HALT
#define   IDCODE_LEN  32                   // number of bits in the ID code

#define   FLASHCON    0x4082               // FLASH Control Instruction address
#define   FLCN_LEN    8                    // number of bits in FLASHCON

#define   FLASHDAT    0x4083               // FLASH Data Instruction address
#define   FLD_RDLEN   10                   // number of bits in an FLASHDAT read
#define   FLD_WRLEN   8                    // number of bits in an FLASHDAT write

#define   FLASHADR    0x4084               // FLASH Address Instruction address
#define   FLA_LEN     16                   // number of bits in FLASHADR

#define   FLASHSCL    0x4085               // FLASH Scale Instruction address
#define   FLSC_LEN    8                    // number of bits in FLASHSCL

//------------------------------------------------------------------------------------
// Global Variable DECLARATIONS
//------------------------------------------------------------------------------------

// The addresses of the following variables are explicitly defined for viewing 
// purposes. If the width of the external memory window is 5 bytes, then each 
// device will take up exactly one row starting from the second row.
char xdata num_devices  _at_ 0x0000;

char xdata num_devices_before _at_ 0x0001; // #devices before and after the isolated 
char xdata num_devices_after  _at_ 0x0002; // device                
char xdata num_IR_bits_before _at_ 0x0003; // #instruction register bits before and
char xdata num_IR_bits_after  _at_ 0x0004; // after the isolated device 


typedef struct JTAG_Information {          // Discovery information
      unsigned char IR_length;             // Instruction register length
      unsigned long id;                    // Identification code for each device
} JTAG_Information;
                                           // Array: one entry per device in the
                                           // JTAG chain
JTAG_Information xdata JTAG_info[MAX_NUM_DEVICES_IN_CHAIN]; 
                                           
//------------------------------------------------------------------------------------
// Function PROTOTYPES
//------------------------------------------------------------------------------------

void SYSCLK_Init (void);
void PORT_Init (void);

void JTAG_StrobeTCK (void);
void JTAG_Reset (void);

void Blink_Led(void);

void init(void);
void JTAG_Discover(void);
void JTAG_Discover_IR(void);
void JTAG_Discover_DR(void);
void JTAG_Isolate(char index);

unsigned long JTAG_IR_Scan (unsigned long instruction, char num_bits) ;
unsigned long JTAG_DR_Scan (unsigned long dat, char num_bits);

void JTAG_IWrite (unsigned int ireg, unsigned long dat, int num_bits);
unsigned long JTAG_IRead (unsigned int ireg, int num_bits);
int FLASH_ByteRead (unsigned int addr, unsigned char *pdat);
int FLASH_ByteWrite (unsigned int addr, unsigned char dat);
int FLASH_PageErase (unsigned int addr);

//------------------------------------------------------------------------------------
// MAIN Routine
//------------------------------------------------------------------------------------
void main (void)
{
   
   long xdata id;
   unsigned char dest;
   int pass;
   int address;
   char device = 0;
   

   WDTCN = 0xde;                            // disable watchdog timer
   WDTCN = 0xad;
   
   PORT_Init ();                            // initialize crossbar and GPIO
   SYSCLK_Init ();                          // initialize oscillator

   LED = 1;                                 // turn on the LED         
   
   init();                                  // initialize JTAG Chain variables
   JTAG_Discover();                         // IDCODE should = 0x10000243 for
                                            // C8051F000 rev D device
   
   JTAG_Isolate(0);                         // isolate device 0
   JTAG_IR_Scan (IDCODE, INST_LENGTH);      // load IDCODE into IR and HALT the DUT
   id = JTAG_DR_Scan (0x0L, IDCODE_LEN);    // get the ID Code of the isolated device
   
   // comment out this code if you have less than two devices in the chain
   JTAG_Isolate(1);
   JTAG_IR_Scan (IDCODE, INST_LENGTH);      // load IDCODE into IR and HALT the DUT
   id = JTAG_DR_Scan (0x0L, IDCODE_LEN);    // get the ID Code of the isolated device
   
   // comment out this code if you have less than three devices in the chain
   JTAG_Isolate(2);
   JTAG_IR_Scan (IDCODE, INST_LENGTH);      // load IDCODE into IR and HALT the DUT
   id = JTAG_DR_Scan (0x0L, IDCODE_LEN);    // get the ID Code of the isolated device
   
   

   for(device = 0; device < num_devices; device++) {
       
      JTAG_Isolate(device);     

      //TEST 1 -- ERASE A FLASH PAGE      
      pass = FLASH_PageErase (0x1000);       // erase page prior to writing
      while (!pass);                         // handle Write Lock condition

           
      //Verify that locations 0x1000 - 0x11FF are 0xFF
      for(address = 0x1000; address < 0x1200; address++){ 
         pass = FLASH_ByteRead (address, &dest);   // dest should return 0xff
         if(!pass || dest != 0xFF) Blink_Led();                  
      }
      
      //TEST 2 -- WRITE A PATTERN TO FLASH PAGE
      
      for(address = 0x1000; address < 0x1200; address++){ 
         dest = address & 0x00FF;               // strip away upper 8 bits
         pass = FLASH_ByteWrite (address, dest);// store LSByte of address at address
         while (!pass);                         // handle Read Lock condition
      }
      
      dest = 0x12;                              // set test variable to non-0xff value
      
      //Verify that locations 0x1000 - 0x11FF are following the pattern
      for(address = 0x1000; address < 0x1200; address++){ 
         pass = FLASH_ByteRead (address, &dest);   
         if(!pass || dest != (address & 0x00FF)) Blink_Led();                  
      }
   }
     
      
   LED = 0;                                     // turn off the led,
                                                // program executed correctly
   while(1);
}
   
//************************************************************************************
// Function and Procedure DEFINITIONS
//************************************************************************************
//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use an 22.1184MHz crystal
// as its clock source.
//
void SYSCLK_Init (void)
{
   int i;                              // delay counter

   OSCXCN = 0x67;                      // start external oscillator with
                                       // 22.1184MHz crystal

   for (i=0; i < 256; i++) ;           // XTLVLD blanking interval (>1ms)

   while (!(OSCXCN & 0x80)) ;          // Wait for crystal osc. to settle

   OSCICN = 0x88;                      // select external oscillator as SYSCLK
                                       // source and enable missing clock
                                       // detector
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports
//
void PORT_Init (void)
{
   XBR0    = 0x04;                     // Enable UART0
   XBR1    = 0x00;
   XBR2    = 0x40;                     // Enable crossbar and weak pull-ups
   P0MDOUT |= 0x01;                    // enable TX0 as a push-pull output
   P1MDOUT |= 0x40;                    // enable P1.6 (LED) as push-pull output
   
   P3MDOUT |= 0xe0;                    // make P3.7-5 push-pull outputs
   P3 &= ~0xe0;                        // TCK, TMS, and TDI all low
}
//------------------------------------------------------------------------------------
// Blink_Led
//------------------------------------------------------------------------------------
// This routine blinks the Green LED forever to indicate an error.
//
void Blink_Led(void)
{
   int i;                                    // millisecond counter
   int ms = 200;                             // stay in each state for ms milliseconds
  
   TCON  &= ~0x30;                           // STOP Timer0 and clear overflow flag
   TMOD  &= ~0x0F;                           // configure Timer0 to 16-bit mode
   TMOD  |=  0x01;                            
   CKCON |=  0x08;                           // Timer0 counts SYSCLKs
   
   
   while (1){
   
   LED = ~LED;
      
      for (i = 0; i < ms; i++) {             // count milliseconds
         TR0 = 0;                            // STOP Timer0
         TH0 = (-SYSCLK/1000) >> 8;          // SET Timer0 to overflow in 1ms
         TL0 = -SYSCLK/1000;
         TR0 = 1;                            // START Timer0

         while(TF0 == 0);                    // wait for overflow 
   
         TF0 = 0;                            // clear overflow indicator      
      }
   }   
}

//------------------------------------------------------------------------------------
// init
//------------------------------------------------------------------------------------
// This routine initializes the variables used in a JTAG chain.
//
void init (void)
{

   num_devices = 1;                         // The default number of devices is one.
                                            // JTAG_Discover() does not have to be
                                            // called if only one device is connected.
   
   num_devices_before = 0;                  // Initializing these variables to zero
   num_devices_after = 0;                   // allows calling the JTAG_IR_Scan() and 
   num_IR_bits_before = 0;                  // the JTAG_DR_Scan() without first 
   num_IR_bits_after = 0;                   // calling JTAG_Isolate() when there is  
                                            // only one device in the chain.
}

//------------------------------------------------------------------------------------
// JTAG_StrobeTCK
//------------------------------------------------------------------------------------
// This routine strobes the TCK pin (brings high then back low again) 
// on the target system.
//
void JTAG_StrobeTCK (void)
{

   TCK = 1;
   TCK = 0;
}

//------------------------------------------------------------------------------------
// JTAG_Reset
//------------------------------------------------------------------------------------
// This routine places the JTAG state machine on the target system in
// the Test Logic Reset state by strobing TCK 5 times while leaving
// TMS high.  Leaves the JTAG state machine in the Run_Test/Idle state.
//
void JTAG_Reset (void)
{
   
   TMS = 1;

   JTAG_StrobeTCK ();                       // move to Test Logic Reset state
   JTAG_StrobeTCK ();
   JTAG_StrobeTCK ();
   JTAG_StrobeTCK ();
   JTAG_StrobeTCK ();

   TMS = 0;

   JTAG_StrobeTCK ();                       // move to Run_Test/Idle state
}

//------------------------------------------------------------------------------------
// JTAG_Discover
//------------------------------------------------------------------------------------
// This routine sequentially queries a chain of JTAG devices and accomplishes the 
// following three tasks.
//    For the global struct array <JTAG_info> 
//      -- fills in the length of each device's instruction register
//      -- fills in each device's IDCODE. 
//    For the global variable <num_devices> 
//      -- updates it with the number of JTAG devices connected in the chain.
//
void JTAG_Discover(void)
{
   
   JTAG_Discover_IR();                 

   // At this point we know num_devices(a global variable) and we know the 
   // length of each device's IR given in the variable JTAG_info[].IR_length
   
   JTAG_Discover_DR();                       // Read and assign the ID for each
                                             // device

} //end discover

//------------------------------------------------------------------------------------
// JTAG_Discover_IR
//------------------------------------------------------------------------------------
// This routine fills a structure with the length of each device's instruction 
// register. It also updates the global variable <num_devices> with the number of
// JTAG devices connected in the chain.
//
// BACKGROUND: When an IRSCAN is issued, a JTAG device must return a 1 as the LSB
//             and zeros in all the other bits.  We shift in all ones so when we
//             encounter two ones in a row, we know we are past the end of the chain.
//             A state machine is implemented in this routine to keep track of 
//             inputs received.
//
// STATE DEFINITONS: 
//             0 - NO INPUTS -- at beginning of chain                             
//             1 - INPUT SEQUENCE: 1 -- could be at a new device or at chain end                           
//             2 - INPUT SEQUENCE: 100..0 -- counting zeros    
//             
//
void JTAG_Discover_IR(void)
{
   
   char state = 0;                          // beginning of chain

   char num_zeros = 0;                      // number of zeros following a one in 
                                            // an IR_SCAN. num_zeros + 1 = IR_length 

   char current_device_index = -1;          // current_device_index + 1 = num_devices
                                            // (on the last iteration)
   bit done = FALSE;                        // TRUE when end of chain is reached
   
   JTAG_Reset();                            // RESET and move to Run_Test/Idle
       
   // advance to Shift_IR State
   TMS = 1;                     
   JTAG_StrobeTCK ();                       // move to SelectDR
   TMS = 1;
   JTAG_StrobeTCK ();                       // move to SelectIR
   TMS = 0;
   JTAG_StrobeTCK ();                       // move to Capture_IR
   TMS = 0;
   JTAG_StrobeTCK ();                       // move to Shift_IR state and get the
                                            // the first input
   
   TDI = 1;                                 // STATE is initially 0
                                            // shift in all ones    
   // for each device 
   do{   
                  
      if(TDO != 1){                         // Error if the first input is not one.
         Blink_Led();                       // Could mean bad connections or 
      }                                     // non-compliant devices.
                                                                             
            
      
      state = 1;                            // received a 1, could be at a new 
                                            // device or at the end of the chain
      
      num_zeros = 0;                        // initialize for the zero counting loop
      
      // for the number of zeros in each device's IR 
      do { 
 
         JTAG_StrobeTCK();                  // get the next bit.
         
         switch(state){
            
            case 1: if(TDO == 0){           // found new device(10)
                       current_device_index++;
                       num_zeros++;
                       state = 2;
                    } else {
                       done = TRUE;         // at end of chain (11)
                    }
                    break;
           
            case 2: if(TDO == 0){         
                      num_zeros++;          // counting zeros (10..0)
                   } else {
                      state = 1;            // past end of current device (10..01)
                   }
                   break;
   
            default: Blink_Led();           // an error has occurred
                                         
         } // end switch

         
      } while ((state != 1) && (!done));    // while the input is not one,
                                            // count zeros until we get a one.
      
      if (!done) {                          // if we are not past the last device
      
         JTAG_info[current_device_index].IR_length = num_zeros + 1;
      }

   } while (!done);                         //while we are not past the last device
   
   num_devices = current_device_index + 1;            
   
   // navigate the JTAG State Machine back to RTI state.
   TMS = 1;
   JTAG_StrobeTCK ();                       // move to Exit1_IR state   
   TMS = 1;
   JTAG_StrobeTCK ();                       // move to Update_IR state
   TMS = 0;
   JTAG_StrobeTCK ();                       // move to Run_Test/Idle state

}

//------------------------------------------------------------------------------------
// JTAG_Discover_DR
//------------------------------------------------------------------------------------
//GOAL: Obtain the ID code of each device(If it supports IDCODE), and fill in
//      the field JTAG_info[].id (32-bit).
//      Assign all zeros if device does not have an IDCODE.
//
//BACKGROUND: After JTAG State Machine Reset, the IDCODE is automatically selected
//            If a device does not have an IDCODE register, the BYPASS 
//            register is selected instead.
//            On a DR_SCAN, each IDCODE register returns a 32-bit ID with LSB = 1
//            and each BYPASS register returns 1-bit = 0.

void JTAG_Discover_DR(void)
{
   
   char current_device_index = 0;  
   
   unsigned char i;                       // loop counter 


   JTAG_Reset ();                         // Reset the JTAG state machine on DUT
                                          // move to Run_Test/Idle
   
   // The IDCODE or the BYPASS Register is automatically selected.

   // Navigate to the Shift_DR state
   TMS = 1;                     
   JTAG_StrobeTCK ();                     // move to SelectDR
   TMS = 0;
   JTAG_StrobeTCK ();                     // move to Capture_DR
   
   TMS = 0;
   TDI = 1;                               // shift in all ones               
   
   current_device_index = 0;

   while (current_device_index < num_devices) {  
      
      JTAG_StrobeTCK ();                  // move to Shift_DR state and get input   
      
      if (TDO == 0) {                     // Device does not have an IDCODE register
         
         JTAG_info[current_device_index].id = 0x00000000L;

      } else { // TDO == 1
         
         JTAG_info[current_device_index].id = 0x80000000L;
            
         for (i = 0; i < 31; i++){        // Get the next 31-bits of the device ID
            
            JTAG_StrobeTCK ();   
            
            JTAG_info[current_device_index].id = 
              JTAG_info[current_device_index].id >> 1;
            
            if (TDO) {
               JTAG_info[current_device_index].id |= 0x80000000L;  
            }
         } // end for

      } // end if-else
      
      current_device_index++;
   
   } // end while

   //fill the rest of the entries with zeros
   for (; current_device_index < MAX_NUM_DEVICES_IN_CHAIN; current_device_index++) {

      JTAG_info[current_device_index].IR_length = 0;
      JTAG_info[current_device_index].id = 0x00000000L;
   }
   
   // Navigate JTAG State Machine back to RTI state
   TMS = 1;
   JTAG_StrobeTCK ();                     // move to Exit1_DR
   TMS = 1;
   JTAG_StrobeTCK ();                     // move to Update DR
   TMS = 0;
   JTAG_StrobeTCK ();                     // move to RTI

}

//------------------------------------------------------------------------------------
// JTAG_Isolate
//------------------------------------------------------------------------------------
// This routine updates 4 global variables.  JTAG_Discover() must be called prior to
// calling this routine in order to set up the data structure.
//
// VARIABLE DEFINITIONS
//    num_IR_bits_before -- number of instruction register bits before the isolated
//                          device            
//    num_IR_bits_after  -- number of instruction register bits after the isolated 
//                          device 
//    num_devices_before -- number of devices before the isolated device 
//    num_devices_after  -- number of device after the isolated device
//
void JTAG_Isolate(char index)
{

   unsigned char i;
                                          
   if ((index > (num_devices - 1)) || (index < 0) ) { 
                                          // check if index is out of range
      Blink_Led();
   }

   num_devices_before = index;

   num_devices_after = num_devices - index - 1;

   num_IR_bits_before = 0;                // initializing for loop
   num_IR_bits_after = 0;
 
   for (i = 0; i < num_devices; i++) {
      
      if (i < index) {
      
         num_IR_bits_before += JTAG_info[i].IR_length;
      
      } else if (i > index) {
 
         num_IR_bits_after += JTAG_info[i].IR_length;
      }

      // last case -- equal, do nothing
   } // end for
} //end isolate


//------------------------------------------------------------------------------------
// JTAG_IR_Scan
//------------------------------------------------------------------------------------
// This routine loads the supplied <instruction> of <num_bits> length into the JTAG 
// Instruction Register on the isolated device. It shifts the BYPASS opcode (all ones)
// into the Instruction Registers of the other devices in the chain.
//
// NOTE: JTAG_Discover() must be called before this function is called.
//
// NOTE: If more than one device is connected in the chain, JTAG_Isolate() must also 
//       be called prior to calling this function. 
//
// The return value is the n-bit value read from the IR.
// Assumes the JTAG state machine starts in the Run_Test/Idle state.
// Leaves JTAG in the Run_Test/Idle state.
//
unsigned long JTAG_IR_Scan (unsigned long instruction, char num_bits) 
{

    unsigned long retval;                         // JTAG instruction read
    char i;                                       // JTAG IR bit counter
    
    retval = 0x0L;
    
    // navigate the JTAG State Machine in all devices to the Shift_IR state
    TMS = 1;                            
    JTAG_StrobeTCK ();                            // move to SelectDR
    TMS = 1;
    JTAG_StrobeTCK ();                            // move to SelectIR
    TMS = 0;
    JTAG_StrobeTCK ();                            // move to Capture_IR
    TMS = 0;
    JTAG_StrobeTCK ();                            // move to Shift_IR state                     

    TDI=1;
    for (i=0; i < num_IR_bits_before; i++) {
      
        JTAG_StrobeTCK();                         // fill the IR of the devices
                                                  // before the isolated device
    }                                             // with all ones, the BYPASS opcode
    
    
    for (i=0; i < num_bits; i++) {

        TDI = (instruction & 0x01);               // determine output
        instruction = instruction >> 1;
    
        retval = retval >> 1;
        if (TDO) {
            retval |= (0x01 << (num_bits - 1));
        }
        
        if ((i == (num_bits - 1)) && (num_IR_bits_after == 0)) {
            TMS = 1;                              // move to Exit1_IR state
        }        

        JTAG_StrobeTCK();                         // move to Shift_IR state
                                                  // advance
    }

    TDI = 1; 
    for (i=0; i < num_IR_bits_after; i++) {       // now process IR bits after the
                                                  // isolated device
        if (i == (num_IR_bits_after - 1)) {
            TMS = 1;                              // move to Exit1_IR state
        }  
    
    
        JTAG_StrobeTCK();                         // fill the IR of the devices
                                                  // after the isolated device
    }                                             // with all ones, the BYPASS opcode.

    // navigate back to the RTI state
   
    TMS = 1;
    JTAG_StrobeTCK ();                            // move to Update_IR
    TMS = 0;
    JTAG_StrobeTCK ();                            // move to RTI state

    return retval;
}

//------------------------------------------------------------------------------------
// JTAG_DR_Scan
//------------------------------------------------------------------------------------
// This routine shifts <num_bits> of <data> into the Data Register of the isolated 
// device in the chain, and returns up to 32-bits of data read from its Data Register.
// 
// Assumes the JTAG state machine starts in the Run_Test/Idle state.
// Leaves in the Run_Test/Idle state.
//
unsigned long JTAG_DR_Scan (unsigned long dat, char num_bits)
{

    unsigned long retval;                        // JTAG return value
    char i;                                      // JTAG DR bit counter

    retval = 0x0L;

    // navigate the JTAG State Machine in all devices to the Shift_DR state
    TMS = 1;                            
    JTAG_StrobeTCK ();                           // move to SelectDR
    TMS = 0;
    JTAG_StrobeTCK ();                           // move to Capture_DR
    TMS = 0;
    JTAG_StrobeTCK ();                           // move to Shift_DR state 
    
    
    TDI = 0;       
    for (i=0; i < num_devices_before; i++) {
                                                  
       JTAG_StrobeTCK();                         // fill the BYPASS Register 
                                                 // of the devices before the 
    }                                            // isolated device with zeros.
                                                  
                                                   

    for (i=0; i < num_bits; i++) {
   
        TDI = (dat & 0x01);                      // determine the output 
        dat = dat >> 1;
        
        retval = retval >> 1;
        if (TDO) {
            retval |= (0x01L << (num_bits - 1));
        }
        
        if ((i == (num_bits - 1)) && (num_devices_after == 0)) {
            TMS = 1;                             // move to Exit1_IR state
        }        

        JTAG_StrobeTCK();                          
                                                 //output and get input
    }
     
    TDI = 0;
    for (i=0; i < num_devices_after; i++) {
  
        if (i == (num_devices_after - 1)) {
            TMS = 1;                             // move to Exit1_IR state
        }  
         
                                                 // move to Shift_DR state,
       JTAG_StrobeTCK();                         // fill the BYPASS Register 
                                                 // of the devices after the 
    }                                            // isolated device with zeros.
                                                     
    // navigate the JTAG State Machine in all devices to the RTI state
    TMS = 1;
    JTAG_StrobeTCK ();                           // move to Update_DR
    TMS = 0;
    JTAG_StrobeTCK ();                           // move to RTI state

    return retval;                               // retval is MSB aligned
}

//------------------------------------------------------------------------------------
// JTAG_IWrite
//------------------------------------------------------------------------------------
// This routine performs an indirect write to register <ireg>, containing <dat>, of
// <num_bits> in length.  It follows the write operation with a polling operation, and
// returns when the operation is completed.  Note: the polling implemented here refers
// to the JTAG register write operation being completed, NOT the FLASH write operation.
// Polling for the FLASH write operation is handled at a higher level
// Examples of valid indirect registers are:
//  FLASHCON - FLASH Control
//  FLASHSCL - FLASH Scale
//  FLASHADR - FLASH Address
//  FLASHDAT - FLASH Data
// Leaves in the Run_Test/Idle state.
//
void JTAG_IWrite (unsigned int ireg, unsigned long dat, int num_bits)
{
   
   bit done;                              // TRUE = write complete; FALSE otherwise
   
   JTAG_IR_Scan (ireg, INST_LENGTH);      // load IR with <ireg>

   dat |= (0x03L << num_bits);            // append 'WRITE' opcode to data
   
   // load DR with <dat>
   JTAG_DR_Scan (dat, num_bits + 2);      // initiate the JTAG write

   // load DR with '0', and check for BUSY bit to go to '0'.
   do {
      done = !(JTAG_DR_Scan (0x0L, 1));   // poll for JTAG_BUSY bit
   } while (!done);
}

//------------------------------------------------------------------------------------
// JTAG_IRead
//------------------------------------------------------------------------------------
// This routine performs an indirect read of register <ireg>, of <num_bits> in length.
// It follows the read operation with a polling operation, and returns when the 
// operation is completed.  Note: the polling implemented here refers to the JTAG 
// register read operation being completed, NOT the FLASH read operation.
// Polling for the FLASH read operation is handled at a higher level.
// Examples of valid indirect registers are:
//  FLASHCON - FLASH Control
//  FLASHSCL - FLASH Scale
//  FLASHADR - FLASH Address
//  FLASHDAT - FLASH Data
// Leaves JTAG in the Run_Test/Idle state.
//
unsigned long JTAG_IRead (unsigned int ireg, int num_bits) {
   
   unsigned long retval;                  // value returned from READ operation
   bit done;                              // TRUE = write complete; FALSE otherwise
   
   JTAG_IR_Scan (ireg, INST_LENGTH);      // load IR with <ireg>

   // load DR with read opcode (0x02)
   JTAG_DR_Scan (0x02L, 2);               // initiate the JTAG read

   do {
      done = !(JTAG_DR_Scan (0x0L, 1));   // poll for JTAG_BUSY bit
   } while (!done);
   
   retval = JTAG_DR_Scan (0x0L, num_bits + 1);  // allow poll operation to
                                                // read remainder of the bits
   retval = retval >> 1;                  // shift JTAG_BUSY bit off the end

   return retval;
}

//------------------------------------------------------------------------------------
// FLASH_ByteRead
//------------------------------------------------------------------------------------
// This routine reads the byte at <addr> and stores it at the address pointed to by
// <pdat>.
// Returns TRUE if the operation was successful; FALSE otherwise (page 
// read-protected).
//
int FLASH_ByteRead (unsigned int addr, unsigned char *pdat)
{
   unsigned long testval;                       // holds result of FLASHDAT read
   bit done;                                    // TRUE/FALSE flag
   int retval;                                  // TRUE if operation successful

   JTAG_IWrite (FLASHSCL, 0x86L, FLSC_LEN);     // set FLASHSCL based on SYSCLK 
                                                // frequency (2MHz = 0x86)

   // set FLASHADR to address to read from
   JTAG_IWrite (FLASHADR, (unsigned long) addr, FLA_LEN);

   JTAG_IWrite (FLASHCON, 0x02L, FLCN_LEN);     // set FLASHCON for FLASH Read 
                                                // operation (0x02)

   JTAG_IRead (FLASHDAT, FLD_RDLEN);            // initiate the read operation

   JTAG_IWrite (FLASHCON, 0x0L, FLCN_LEN);      // set FLASHCON for 'poll' operation

   do {
      done = !(JTAG_IRead (FLASHDAT, 1));       // poll for FLBUSY to de-assert
   } while (!done);

   testval = JTAG_IRead (FLASHDAT, FLD_RDLEN);  // read the resulting data

   retval = (testval & 0x02) ? FALSE: TRUE;     // FLFail is next to LSB

   testval = testval >> 2;                      // shift data.0 into LSB position

   *pdat = (unsigned char) testval;             // place data in return location

   return retval;                               // return FLASH Pass/Fail
}

//------------------------------------------------------------------------------------
// FLASH_ByteWrite
//------------------------------------------------------------------------------------
// This routine writes the data <dat> to FLASH at the address <addr>.
// Returns TRUE if the operation was successful; FALSE otherwise (page 
// write-protected).
//
int FLASH_ByteWrite (unsigned int addr, unsigned char dat)
{
   unsigned long testval;                       // holds result of FLASHDAT read
   int done;                                    // TRUE/FALSE flag
   int retval;                                  // TRUE if operation successful

   JTAG_IWrite (FLASHSCL, 0x86L, FLSC_LEN);     // set FLASHSCL based on SYSCLK 
                                                // frequency (2MHz = 0x86)

   // set FLASHADR to address to write to
   JTAG_IWrite (FLASHADR, (unsigned long) addr, FLA_LEN);

   JTAG_IWrite (FLASHCON, 0x10L, FLCN_LEN);     // set FLASHCON for FLASH Write 
                                                // operation (0x10)

   // initiate the write operation
   JTAG_IWrite (FLASHDAT, (unsigned long) dat, FLD_WRLEN);

   JTAG_IWrite (FLASHCON, 0x0L, FLCN_LEN);      // set FLASHCON for 'poll' operation

   do {
      done = !(JTAG_IRead (FLASHDAT, 1));       // poll for FLBusy to de-assert
   } while (!done);

   testval = JTAG_IRead (FLASHDAT, 2);          // read FLBusy and FLFail

   retval = (testval & 0x02) ? FALSE: TRUE;     // FLFail is next to LSB

   return retval;                               // return FLASH Pass/Fail
}

//------------------------------------------------------------------------------------
// FLASH_PageErase
//------------------------------------------------------------------------------------
// This routine performs an erase of the page in which <addr> is contained.
// This routine assumes that no FLASH operations are currently in progress.
// This routine exits with no FLASH operations currently in progress.
// Returns TRUE if the operation was successful; FALSE otherwise (page protected).
//
int FLASH_PageErase (unsigned int addr)
{
   unsigned long testval;                       // holds result of FLASHDAT read
   bit done;                                    // TRUE/FALSE flag
   int retval;                                  // TRUE if operation successful

   JTAG_IWrite (FLASHSCL, 0x86L, FLSC_LEN);     // set FLASHSCL based on SYSCLK 
                                                // frequency (2MHz = 0x86)

   // set FLASHADR to address within page to erase
   JTAG_IWrite (FLASHADR, (unsigned long) addr, FLA_LEN);   

   JTAG_IWrite (FLASHCON, 0x20L, FLCN_LEN);     // set FLASHCON for FLASH Erase 
                                                // operation (0x20)

   JTAG_IWrite (FLASHDAT, 0xa5L, FLD_WRLEN);    // set FLASHDAT to 0xa5 to initiate 
                                                // erase procedure

   JTAG_IWrite (FLASHCON, 0x0L, FLCN_LEN);      // set FLASHCON for 'poll' operation

   do {
      done = !(JTAG_IRead (FLASHDAT, 1));       // poll for FLBusy to de-assert
   } while (!done);

   testval = JTAG_IRead (FLASHDAT, 2);          // read FLBusy and FLFail

   retval = (testval & 0x02) ? FALSE: TRUE;     // FLFail is next to LSB

   // set return value based on FLFail bit
   return retval;                               // return FLASH Pass/Fail
}
