/***************************************************************************//**
 * @file mp3config.h
 * @brief Configuration parameters for MP3 Player.
 * @version  1.00

 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#ifndef __MP3CONFIG_H__
#define __MP3CONFIG_H__

// Compile options
#define MP3_MEDIA_SELECT        0       // Storage medium: 0 for internal flash, 1 for uSD card
#define AUDIO_OUTPUT_SELECT     0       // Audio output: 0 for VDAC, 1 for I2S
#define I2S_CODEC_SELECT        0       // I2S codec: 0 for NXP UDA1334ATS, 1 for Tempo TSCS25x            

#include <stdio.h>
#include <string.h>
#include "em_chip.h"
#include "em_cmu.h"
#include "em_common.h"
#include "em_device.h"
#include "em_emu.h"
#include "em_gpio.h"
#include "em_ldma.h"
#include "bspconfig.h"
#include "display.h"
#include "textdisplay.h"
#include "retargettextdisplay.h"
#include "assembly.h"
#include "mp3dec.h"

#if (AUDIO_OUTPUT_SELECT == 1)
#include "em_usart.h"
#include "i2cspm.h"
#include "udelay.h"
#elif (AUDIO_OUTPUT_SELECT == 0)
#include "em_prs.h"
#include "em_timer.h"
#include "em_vdac.h"
#else
#error Invalid value: AUDIO_OUTPUT_SELECT
#endif

#if (MP3_MEDIA_SELECT == 1)
#include "diskio.h"
#include "ff.h"
#include "microsd.h"
#elif (MP3_MEDIA_SELECT == 0)
#include "s_48k.h"              // MP3 data file for internal flash
#else
#error Invalid value: MP3_MEDIA_SELECT
#endif

#if (AUDIO_OUTPUT_SELECT == 1)
// Defines for DPLL
#define FREF                    32768           // Use LFXO as reference
#define TARGET_I2SFREQ_48K      36864000        // 36.864 MHz for 48 KHz
#define DPLL_48K_FACTOR_N       2249            // Recommend to be >300 
#define DPLL_48K_FACTOR_M       (FREF * (DPLL_48K_FACTOR_N + 1))/TARGET_I2SFREQ_48K - 1
#define TARGET_I2SFREQ_44K1     33868800        // 33.8688 MHz for 44.1 KHz 
#define DPLL_44K1_FACTOR_N      2067            // Recommend to be >300 
#define DPLL_44K1_FACTOR_M      (FREF * (DPLL_44K1_FACTOR_N + 1))/TARGET_I2SFREQ_44K1 - 1

// Defines for I2S USART and DMA channel, BSP_STK_BRD2204A is for EFM32GG11 STK
#if defined(BSP_STK_BRD2204A)
#define I2S_USART       USART4
#define I2S_CMUCLOCK    cmuClock_USART4
#define I2S_TXLOC       USART_ROUTELOC0_TXLOC_LOC2
#define I2S_CLKLOC      USART_ROUTELOC0_CLKLOC_LOC0
#define I2S_CSLOC       USART_ROUTELOC0_CSLOC_LOC0
#define I2S_TXPORT      gpioPortI
#define I2S_TXPIN       0
#define I2S_CLKPORT     gpioPortC
#define I2S_CLKPIN      4
#define I2S_CSPORT      gpioPortC
#define I2S_CSPIN       5
#define I2S_RSTPORT     gpioPortE
#define I2S_RSTPIN      8
#define I2S_DMAREQ      0x00100000 + LDMA_CH_REQSEL_SIGSEL_USART3TXBL
#define I2S_DMAREQ_R    0x00100000 + LDMA_CH_REQSEL_SIGSEL_USART3TXBLRIGHT
#else
#define I2S_USART       USART3
#define I2S_CMUCLOCK    cmuClock_USART3
#define I2S_TXLOC       USART_ROUTELOC0_TXLOC_LOC10
#define I2S_CLKLOC      USART_ROUTELOC0_CLKLOC_LOC10
#define I2S_CSLOC       USART_ROUTELOC0_CSLOC_LOC29
#define I2S_TXPORT      gpioPortB
#define I2S_TXPIN       6
#define I2S_CLKPORT     gpioPortB
#define I2S_CLKPIN      8
#define I2S_CSPORT      gpioPortD
#define I2S_CSPIN       8
#define I2S_RSTPORT     gpioPortD
#define I2S_RSTPIN      10
#define I2S_DMAREQ      ldmaPeripheralSignal_USART3_TXBL
#define I2S_DMAREQ_R    ldmaPeripheralSignal_USART3_TXBLRIGHT
#endif

// USART clock divider for different sampling frequencies
#define I2S_CLKDIV_8K           0x00004700
#define I2S_CLKDIV_16K          0x00002300
#define I2S_CLKDIV_32K          0x00001100
#define I2S_CLKDIV_LOW          0x00002f00
#define I2S_CLKDIV_MIDDLE       0x00001700
#define I2S_CLKDIV_HIGH         0x00000b00

// DMA for left channel mono and stereo
#define DMA_CH_I2S              0
#define DMA_CH_I2SMASK          0x01
#define LIST_SIZE               4

// DMA for right channel mono
#define DMA_CH_I2S_RIGHT        1
#define DMA_CH_I2SMASK_RIGHT    0x02
#define LIST_SIZE_RIGHT         2

#if  (I2S_CODEC_SELECT == 1)
#define I2C_CODEC_ADDR          0xE2            // Codec I2C slave address    
#define I2C_NUMBER              I2C0            // Defines for EFM32GG11B I2C
#define I2C_SDALOC              4
#define I2C_SCLLOC              4
#define I2C_SDAPORT             gpioPortC
#define I2C_SDAPIN              0
#define I2C_SCLPORT             gpioPortC
#define I2C_SCLPIN              1

void setupCodec(void);
#endif

void setupI2s(void);
void setupI2sDpll(uint32_t freq);
void setupI2sDma(void);
#elif (AUDIO_OUTPUT_SELECT == 0)
// Defines for VDAC, TIMER, PRS and DMA channel
#define VDAC_HFRCO      cmuHFRCOFreq_38M0Hz
#define VDAC_CLOCK      1000000
#define VDAC_PRS_SEL    vdacPrsSelCh0
#define VDAC_PRS_CH     0
#define VDAC_CH_OFFSET  0x0800
#define VDAC_DMAREQ     ldmaPeripheralSignal_VDAC0_CH0

#if defined(BSP_STK_BRD2204A)
#define VDAC_OPA1_ALT   VDAC_OPA_OUT_ALTOUTEN + VDAC_OPA_OUT_ALTOUTPADEN_OUT0
#else
#define VDAC_OPA1_ALT   VDAC_OPA_OUT_ALTOUTEN + VDAC_OPA_OUT_ALTOUTPADEN_OUT1
#endif

#define TIMER_VDAC      TIMER0
#define TIMER_CLOCK     cmuClock_TIMER0
#define TIMER_PRS       PRS_CH_CTRL_SOURCESEL_TIMER0
#define TIMER_SRC       PRS_CH_CTRL_SIGSEL_TIMER0OF

#define DMA_CH_VDAC     0
#define DMA_CH_VDACMASK 0x01
#define LIST_SIZE       2

void setupVdac(void);
void setupVdacDma(void);
#else
#error Invalid value: AUDIO_OUTPUT_SELECT
#endif

// DMA descriptor initializer for half-word transfers from memory to a peripheral
#define LDMA_DESCRIPTOR_LINKREL_M2P_HALF(src, dest, count, linkjmp)   \
{                                                                     \
  .xfer =                                                             \
  {                                                                   \
    .structType   = ldmaCtrlStructTypeXfer,                           \
    .structReq    = 0,                                                \
    .xferCnt      = (count) - 1,                                      \
    .byteSwap     = 0,                                                \
    .blockSize    = ldmaCtrlBlockSizeUnit1,                           \
    .doneIfs      = 1,                                                \
    .reqMode      = ldmaCtrlReqModeBlock,                             \
    .decLoopCnt   = 0,                                                \
    .ignoreSrec   = 0,                                                \
    .srcInc       = ldmaCtrlSrcIncOne,                                \
    .size         = ldmaCtrlSizeHalf,                                 \
    .dstInc       = ldmaCtrlDstIncNone,                               \
    .srcAddrMode  = ldmaCtrlSrcAddrModeAbs,                           \
    .dstAddrMode  = ldmaCtrlDstAddrModeAbs,                           \
    .srcAddr      = (uint32_t)(src),                                  \
    .dstAddr      = (uint32_t)(dest),                                 \
    .linkMode     = ldmaLinkModeRel,                                  \
    .link         = 1,                                                \
    .linkAddr     = (linkjmp) * 4                                     \
  }                                                                   \
}

// Frame data, SD card read buffer and MP3 decode buffer size
#define FRAME128K_SIZE  430
#define FRAME320K_SIZE  1200
#define READBUF_SIZE    2048
#define DECODEBUF_SIZE  MAX_NCHAN * MAX_NGRAN * MAX_NSAMP

void mp3DecodeFromMedia(void);

#endif /* __MP3CONFIG_H__ */
