/***************************************************************************//**
 * @file mp3i2s.c
 * @brief MP3 Player audio output through I2S
 * @version  1.00

 *******************************************************************************
 * # License
 * <b>Copyright 2017 Silicon Laboratories, Inc. http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "mp3config.h"

#if (AUDIO_OUTPUT_SELECT == 1)
// External variables
extern LDMA_TransferCfg_t descConf;
extern LDMA_Descriptor_t descLink[LIST_SIZE];
extern LDMA_TransferCfg_t descConfRight;
extern LDMA_Descriptor_t descLinkRight[LIST_SIZE_RIGHT];
extern MP3FrameInfo mp3FrameInfo;
extern short outBuf[DECODEBUF_SIZE * 2];

#if (I2S_CODEC_SELECT == 1)
/**************************************************************************//**
 * @brief
 *   Tempo TSCS25x codec initialization.
 * @details
 *   Use PLL1 for 48 KHz and PLL2 for 44.1 KHz.
 *****************************************************************************/
void setupCodec(void)
{
  uint8_t i;
  uint8_t regData;
  I2C_TransferSeq_TypeDef seq;
  I2CSPM_Init_TypeDef i2cInit = I2CSPM_INIT_DEFAULT;
  
  uint8_t codecRegList[] =
  {
    0x13, 0x02,         // 16 bit I2S
    0x18, 0x00,         // Unmute DAC
    0x1b, 0x61,         // Power up DAC O/P buffer and VREF
    0x1c, 0x00,         // Timeout disable

    0x4e, 0x02,         // PLL1 1.536 MHz
    0x4f, 0x03,
    0x50, 0xe0,
    0x51, 0x01,
    0x52, 0x1a,          

    0x53, 0x03,         // PLL2 1.4112 MHz
    0x54, 0x03,
    0x55, 0xd0,
    0x56, 0x02,
    0x57, 0x1b,         // Use default value 0x14 for PLL control register 0x60

    0x77, 0x05,         // TIMEBASE
    0x8f, 0x22,         // PLL1 & PLL2 reference from dac_bclk
    0x61, 0x06,         // Power up PLL1 and PLL2
    
    0xa3, 0x06,         // Hidden register
    0xa4, 0x06,
    0x71, 0x02,
    0x6f, 0x01,
    
    0x21, 0x2c,         // Power down when mute detected
    0x6b, 0x60,         // Hidden register

    0x00, 0x50,         // Left HP volume
    0x01, 0x50,         // Right HP volume
  };

#if defined(BSP_STK_BRD2204A)
  // Overwrite default EFM32GG11B STK I2C setting  
  i2cInit.port = I2C_NUMBER;          
  i2cInit.sclPort = I2C_SCLPORT;
  i2cInit.sclPin = I2C_SCLPIN;
  i2cInit.sdaPort = I2C_SDAPORT;
  i2cInit.sdaPin = I2C_SDAPIN;
  i2cInit.portLocationScl = I2C_SCLLOC;
  i2cInit.portLocationSda = I2C_SDALOC;
#endif
    
  // Reset Codec
  GPIO_PinModeSet(I2S_RSTPORT, I2S_RSTPIN, gpioModePushPull, 1);

  GPIO_PinOutClear(I2S_RSTPORT, I2S_RSTPIN);
  for (i=0; i<10; i++)
  {
    UDELAY_Delay(1000);
  }
  GPIO_PinOutSet(I2S_RSTPORT, I2S_RSTPIN);
  for (i=0; i<10; i++)
  {
    UDELAY_Delay(1000);
  }

  // Initialize I2C interface
  i2cInit.i2cRefFreq = SystemCoreClockGet();
  I2CSPM_Init(&i2cInit);

  // Initialize Tempo TSCS25x codec
  seq.addr = I2C_CODEC_ADDR;
  for (i=0; i<sizeof(codecRegList); i+=2)
  {
    // Write to register
    seq.flags = I2C_FLAG_WRITE;
    seq.buf[0].data = &codecRegList[i];
    seq.buf[0].len  = 2;
    if (I2CSPM_Transfer(i2cInit.port, &seq) != i2cTransferDone)
    {
      while (1)         // I2S codec initialization failed
        ;
    }
    // Read back from register
    seq.flags = I2C_FLAG_WRITE_READ;
    seq.buf[0].data = &codecRegList[i];
    seq.buf[0].len  = 1;
    seq.buf[1].data = &regData;
    seq.buf[1].len  = 1;
    if (I2CSPM_Transfer(i2cInit.port, &seq) != i2cTransferDone)
    {
      while (1)         // I2S codec initialization failed
        ;
    }
    // Compare write and read data
    if (regData != codecRegList[i+1])
    {
      while (1)         // I2S codec initialization failed
        ;
    }
  }   
}  
#endif

/**************************************************************************//**
 * @brief
 *   Setup USART in I2S master mode.
 * @details
 *   USART is initialized in I2S mode to feed the I2S DAC.
 *   Baudrate is set based on information from the MP3 file header later.
 *****************************************************************************/
void setupI2s(void)
{
  USART_InitI2s_TypeDef init = USART_INITI2S_DEFAULT;

  // I2S master TX only
  GPIO_PinModeSet(I2S_TXPORT, I2S_TXPIN, gpioModePushPull, 0);
  GPIO_PinModeSet(I2S_CLKPORT, I2S_CLKPIN, gpioModePushPull, 0);
  GPIO_PinModeSet(I2S_CSPORT, I2S_CSPIN, gpioModePushPull, 0);

  // Configure USART for basic I2S operation, setup baudrate later
  USART_InitI2s(I2S_USART, &init);
    
  // Enable pins at location
  I2S_USART->ROUTEPEN = USART_ROUTEPEN_TXPEN + USART_ROUTEPEN_CLKPEN + USART_ROUTEPEN_CSPEN;
  I2S_USART->ROUTELOC0 = I2S_TXLOC + I2S_CLKLOC + I2S_CSLOC;

  // Use peripheral transfer configuration macro
  descConf = (LDMA_TransferCfg_t)LDMA_TRANSFER_CFG_PERIPHERAL(I2S_DMAREQ);
  descConfRight = (LDMA_TransferCfg_t)LDMA_TRANSFER_CFG_PERIPHERAL(I2S_DMAREQ_R);
}

/**************************************************************************//**
 * @brief
 *   Setup DPLL for I2S
 * @details
 *   Target DPLL HFRCO frequency depends on I2S sampling frequency fs.
 * @param[in] freq
 *   I2S sampling frequency.
 *****************************************************************************/
void setupI2sDpll(uint32_t freq)
{
  CMU_DPLLInit_TypeDef dpllInit = CMU_DPLL_LFXO_TO_40MHZ;

  // Setup and start DPLL if current HFRCO is not for target I2S fs
  switch (freq)
  {
  case 8000:
  case 16000:
  case 32000:
  case 12000:
  case 24000:
  case 48000:
    if (SystemCoreClockGet() != TARGET_I2SFREQ_48K)
    {
      dpllInit.frequency = TARGET_I2SFREQ_48K;
      dpllInit.n = DPLL_48K_FACTOR_N;
      dpllInit.m = DPLL_48K_FACTOR_M;
      if (!CMU_DPLLLock(&dpllInit))
      {
        // DPLL lock failed 
        while (1);
      }
    }
    break;
    
  case 11025:
  case 22050:
  case 44100:
    if (SystemCoreClockGet() != TARGET_I2SFREQ_44K1)
    {
      dpllInit.frequency = TARGET_I2SFREQ_44K1;
      dpllInit.n = DPLL_44K1_FACTOR_N;
      dpllInit.m = DPLL_44K1_FACTOR_M;
      if (!CMU_DPLLLock(&dpllInit))
      {
        // DPLL lock failed 
        while (1);
      }
    }
    break;
    
  default:
    break;
  }
}

/**************************************************************************//**
 * @brief
 *   Setup LDMA in ping pong mode
 * @details
 *   The LDMA is set up to transfer data from memory to the I2S.
 *   Transfer size is mono/stereo and sampling frequency dependent.
 *****************************************************************************/
void setupI2sDma(void)
{
  // Setup I2S baud rate for mono and stereo
  switch (mp3FrameInfo.samprate)
  {
  case 8000:
    I2S_USART->CLKDIV = I2S_CLKDIV_8K;
    break;
    
  case 11025:
  case 12000:
    I2S_USART->CLKDIV = I2S_CLKDIV_LOW;
    break;
    
  case 16000:
    I2S_USART->CLKDIV = I2S_CLKDIV_16K;
    break;
    
  case 22050:
  case 24000:
    I2S_USART->CLKDIV = I2S_CLKDIV_MIDDLE;
    break;
    
  case 32000:
    I2S_USART->CLKDIV = I2S_CLKDIV_32K;
    break;
    
  case 44100:
  case 48000:
    I2S_USART->CLKDIV = I2S_CLKDIV_HIGH;
    break;
    
  default:
    break;
  }

  if (mp3FrameInfo.nChans == 1)
  {
    // Setup LDMA for mono, enable DMASPLIT and use left channel data for right channel
    I2S_USART->I2SCTRL |= USART_I2SCTRL_DMASPLIT;
    switch (mp3FrameInfo.samprate)
    {
    case 8000:
    case 16000:
    case 11025:
    case 22050:
    case 12000:
    case 24000:
      descLink[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &I2S_USART->TXDOUBLE, MAX_NSAMP, 1);
      descLink[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &I2S_USART->TXDOUBLE, MAX_NSAMP, -1);
      descLinkRight[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &I2S_USART->TXDOUBLE, MAX_NSAMP, 1);
      descLinkRight[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &I2S_USART->TXDOUBLE, MAX_NSAMP, -1);
      break;
      
    case 32000:
    case 44100:
    case 48000:
      descLink[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, 1);
      descLink[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, -1);
      descLinkRight[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, 1);
      descLinkRight[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, -1);
      break;
      
    default:
      break;
    }
    // DMA interrupt only on right channel 
    descLink[0].xfer.doneIfs = 0;
    descLink[1].xfer.doneIfs = 0;
    descLink[0].xfer.link = 1;
    descLink[1].xfer.link = 1;
    descLinkRight[0].xfer.link = 1;
    descLinkRight[1].xfer.link = 1;
  }
  else
  {
    // Setup LDMA for stereo, disable DMASPLIT and data format is LRLRLR...
    I2S_USART->I2SCTRL &= ~USART_I2SCTRL_DMASPLIT;
    switch (mp3FrameInfo.samprate)
    {
    case 8000:
    case 16000:
    case 11025:
    case 22050:
    case 12000:
    case 24000:
      descLink[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, 1);
      descLink[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, -1);
      break;
      
    case 32000:
    case 44100:
    case 48000:
      descLink[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, 1);
      descLink[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 2), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, 1);
      descLink[2] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, 1);
      descLink[3] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 6), &I2S_USART->TXDOUBLE, MAX_NSAMP * 2, -3);
      descLink[0].xfer.doneIfs = 0;
      descLink[2].xfer.doneIfs = 0;
      break;
      
    default:
      break;
    }
    descLink[0].xfer.link = 1;
    descLink[1].xfer.link = 1;
    descLink[2].xfer.link = 1;
    descLink[3].xfer.link = 1;
  }
}
#endif
