/***************************************************************************//**
 * @file mp3media.c
 * @brief MP3 Player reads data from uSD card or internal flash
 * @version  1.00

 *******************************************************************************
 * # License
 * <b>Copyright 2017 Silicon Laboratories, Inc. http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "mp3config.h"

#if (MP3_MEDIA_SELECT == 1)
// uSD card and file system specific
volatile bool eofReached;
uint8_t readBuf[READBUF_SIZE];
FIL mp3File;                    
#endif

// Descriptor linked list for LDMA transfer
LDMA_TransferCfg_t descConf;
LDMA_Descriptor_t descLink[LIST_SIZE];
#if (AUDIO_OUTPUT_SELECT == 1)
LDMA_TransferCfg_t descConfRight;
LDMA_Descriptor_t descLinkRight[LIST_SIZE_RIGHT];
#endif

// Ping-Pong DMA specific
volatile uint32_t nFrames;
volatile bool getNextFrame;

// MP3 deocde specific
HMP3Decoder mp3Instance;
MP3FrameInfo mp3FrameInfo;

// MP3 decode output buffer
short outBuf[DECODEBUF_SIZE * 2];

// Global variables
int bytesLeft;
uint8_t *readPtr;
volatile int16_t outOfData;

/***************************************************************************//**
 * @brief
 *   Interrupt handler for LDMA cycle completion handling.
******************************************************************************/
void LDMA_IRQHandler(void)
{
  // Read and clear LDMA interrupt flag
  uint32_t ldmaIntFlag = LDMA->IFC;

#if (AUDIO_OUTPUT_SELECT == 1)
  if (ldmaIntFlag & (DMA_CH_I2SMASK + DMA_CH_I2SMASK_RIGHT))
#else
  if (ldmaIntFlag & DMA_CH_VDACMASK)
#endif
  {
    // Ready to decode next MP3 frame
    nFrames++;
    getNextFrame = true;
  }
  else
  { 
    // Bus error, loop here to enable the debugger to see what has happened
    while(1);
  }  
}

#if (MP3_MEDIA_SELECT == 1)
/***************************************************************************//**
 * @brief
 *   Store MP3 data from SD card to buffer for decoding.
 * @param[in] *readBuf
 *   Pointer to the buffer to store MP3 data.
 * @param[in] *readPtr
 *   Pointer to last read location.
 * @param[in] bufSize
 *   Size of buffer to store MP3 data.
 * @param[in] byteLeft
 *   Bytes left in the buffer not yet processed.
 * @param[in] *infile
 *   Pointer to the open MP3 file object.
 * @return
 *   Number of bytes read.
******************************************************************************/
static int fillReadBuffer(unsigned char *readBuf, unsigned char *readPtr, int bufSize, int bytesLeft, FIL *infile)
{
  uint16_t nRead;

  // Move last, small chunk from end of buffer to start, then fill with new data
  memmove(readBuf, readPtr, bytesLeft);				
  f_read(infile, readBuf + bytesLeft, bufSize - bytesLeft, &nRead);

  // Zero-pad to avoid finding false sync word after last frame (from old data in readBuf)
  if (nRead < bufSize - bytesLeft)
  {
    memset(readBuf + bytesLeft + nRead, 0, bufSize - bytesLeft - nRead);	
  }
  return (int) nRead;
}
#endif

/***************************************************************************//**
 * @brief
 *   Store decoded PCM data to output buffer.
 * @param[in] *readFile (for MP3 file in uSD card)
 *   Pointer to the open MP3 file object.
 * @param[in] *outBuf
 *   Pointer to the buffer to store decoded PCM data.
 * @return
 *   1:Error or EOF, 0:OK, -1:Retry.
******************************************************************************/
#if (MP3_MEDIA_SELECT == 1)
static int fillMP3DecodeBuffer(FIL *readFile, short *outbuf)
{
  int err, offset;
  int nRead;

  // Somewhat arbitrary trigger to refill buffer - should always be enough for a full frame
  if (bytesLeft < 2 * MAINBUF_SIZE && !eofReached) 
  {
    nRead = fillReadBuffer(readBuf, readPtr, READBUF_SIZE, bytesLeft, readFile);
    bytesLeft += nRead;
    readPtr = readBuf;
    if (nRead == 0)
    {
      eofReached = true;
    }
  }
  
  // Find start of next MP3 frame, retry or EOF if no sync found
  offset = MP3FindSyncWord(readPtr, bytesLeft);
  if (offset < 0)
  {
    if (eofReached)
    {
      outOfData = 1;
    }
    else
    {
      bytesLeft = 0;
      outOfData = -1;
    }
    return outOfData;
  }

  readPtr += offset;
  bytesLeft -= offset;

  // Retry if not a valid MP3 frame header
  err = MP3GetNextFrameInfo(mp3Instance, &mp3FrameInfo, readPtr);
  if (err < 0)
  {
    readPtr++;
    bytesLeft--;
    outOfData = -1;
    return outOfData;
  }
  
  /* Retry if not enough data for decode */
  if (bytesLeft < FRAME320K_SIZE)
  {
    outOfData = -1;
    return outOfData;
  }
#else
static int fillMP3DecodeBuffer(short *outbuf)
{
  int err, offset;

  // Find sync word if not end of MP3 data
  offset = MP3FindSyncWord(readPtr, bytesLeft);
  if (offset < 0)
  {
    outOfData = 1;
    return outOfData;
  }
  
  readPtr += offset;
  bytesLeft -= offset;

  // Retry if not a valid MP3 frame header
  err = MP3GetNextFrameInfo(mp3Instance, &mp3FrameInfo, readPtr);
  if (err < 0)
  {
    readPtr++;
    bytesLeft--;
    outOfData = -1;
    return outOfData;
  }

  // End if not enough data for decode
  if (bytesLeft < FRAME128K_SIZE)
  {
    outOfData = 1;
    return outOfData;
  }
#endif  

  // Decode one MP3 frame
  err = MP3Decode(mp3Instance, &readPtr, &bytesLeft, outbuf, 0);
  if (err < 0)
  {
    /* Error occurred */
    switch (err)
    {
    case ERR_MP3_INDATA_UNDERFLOW:
      outOfData = 1;
      printf("ERR_MP3_INDATA_UNDERFLOW\n");
      break;

    case ERR_MP3_MAINDATA_UNDERFLOW:
    case ERR_MP3_INVALID_HUFFCODES:
      // Do nothing - next call to decode will provide more mainData
      outOfData = 0;
      break;

    case ERR_MP3_FREE_BITRATE_SYNC:
      outOfData = 1;
      printf("ERR_MP3_BITRATE_SYNC\n");
      break;

    default:
      outOfData = 1;
      printf("MP3 DECODE ERROR\n");
      break;
    }
  }
  else
  {
    outOfData = 0;
#if (AUDIO_OUTPUT_SELECT == 0)
    // Convert PCM data to 12 bit for VDAC
    for(err = 0; err < DECODEBUF_SIZE; err++)
    {
      outbuf[err] = (outbuf[err] >> 4) + VDAC_CH_OFFSET;
    }
#endif
  }
  return outOfData;
}

/***************************************************************************//**
 * @brief
 *   Decode MP3 data and output to I2S or VDAC.
******************************************************************************/
static void MP3Play(void)
{
  // Reset parameters
#if (MP3_MEDIA_SELECT == 1)  
  bytesLeft = 0;
  eofReached = false;
#else
  bytesLeft = MP3DATA_SIZE;
  readPtr = (uint8_t *)MP3DATA;
#endif
  getNextFrame = false;
  
  // Fill 2 buffers in advance
  do
  {
#if (MP3_MEDIA_SELECT == 1)  
    fillMP3DecodeBuffer(&mp3File, outBuf);
#else
    fillMP3DecodeBuffer(outBuf);
#endif
  } while (outOfData != 0);

  printf("\nmp3FrameInfo\n");
  printf("outputSamps: %u\n", mp3FrameInfo.outputSamps);
  printf("bitrate:     %u\n", mp3FrameInfo.bitrate);
  printf("bits p.s:    %u\n", mp3FrameInfo.bitsPerSample);
  printf("nChans:      %u\n", mp3FrameInfo.nChans);
  printf("sampRate:    %u\n", mp3FrameInfo.samprate);
  printf("layer:       %u\n", mp3FrameInfo.layer);
  printf("version:     %u\n", mp3FrameInfo.version);
    
  // Use first valid frame header to setup I2S or VDAC 
#if (AUDIO_OUTPUT_SELECT == 1)
  setupI2sDpll(mp3FrameInfo.samprate);
  setupI2sDma();
#else
  setupVdacDma();
#endif
  
  do
  {
#if (MP3_MEDIA_SELECT == 1)  
    fillMP3DecodeBuffer(&mp3File, (outBuf + DECODEBUF_SIZE));
#else
    fillMP3DecodeBuffer((outBuf + DECODEBUF_SIZE));
#endif
  } while (outOfData != 0);
  
  // Start Ping-Pong transfer
  nFrames = 2;
#if (AUDIO_OUTPUT_SELECT == 1)
  LDMA_StartTransfer(DMA_CH_I2S, (void*)&descConf, (void*)&descLink);
  if (mp3FrameInfo.nChans == 1)
  {
    LDMA_StartTransfer(DMA_CH_I2S_RIGHT, (void*)&descConfRight, (void*)&descLinkRight);
  }
#else
  LDMA_StartTransfer(DMA_CH_VDAC, (void*)&descConf, (void*)&descLink);
  TIMER_VDAC->CMD = TIMER_CMD_START;      // Start TIMER to trigger VDAC
#endif
  
  // Play until end of file
  do
  {
    // Wait buffer ready to refill
    while (!getNextFrame)
    {
      EMU_EnterEM1();
    }
    getNextFrame = false;

    if (!(nFrames & 0x01))
    {
#if (MP3_MEDIA_SELECT == 1)  
      fillMP3DecodeBuffer(&mp3File, (outBuf + DECODEBUF_SIZE));
#else
      fillMP3DecodeBuffer((outBuf + DECODEBUF_SIZE));
#endif
    }
    else
    {
#if (MP3_MEDIA_SELECT == 1)  
      fillMP3DecodeBuffer(&mp3File, outBuf);
#else
      fillMP3DecodeBuffer(outBuf);
#endif
    }
  } while (outOfData == 0);
  
  // End of play
#if (AUDIO_OUTPUT_SELECT == 1)
  descLink[0].xfer.link = 0;
  descLink[1].xfer.link = 0;
  descLink[2].xfer.link = 0;
  descLink[3].xfer.link = 0;
  descLinkRight[0].xfer.link = 0;
  descLinkRight[1].xfer.link = 0;
#else
  TIMER_VDAC->CMD = TIMER_CMD_STOP;
  if (mp3FrameInfo.nChans == 1)
  {
    VDAC0->CMD = VDAC_CMD_CH0DIS;
  }
  else
  {
    VDAC0->CMD = VDAC_CMD_CH0DIS + VDAC_CMD_CH1DIS;
  }    
  descLink[0].xfer.link = 0;
  descLink[1].xfer.link = 0;
#endif  
}

/***************************************************************************//**
 * @brief
 *   Decode MP3 data from storage media.
******************************************************************************/
void mp3DecodeFromMedia(void)
{
#if (MP3_MEDIA_SELECT == 1)  
  // Return value, directory search object and file information
  DIR dj;
  FATFS Fatfs;
  FILINFO fno;
  FRESULT res;

  // Initialize filesystem for SD card 
  MICROSD_Init();
  res = f_mount(&Fatfs, "", 0);
  if (res != FR_OK)
  {
    printf("No FAT file system in uSD card\n");
    EMU_EnterEM3(false);
    while(1);
  }
#endif
  
  // Initialize decoder
  mp3Instance = MP3InitDecoder();
  if(mp3Instance == 0)
  {
    printf("Could not init MP3 decoder\n");
    EMU_EnterEM3(false);
    while(1);
  }

  // Start playing
#if (AUDIO_OUTPUT_SELECT == 1)
  printf("MP3 Player I2S demo\n");
#else
  printf("MP3 Player VDAC demo\n");
#endif

#if (MP3_MEDIA_SELECT == 1)  
  while(1)
  {
    // Start to search for MP3 files
    res = f_findfirst(&dj, &fno, "", "*.mp3");  
    if (!fno.fname[0])
    {
      printf("No MP3 file in uSD card\n");
      MP3FreeDecoder(mp3Instance);
      EMU_EnterEM3(false);
      while(1);
    }

    while (res == FR_OK && fno.fname[0])
    {
      // Open and play MP3 file
      res = f_open(&mp3File, fno.fname, FA_READ);
      if (res != FR_OK)
      {
        printf("MP3 file open error\n");
      }
      else
      {
        printf("\nPlaying %s\n", fno.fname);
        MP3Play();
      }
      // Close current file and search for next item
      f_close(&mp3File);
      res = f_findnext(&dj, &fno);               
    }
  }
#else
  while(1)
  {
    printf("\nPlaying MP3 from internal Flash\n");
    MP3Play();
  }
#endif
}
